﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../common.h"

namespace {

const size_t  buffSize = 100 * 1024; // 100KB
const size_t  txDataSize = 1400; // 1400Bytes
NN_ALIGNAS(4096) char  g_scanBuffer[ buffSize ];
uint8_t  g_txBuffer[ txDataSize ];
uint8_t  g_rxBuffer[2048]; // 受信用バッファ
uint8_t  g_rxBufferAf[1024]; // ActionFrame受信用バッファ
nn::wlan::WlanState g_wlanState;

const size_t ThreadStackSize = 4096;              // スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackRx[ ThreadStackSize ];   // 受信用スレッドスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackRxAf[ ThreadStackSize ];   // ActionFrame受信用スレッドスタック

nn::os::ThreadType  g_ThreadRx;
nn::os::ThreadType  g_ThreadRxAf;
static bool g_ExitFlagRx = false;
static bool g_ExitFlagRxAf = false;

// データパス用
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackRxSoc[ ThreadStackSize ];   // 受信用スレッドスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackTxSoc[ ThreadStackSize ];   // 送信用スレッドスタック
nn::os::ThreadType  g_ThreadRxSoc;
nn::os::ThreadType  g_ThreadTxSoc;

nn::wlan::ScanParameters g_scanParam = {
        nn::wlan::ScanType_Passive,
        {1, 6, 11, 36, 40, 44, 48},
        7,  // チャンネルリストに記載したうちの何チャンネル分をスキャンするか
        120,
        0,
        NULL,
        0,
        nn::wlan::MacAddress::CreateBroadcastMacAddress()
};

}


void RxThreadFunc(void* arg)
{
    nn::Result result;
    g_ExitFlagRx = true;

    uint32_t* rxId = reinterpret_cast<uint32_t*>(arg);

    while( g_ExitFlagRx )
    {
        size_t rxSize = 0;
        result = nn::wlan::Local::GetFrameRaw(g_rxBuffer, sizeof(g_rxBuffer), &rxSize, *rxId); // ブロックされる
        if( result.IsSuccess() )
        {
            NN_LOG("Received DataFrame\n");
            //WlanTest::DumpBuffer(g_rxBuffer, rxSize);
        }
        else if( result.GetDescription() == nn::wlan::ResultGetFrameCancelled().GetDescription() )
        {
            NN_LOG("GetFrame Cancelled.\n");
        }
    }
}

void RxThreadFuncAf(void* arg)
{
    nn::Result result;
    g_ExitFlagRxAf = true;

    uint32_t* rxId = reinterpret_cast<uint32_t*>(arg);
    nn::wlan::MacAddress mac;
    //char macStr[nn::wlan::MacAddress::MacStringSize];

    while( g_ExitFlagRxAf )
    {
        size_t rxSize = 0;
        result = nn::wlan::Local::GetActionFrame(&mac, g_rxBufferAf, sizeof(g_rxBufferAf), &rxSize, *rxId); // ブロックされる
        if( result.IsSuccess() )
        {
            //NN_LOG("Received ActionFrame from : %s\n", mac.GetString(macStr));
            //WlanTest::DumpBuffer(g_rxBufferAf, rxSize);
        }
        else if( result.GetDescription() == nn::wlan::ResultGetFrameCancelled().GetDescription() )
        {
            NN_LOG("GetFrame Cancelled.\n");
        }
    }
}


void TestCaseLocalSta()
{
    // 接続に変化があった際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType connectionEvent;

    // 接続状態を格納するためのオブジェクト
    nn::wlan::ConnectionStatus connectionStatus;

    nn::wlan::InitializeLocalManager();

    nn::wlan::Local::GetState(&g_wlanState);
    NN_LOG("WLAN STATE : %s\n", WlanTest::wlanStateStr[g_wlanState]);

    nn::wlan::Local::OpenClientMode();
    NN_LOG("WlanTest: OpenClientMode done\n");

    nn::wlan::Local::GetState(&g_wlanState);
    NN_LOG("WLAN STATE : %s\n", WlanTest::wlanStateStr[g_wlanState]);

    nn::wlan::Local::GetConnectionEvent(&connectionEvent);
    NN_LOG("WLANClient: GetConnectionEvent\n");

    nn::wlan::MacAddress macAddr;
    nn::wlan::Local::GetMacAddress(&macAddr);
    char macStr[nn::wlan::MacAddress::MacStringSize];
    NN_LOG("mac addr : %s\n", macAddr.GetString(macStr));

    nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Passive,
            {1, 6, 11, 36, 40, 44, 48},
            7,  // チャンネルリストに記載したうちの何チャンネル分をスキャンするか
            120,
            0,
            NULL,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };
    nn::wlan::ScanIeMatchInfo info = {
            {0},
            0
    };
    info.matchLength = 5;
    info.matchData[0] = 0x00;
    info.matchData[1] = 0x22;
    info.matchData[2] = 0xAA;
    info.matchData[3] = 0xbb;
    info.matchData[4] = 0xcc;

    nn::wlan::Local::StartScan(g_scanBuffer, buffSize, scanParam, &info);
    NN_LOG("WlanTest: Scan finished.\n");

    // Dump scan buffer
    WlanTest::DumpScanResultWithReader(g_scanBuffer);

    nn::wlan::Local::GetState(&g_wlanState);
    NN_LOG("WLAN STATE : %s\n", WlanTest::wlanStateStr[g_wlanState]);

    // 接続試行
    // MASTERの情報を抜き出す
    nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);
    uint32_t bssCount = resultReader.GetCount();
    nn::Bit8 matchingOui[3] = {
            WlanTest::localVendorIe[0],
            WlanTest::localVendorIe[1],
            WlanTest::localVendorIe[2],
    };  // MASTERのVIEのOUI
    nn::Bit8 matchingPId[2] = {
            WlanTest::localVendorIe[3],
            WlanTest::localVendorIe[4],
    };  // MASTERのVIEのPID
    const nn::wlan::VendorInfoElementReader* vie;

    nn::wlan::Ssid masterSsid;           // MASTERのSSID格納先
    nn::wlan::MacAddress masterBssid;  // MASTERのBSSID格納先
    int16_t masterChannel;               // MASTERのChannel格納先
    bool Exist = false;                  // MASTERが見つかっているかどうかのフラグ
    for(uint32_t i = 0; i < bssCount; i++)
    {
        nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
        vie = beacon.GetVendorIeWithOuiAndProtocolId(matchingOui, matchingPId);
        if( vie != NULL )
        {
            NN_LOG("WlanClien: Found Local Master! ch:%d\n", beacon.GetChannel());
            // SSID, BSSID, Channelを取得
            masterSsid.Set(WlanTest::localMasterSsid); // MASTERのSSIDは自明なものとする
            masterBssid = beacon.GetBssid();
            masterChannel = beacon.GetChannel();
            Exist = true;
        }
    }

    // MASTERが見つかっていなかったら死ぬ
    NN_ASSERT(Exist, "Failed to find Local MASTER.\n");

    // 相手先のセキュリティ情報入力
    nn::wlan::Security security = {
            nn::wlan::SecurityMode_StaticAes,
            nn::wlan::SecurityMode_StaticAes,
            0,
            "",
    };
    std::memcpy(&security.key[0], &WlanTest::staticAesKey[0], 16);

    // 接続出来るまで試行し続ける
    while( 1 )
    {
#if 0 // SSIDで相手先に接続する場合
        nn::wlan::Ssid peerSsid(WlanTest::localMasterSsid);
        nn::wlan::Local::Connect(peerSsid,
                nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                static_cast<uint16_t>(nn::wlan::WirelessChannel_1ch),
                security,
                false,
                nn::wlan::BeaconIndication_Disable);
#else
        nn::wlan::Local::Connect(masterSsid,
                masterBssid,
                masterChannel,
                security,
                false,
                nn::wlan::BeaconIndication_Disable);
#endif
        // Connect関数は同期型なので、この時点で既にイベントはシグナルされているはず
        nn::os::WaitSystemEvent(&connectionEvent);
        NN_LOG("WlanTest: Connection event signaled\n");

        nn::wlan::Local::GetConnectionStatus(&connectionStatus);
        WlanTest::PrintConnectionStatus(&connectionStatus);

        if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("WlanTest: CONNECTED!!!\n");
            break;
        }

        // 3秒待って再試行
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
    }

    NN_LOG("WlanTest: Waiting for connection status change\n");

    // 送信前にマッチング用データを登録しておく。そうしないと送信許可されない。
    nn::Result      result;

    /*
     * マッチング用データ登録作業 ここから--------------------------------------------------------------------------------------
     */
    // マッチング用データ（適当）
    nn::wlan::ReceivedDataMatchInfo matchInfo1 = {
            { 0x00, 0x22, 0xAA, 0x00, 0x01 },  // OUI + PI
            5
    };
    nn::wlan::ReceivedDataMatchInfo matchInfo2 = {
            { 0x00, 0x22, 0xAA, 0xAB, 0xCD },  // OUI + PI
            5
    };
    uint32_t rxId;  // 受信エントリ番号受取り用変数
    uint16_t ethertypes[] = { 0x88b7 };  // 受信したいプロトコルID（EtherType）

    // 受信エントリ作成
    result = nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30); // capacityは適当。とりあえず30個は保持できるようにしておく。
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("WlanTest: Created Rx Entry [%d]\n", rxId);

    // 作成した受信エントリにマッチング用データを登録
    result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo1);
    NN_ASSERT( result.IsSuccess() );
    result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo2);
    NN_ASSERT( result.IsSuccess() );
    /*
     * マッチング用データ登録作業 ここまで--------------------------------------------------------------------------------------
     */

    // 受信スレッド作成
    result = nn::os::CreateThread( &g_ThreadRx, RxThreadFunc, &rxId, g_ThreadStackRx, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadRx." );
    nn::os::StartThread( &g_ThreadRx );


    /*
     * ActionFrame受信準備
     */
    uint32_t rxIdAf;         // 受信エントリ番号受取り用変数
    uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Beacon),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
    };
    // ActionFrame用受信エントリ作成
    result = nn::wlan::Local::CreateRxEntryForActionFrame(&rxIdAf, afTypes, sizeof(afTypes) / sizeof(uint16_t), 10); // ActionFrameはデータパスと異なり、溜めておける数に限りがあるので少な目
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("WlanTest: Created Rx Entry for action frame[%d]\n", rxIdAf);

    // ActionFrame受信スレッド作成
    result = nn::os::CreateThread( &g_ThreadRxAf, RxThreadFuncAf, &rxIdAf, g_ThreadStackRxAf, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadRxAf." );
    nn::os::StartThread( &g_ThreadRxAf );


    // 10発データを送信して、切断する。その間に切断が起きたら中断する。
    bool isDisconnect = false;
    for( uint8_t i = 0; i < 10; i++ )
    {
        if( nn::os::TryWaitSystemEvent(&connectionEvent) == true )
        {
            isDisconnect = true;
            break;
        }

        // 送信データ生成
        // 宛先 : Broadcastアドレス
        // 送信元 : 自局MACアドレス
        WlanTest::GenerateTxDataForLocal(g_txBuffer, nn::wlan::MacAddress::CreateBroadcastMacAddress(), macAddr, 64);
        nn::wlan::Local::PutFrameRaw(reinterpret_cast<uint8_t*>(g_txBuffer), 64);

        // ActionFrameの送信
        nn::Bit8 oui[3] = { 0x00, 0x22, 0xAA };
        WlanTest::GenerateTxDataForLocal(g_txBuffer, nn::wlan::MacAddress::CreateBroadcastMacAddress(), macAddr, 128);
        g_txBuffer[0] = 0x7F;  // Category
        std::memcpy(&g_txBuffer[1], &oui[0], 3);  // OUI
        g_txBuffer[4] = nn::wlan::ActionFrameType_Local;  // Subtype
        nn::wlan::Local::PutActionFrameOneShot(nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                reinterpret_cast<uint8_t*>(g_txBuffer), 128, static_cast<nn::wlan::WirelessChannel>(connectionStatus.channel), 3);

        if( i % 3 == 0 )
            nn::wlan::Local::CancelGetFrame(rxId);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));
    }

    if( isDisconnect == false )
    {
        // 切断する
        // 切断前に一発送信しておく
        WlanTest::GenerateTxDataForLocal(g_txBuffer, nn::wlan::MacAddress::CreateBroadcastMacAddress(), macAddr, 512);
        // Protocol Identifier
        // 切断を意味するフラグを立てておく
        g_txBuffer[17] = 0xAB;
        g_txBuffer[18] = 0xCD;
        nn::wlan::Local::PutFrameRaw(reinterpret_cast<uint8_t*>(g_txBuffer), 512);

        nn::wlan::Local::Disconnect(
                nn::wlan::LocalCommunicationMode_ClientSpectator,
                NULL);
    }

    // 受信のキャンセルと受信スレッドの破棄
    g_ExitFlagRx = false;
    nn::wlan::Local::CancelGetFrame(rxId);
    nn::os::WaitThread( &g_ThreadRx );
    nn::os::DestroyThread( &g_ThreadRx );

    g_ExitFlagRxAf = false;
    nn::wlan::Local::CancelGetActionFrame(rxIdAf);
    nn::os::WaitThread( &g_ThreadRxAf );
    nn::os::DestroyThread( &g_ThreadRxAf );

    nn::wlan::Local::GetConnectionStatus(&connectionStatus);
    WlanTest::PrintConnectionStatus(&connectionStatus);

    // 受信エントリ削除
    result = nn::wlan::Local::DeleteRxEntry(rxId);
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("WlanTest: Delete Rx Entry [%d]\n", rxId);
    result = nn::wlan::Local::DeleteRxEntryForActionFrame(rxIdAf);
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("WlanTest: Delete Rx Entry [%d]\n", rxIdAf);

    nn::wlan::Local::CloseClientMode();
    NN_LOG("WlanTest: CloseClientMode done\n");

    nn::wlan::Local::GetState(&g_wlanState);
    NN_LOG("WLAN STATE : %s\n", WlanTest::wlanStateStr[g_wlanState]);

    NN_LOG("\n\n End WlanTest \n\n");
    nn::wlan::FinalizeLocalManager();
} //NOLINT(impl/function_size)

void TestCaseLocalStaWithDataPath()
{
    nn::Result result;
    nn::os::SystemEventType connectionEvent;
    nn::wlan::ConnectionStatus connectionStatus;

    nn::wlan::InitializeLocalManager();
    nn::wlan::Local::OpenClientMode();

    nn::wlan::Local::GetConnectionEvent(&connectionEvent);

    // MASTERの接続情報を取得する
    nn::wlan::Ssid masterSsid;           // MASTERのSSID格納先
    nn::wlan::MacAddress masterBssid;    // MASTERのBSSID格納先
    int16_t masterChannel;               // MASTERのChannel格納先
    nn::Bit8 matchingOui[3] = {
            WlanTest::localVendorIe[0],
            WlanTest::localVendorIe[1],
            WlanTest::localVendorIe[2],
    };  // MASTERのVIEのOUI

    // MASTERが見つかるまでスキャン
    bool loop = true;
    while( loop )
    {
        nn::wlan::Local::StartScan(g_scanBuffer, buffSize, g_scanParam);
        NN_LOG("WlanTest: Scan finished.\n");

        // MASTERの情報を抜き出す
        nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);
        uint32_t bssCount = resultReader.GetCount();
        const nn::wlan::VendorInfoElementReader* vie;

        for(uint32_t i = 0; i < bssCount; i++)
        {
            nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
            vie = beacon.GetVendorIeWithOui(matchingOui);
            if( vie != NULL )
            {
                NN_LOG("WlanClien: Found Local Master! ch:%d\n", beacon.GetChannel());
                // SSID, BSSID, Channelを取得
                masterSsid.Set(WlanTest::localMasterSsid); // MASTERのSSIDは自明なものとする
                masterBssid = beacon.GetBssid();
                masterChannel = beacon.GetChannel();
                loop = false;
            }
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

    // 相手先のセキュリティ情報入力
    nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,//StaticAes,
            nn::wlan::SecurityMode_Open,//StaticAes,
            0,
            "",
    };
    std::memcpy(&security.key[0], &WlanTest::staticAesKey[0], 16);

    // 接続出来るまで試行し続ける
    while( 1 )
    {
        nn::wlan::Local::Connect(masterSsid,
                masterBssid,
                -1, //masterChannel,
                security,
                false,
                nn::wlan::BeaconIndication_Disable);

        nn::os::WaitSystemEvent(&connectionEvent);

        nn::wlan::Local::GetConnectionStatus(&connectionStatus);
        WlanTest::PrintConnectionStatus(&connectionStatus);

        if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("WlanTest: CONNECTED!!!\n");
            break;
        }

        // 3秒待って再試行
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
    }

    // wlanプロセスにデータ送受信可能フラグを立たせるためにマッチング用データを登録しておく
    // マッチング用データ（適当）
    nn::wlan::ReceivedDataMatchInfo matchInfo1 = {
            { 0x00, 0x22, 0xAA, 0x00, 0x01 },  // OUI + PI
            5
    };
    uint32_t rxId;  // 受信エントリ番号受取り用変数
    uint16_t ethertypes[] = { 0x88b7 };  // 受信したいプロトコルID（EtherType）
    // 受信エントリ作成
    result = nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30); // capacityは適当。とりあえず30個は保持できるようにしておく。
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("WlanTest: Created Rx Entry [%d]\n", rxId);
    // 作成した受信エントリにマッチング用データを登録
    result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo1);
    NN_ASSERT( result.IsSuccess() );

    // データ通信用スレッドの開始および実行
    {
        nn::wlan::InitializeSocketManager();
        result = nn::os::CreateThread( &g_ThreadRxSoc, WlanTest::DataPathRxThreadFunc, NULL, g_ThreadStackRxSoc, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadRxSoc." );
        nn::os::StartThread( &g_ThreadRxSoc );
        result = nn::os::CreateThread( &g_ThreadTxSoc, WlanTest::DataPathTxThreadFunc, NULL, g_ThreadStackTxSoc, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadTxSoc." );
        nn::os::StartThread( &g_ThreadTxSoc );
    }

    while( 1 )
    {
        nn::os::WaitSystemEvent(&connectionEvent);
        nn::wlan::Local::GetConnectionStatus(&connectionStatus);
        if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("Disconnected!\n");
            WlanTest::PrintConnectionStatus(&connectionStatus);
            break;
        }
    }

    // 一応切断関数を呼んでおく
    nn::wlan::Local::Disconnect(
            nn::wlan::LocalCommunicationMode_ClientSpectator,
            NULL);
    // 受信エントリ削除
    result = nn::wlan::Local::DeleteRxEntry(rxId);
    NN_ASSERT( result.IsSuccess() );

    nn::wlan::Local::CloseClientMode();

    // データパスを閉じる
    {
        nn::os::WaitThread( &g_ThreadRxSoc );
        nn::os::DestroyThread( &g_ThreadRxSoc );
        nn::os::WaitThread( &g_ThreadTxSoc );
        nn::os::DestroyThread( &g_ThreadTxSoc );
        nn::wlan::FinalizeSocketManager();  // データパス用のセッションの終了
    }

    // 終了処理
    nn::wlan::FinalizeLocalManager();
} //NOLINT(impl/function_size)

extern "C" void nnMain()
{
    NN_LOG("\n\n Start WlanTest LocalClient \n\n");
    WlanTest::SystemInitialize();
#if 1
    TestCaseLocalSta();
#else
    for( int i = 1; i < 11; i++ )
    {
        NN_LOG("[%d]Start Local Client\n", i);
        TestCaseLocalStaWithDataPath();
    }
#endif
    WlanTest::SystemFinalize();
}

