﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include <nn/oe.h>

#include "Program.h"

namespace WlanTest {

/*---------------------------------------------------------------------------
  Application
---------------------------------------------------------------------------*/
Application::Application()
{
    m_NeedExit = false;
}

Application::~Application()
{
    Exit();
}

void Application::Execute()
{
    Initialize();

    m_NeedExit = false;
    Pad &pad = Pad::GetInstance();
    BluetoothChannelMap &btChannelMap = BluetoothChannelMap::GetInstance();
    while( !m_NeedExit )
    {
        HandleSystemNotifications();
        pad.UpdatePadState();
        btChannelMap.Update();
        ApplicationImpl();
        Sleep(nn::TimeSpan::FromMilliSeconds(10)); // 少しウェイトを入れる
        Display::GetInstance().SwapBuffer();
    }

    Finalize();
}

void Application::Exit()
{
    m_NeedExit = true;
}

void Application::HandleSystemNotifications()
{
    nn::oe::Message message;
    while( nn::oe::TryPopNotificationMessage(&message) )
    {
        switch( message )
        {
        case nn::oe::MessageFocusStateChanged:
            // プログラムのフォーカス状態に変更があった際に通知されます。
            // nn::oe::SetFocusHandlingMode() の指定によっては通知されません。
            {
                auto state = nn::oe::GetCurrentFocusState();
                switch (state)
                {
                case nn::oe::FocusState_InFocus:
                    // インフォーカス状態
                    break;
                case nn::oe::FocusState_OutOfFocus:
                    // アウトフォーカス状態
                    break;
                case nn::oe::FocusState_Background:
                    // BG フォーカス状態
                    break;
                default:
                    break;
                }
            }
            break;

        case nn::oe::MessageResume:
            // アプリケーションが一時中断状態から復帰した際に通知されます。
            // nn::oe::SetResumeNotificationEnabled() の指定によっては通知されません。
            break;

        case nn::oe::MessageOperationModeChanged:
            // 動作モード（携帯／据置）が変更されたことを示す通知です。
            // nn::oe::GetOperationMode() により現在の動作モードを取得することができます。
            // nn::oe::SetOperationModeChangedNotificationEnabled() の指定によっては通知されません。
            break;

        case nn::oe::MessagePerformanceModeChanged:
            // 性能モード（ノーマル／ブースト）が変更されたことを示す通知です。
            // nn::oe::GetPerformanceMode() により現在の性能モードを取得することができます。
            // nn::oe::SetPerformanceModeChangedNotificationEnabled() の指定によっては通知されません。
            break;

        case nn::oe::MessageExitRequest:
            // 自プログラムへの終了要求を示す通知です。
            // このメッセージは事前に nn::oe::EnterExitRequestHandlingSection() を
            // 発行していた場合にのみ通知されます。速やかに終了前の処理を行ない、最終的に、
            // nn::oe::LeaveExitRequestHandlingSection() を発行するとアプリが終了します。
            {
                Exit();
            }
            break;

        default:
            // 未知のメッセージは無視するようにします
            NN_LOG("Unhandled system message = 0x%08x\n", message);
            break;
        }
    }
}

bool Application::IsWlanFlightModeEnabled()
{
    return !nn::nifm::IsWirelessCommunicationEnabled();
}

void Application::SetWlanFlightMode(bool isEnabled)
{
    nn::nifm::SetWirelessCommunicationEnabledForTest(!isEnabled);
}

bool Application::IsBtFlightModeEnabled()
{
    nn::btm::BtmState state;
    nn::btm::GetState(&state);
    return state == nn::btm::BtmState_RadioOff;
}

void Application::SetBtFlightMode(bool isEnabled)
{
    nn::btm::EnableRadio(!isEnabled);
}

/*---------------------------------------------------------------------------
  TestSelector
---------------------------------------------------------------------------*/
TestSelector::TestSelector()
{
    m_Selected = false;

    m_pLocalMasterTest = new LocalMasterTest();
    m_pLocalClientTest = new LocalClientTest();
    m_pLcsMasterTest = new LcsMasterTest();
    m_pLcsClientTest = new LcsClientTest();
    m_pInfraTest = new InfraTest();
    m_pHostDrivenDetectorTest = new HostDrivenDetectorTest();
    m_pLocalScanTest = new LocalScanTest();
    m_pInfraScanTest = new InfraScanTest();
    m_pRcViewer = new RcViewer();

    InitializeInfoPage();

    //##?
    //m_Page.Add(m_Title);
    m_Title.Text = " - Wireless Integrated Test -";
    m_Title.X = 0;
    m_Title.Y = 10;
    m_Title.Alignment = MiddleCenter;
    m_Title.FitSize();

    m_Page.Add(m_Selector);
    m_Selector.X = DISPLAY_CONTENT_START_X;
    m_Selector.Y = DISPLAY_CONTENT_START_Y;
    m_Selector.Height = DISPLAY_CONTENT_HEIGHT;
    m_Selector.Register("Local Master", m_pLocalMasterTest);
    m_Selector.Register("Local Client", m_pLocalClientTest);
    m_Selector.Register("LCS Master", m_pLcsMasterTest);
    m_Selector.Register("LCS Client", m_pLcsClientTest);
    m_Selector.Register("UDP Server/Client", m_pInfraTest);
    m_Selector.Register("Host Driven Aloe", m_pHostDrivenDetectorTest);
    m_Selector.Register("AP Scan(Local)", m_pLocalScanTest);
    m_Selector.Register("AP Scan(Infra)", m_pInfraScanTest);
    m_Selector.Register("RC Viewer", m_pRcViewer);

    m_Page.Add(m_BlankLine);

    m_Page.Add(m_A);
    m_A.X = DISPLAY_PAD_START_X;
    m_A.Y = DISPLAY_PAD_START_Y;
    m_A.Text = "<A> Select";
    m_A.FitSize();

    m_Page.Add(m_WlanFwInfo);
    m_WlanFwInfo.X = Display::GetInstance().GetWidth() / 2 - 90;
    m_WlanFwInfo.Y = DISPLAY_CONTENT_START_Y;
}

TestSelector::~TestSelector()
{
    delete m_pLocalMasterTest;
    delete m_pLocalClientTest;
    delete m_pLcsMasterTest;
    delete m_pLcsClientTest;
    delete m_pInfraTest;
    delete m_pHostDrivenDetectorTest;
    delete m_pLocalScanTest;
    delete m_pInfraScanTest;
    delete m_pRcViewer;
}

void TestSelector::InitializeInfoPage()
{
    uint32_t h = Display::GetInstance().GetFontHeight();

    m_InfoPage.Clear();
    m_InfoPage.Add(m_BlankLine);
    m_InfoPage.Add(m_BlankLine);
    m_BlankLine.Text = " ";
    m_BlankLine.X = 0;
    m_BlankLine.Y = 10;
    m_BlankLine.Alignment = MiddleLeft;
    m_BlankLine.FitSize();

    for(int i=0; i<2; ++i)
    {
        m_InfoPage.Add(m_Separator[i]);
        m_Separator[i].Text = DISPLAY_SEPARATOR;
        m_Separator[i].X = 0 + i * -15;
        m_Separator[i].Y = h;
        m_Separator[i].Alignment = MiddleLeft;
        m_Separator[i].FitSize();
    }

    m_InfoPage.Add(m_Path);
    m_Path.Text = "Home";
    m_Path.X = 5;
    m_Path.Y = 0;
    m_Path.Alignment = MiddleLeft;
    m_Path.FitSize();

    m_InfoPage.Add(m_BlankLine);
}

void TestSelector::ApplicationImpl()
{
    static nn::os::Tick tick;
    static bool isHoldStart;
    static bool isHoldSelect;
    static bool isHoldZr;
    static bool isHoldZl;

    Pad &pad = Pad::GetInstance();
    bool isTriggered =
        pad.IsTrigger(Button::A)  || pad.IsTrigger(Button::B) ||
        pad.IsTrigger(Button::X)  || pad.IsTrigger(Button::Y) ||
        pad.IsTrigger(Button::L)  || pad.IsTrigger(Button::R) ||
        pad.IsTrigger(Button::ZL) || pad.IsTrigger(Button::ZR) ||
        pad.IsTrigger(Button::START) || pad.IsTrigger(Button::SELECT) ||
        pad.IsTrigger(Button::LEFT)  || pad.IsTrigger(Button::UP) ||
        pad.IsTrigger(Button::RIGHT) || pad.IsTrigger(Button::DOWN);

    static bool configMode = false;
    if( pad.IsTrigger(Button::SELECT) )
    {
        static string path;
        configMode = !configMode;
        if( configMode )
        {
            m_ConfigPage.UpdateItems();
            path = m_Path.Text;
            m_Path.Text = "";
            m_Path.FitSize();
        }
        else
        {
            m_Path.Text = path;
            m_Path.FitSize();
        }
    }

    m_InfoPage.Show(Display::GetInstance());

    if( configMode )
    {
        m_ConfigPage.Show(Display::GetInstance());
        Pad::GetInstance().Clear();
    }

    if( !m_Selected )
    {
        m_Page.Show(Display::GetInstance());

        isHoldStart = isHoldStart || pad.IsHold(Button::START);
        isHoldSelect = isHoldSelect || pad.IsHold(Button::SELECT);
        isHoldZl = isHoldZl || pad.IsHold(Button::ZL);
        isHoldZr = isHoldZr || pad.IsHold(Button::ZR);

        if( Pad::GetInstance().IsTrigger(Button::A) )
        {
            Pad::GetInstance().Clear();
            m_Path.Text = m_Selector.GetItemName();
            m_Path.FitSize();
            m_Selected = true;
        }
        else if( Pad::GetInstance().IsTrigger(Button::START) )
        {
            m_Selector.SetCursorIndex(0);
        }
        else if( isHoldStart && isHoldSelect && isHoldZl && isHoldZr )
        {
            Exit();
        }
    }
    else
    {
        if( pad.IsTrigger(Button::START) || m_NeedExit )
        {

#ifdef HID_SHELL_MODE
            static uint64_t testCount = 1;
            static uint64_t modeCount[10] = {};
            NN_ASSERT( sizeof(modeCount) >= m_Selector.GetSize() );
            NN_LOG("# %d : %s\n", testCount++, m_Selector.GetItemName().c_str());
            modeCount[m_Selector.GetCursorIndex()]++;
            m_Selector.GetSelectingItem()->PrintResult();
#endif

            m_Selector.GetSelectingItem()->Exit();
            InitializeInfoPage();
            m_Selected = false;

#ifdef HID_SHELL_MODE
            NN_LOG("\n-----------------------------------------------------\n");
            for(int i=0; i<m_Selector.GetSize(); ++i)
            {
                NN_LOG(" %s : %d\n", m_Selector.GetItemName(i).c_str(), modeCount[i]);
            }
            NN_LOG("-----------------------------------------------------\n");
#endif
        }
        else
        {
            m_Selector.GetSelectingItem()->Execute();
        }
    }

    if( (nn::os::GetSystemTick() - tick).ToTimeSpan() >= nn::TimeSpan::FromMilliSeconds(1000) )
    {
        isHoldStart = false;
        isHoldSelect = false;
        isHoldZr = false;
        isHoldZl = false;
        tick = nn::os::GetSystemTick();
    }

    Display::GetInstance().DisableConsoleOutput();
    if( isTriggered )
    {
        Display::GetInstance().EnableConsoleOutput();
    }
}

/*---------------------------------------------------------------------------
  TestConductor
---------------------------------------------------------------------------*/
TestConductor::TestConductor(Test* pTest)
{
    m_pTest = pTest;
}

TestConductor::~TestConductor()
{
}

void TestConductor::InitializeInfoPage()
{
}

void TestConductor::ApplicationImpl()
{
    if(m_pTest != nullptr)
    {
        bool isDone;
        isDone = Pad::GetInstance().IsTrigger(Button::START) || m_pTest->IsDone();

        if(isDone)
        {
            m_pTest->Exit();
            Exit();
        }
        else
        {
            m_pTest->Execute();
        }
    }

    Display::GetInstance().DisableConsoleOutput();
}

} // namespace WlanTest
