﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Master.h"

namespace {

uint8_t  g_txBuffer[2000];

}
namespace WlanTest {

const char* MasterParam::ITEM_STR_NUM_OF_CLIENTS  = "NUM_OF_CLIENTS";
const char* MasterParam::ITEM_STR_CHANNEL         = "CHANNEL";
const char* MasterParam::ITEM_STR_SSID            = "SSID";
const char* MasterParam::ITEM_STR_SECURITY_MODE   = "SECURITY_MODE";
const char* MasterParam::ITEM_STR_SECURITY_KEY    = "SECURITY_KEY";
const char* MasterParam::ITEM_STR_HIDDEN_SSID     = "STEALTH_MODE";
const char* MasterParam::ITEM_STR_INACTIVE_PERIOD = "INACTIVE_PERIOD";


/*---------------------------------------------------------------------------
　　　　　Master
---------------------------------------------------------------------------*/

Master::Master()
{
    // m_Param.numOfClients = 15;                   //!> 接続を許可する Client 数です。1-8の値です。
    // m_Param.channel = 1;                         //!> 使用するチャンネルです。1-14の値です。
    // m_Param.hiddenSsid = false;                  //!> ステルスSSID設定フラグです。 false:any true:hidden
    // m_Param.defaultAuthorize = true;             //!> Master 開始後 Client が接続した際に Authorize 状態にします。false:Unauthorize true:Authorize
    // m_Param.inactivePeriod = 10;                 //!> 通信がないClientを切断するまでの時間[sec]です。
    // m_Param.supportedRates = RATE_SET_11G_MASK;  //!> IEEE802.11 Supported Rates です。
    // m_Param.basicRates = RATE_SET_11B_MASK;      //!> IEEE802.11 Supported Rates の Basic Rates を示します。
    // m_Param.security = Security();               //!> 接続先のセキュリティプロファイル
    // m_Param.ssid = Ssid("sap_master");           //!> 使用する SSID です。
    // m_Param.beaconIntervalTu = 100;              //!> 使用する Beacon Interval[TU] です。
    // m_Ctw = 10;

    // m_Param.numOfClients = 15;                   //!> 接続を許可する Client 数です。1-8の値です。
    m_Param.channel = 1;                           //!> 使用するチャンネルです。1-14の値です。
    m_Param.hiddenSsid = false;                    //!> ステルスSSID設定フラグです。 false:any true:hidden
    // m_Param.defaultAuthorize = true;             //!> Master 開始後 Client が接続した際に Authorize 状態にします。false:Unauthorize true:Authorize
    m_Param.inactivePeriod = 10;                   //!> 通信がないClientを切断するまでの時間[sec]です。
    m_Param.autoKeepAlive = false; // inactivePeriod = 0で同等のことが出来るので使用しない。（一応値は入れておく）
    m_Param.supportedRates = nn::wlan::RateSetLegacy_11gMask;  //!> IEEE802.11 Supported Rates です。
    m_Param.basicRates = nn::wlan::RateSetLegacy_11gMask;      //!> IEEE802.11 Supported Rates の Basic Rates を示します。
    m_Param.security = nn::wlan::Security();       //!> 接続先のセキュリティプロファイル
    m_Param.ssid = nn::wlan::Ssid("sap_master");   //!> 使用する SSID です。

    m_NumOfClients = 1;
    // m_Param.beaconIntervalTu = 100;                //!> 使用する Beacon Interval[TU] です。
    m_Param.beaconInterval = 100;                  //!> 使用する Beacon Interval[TU] です。
    m_Ctw = 10;
    // SapTxQueueParam param;
    // param.eCWmin = 4;
    // param.eCWmax = 10;
    // param.aifsn = 2;
    // param.queueType = TX_QUEUE_1; // toriaezu
    // m_TxQueueParam[0] = param;
    // m_TxQueueParam[1] = param;
    // m_TxQueueParam[0].queueType = TX_QUEUE_1;
    // m_TxQueueParam[1].queueType = TX_QUEUE_2;
    m_NumOfDeauth = 3;
    m_Updated = 0x00;
    m_ActionFrameSize = 0;
    std::memset(m_respondOui, 0x00, sizeof(m_respondOui));
    std::memset(m_ClientList, 0x00, sizeof(m_ClientList));
    std::memset(m_ClientStatus, 0x00, sizeof(m_ClientStatus));

    m_IsWaiting = false;
}

Master::Master(nn::wlan::MasterBssParameters param)
{
    m_Param = param;
}

Master::~Master()
{
}

nn::Result Master::Initialize()
{
    nn::Result result = WlanInitialize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanInitialize()\n");
        return result;
    }

    // 前回の BSS 構築処理時間の初期化
    m_ConnectionTime = 0;
    std::memset(&m_Param, 0, sizeof(m_Param));

    ModeChanger::ToLocalMasterMode();
    result = nn::wlan::Socket::EnableTsfTimerFunction();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : EnableTsfTimerFunction()\n");
        return result;
    }

    // 送受信する際に Rx エントリが必須なので初期化時に作成する
    CreateRxEntry();

    m_IsInitialized = true;

    return result;
}

nn::Result Master::Finalize()
{
    nn::Result result = DeleteRxEntry();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : DeleteRxEntry()\n");
    }

    result = nn::wlan::Socket::DisableTsfTimerFunction();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : DisableTsfTimerFunction()\n");
    }

    result = WlanFinalize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanFinalize()\n");
    }

    ModeChanger::ToReadyMode();
    m_IsInitialized = false;

    return result;
}

nn::Result Master::Open()
{
    nn::Result result = nn::ResultSuccess();

    // 接続可能なクライアント数を設定
    if( m_NumOfClients != ConnectableClientCount_Any )
    {
        nn::Result setResult = nn::wlan::Local::SetMaxAssociationNumber(m_NumOfClients);
        NN_ASSERT(setResult.IsSuccess());
    }

    char buf[nn::wlan::Ssid::SsidHexStringLengthMax] = {0};
    NN_LOG(" === CreateBss Params ===============================\n");
    NN_LOG("   Channel           : %d\n", m_Param.channel);
    NN_LOG("   Hidden ssid       : %d\n", m_Param.hiddenSsid);
    NN_LOG("   Inactive period   : %u\n", m_Param.inactivePeriod);
    NN_LOG("   Auto keep alive   : %d\n", m_Param.autoKeepAlive);
    NN_LOG("   Security mode     : %d\n", m_Param.security.groupPrivacyMode);
    NN_LOG("   Security key      : %s\n", reinterpret_cast<char*>(m_Param.security.key));
    NN_LOG("   Ssid              : %s\n", m_Param.ssid.GetHexString(buf));
    NN_LOG("   Beacon interval   : %d\n", m_Param.beaconInterval);
    NN_LOG("   Action frame Size : %d\n", m_ActionFrameSize);
    NN_LOG(" ====================================================\n");

    nn::os::Tick startTick = nn::os::GetSystemTick();
    result = nn::wlan::Local::CreateBss(m_Param);
    nn::os::Tick endTick = nn::os::GetSystemTick();

    if( result.IsSuccess() )
    {
        // CreateBss() の時間を計測
        nn::wlan::ConnectionStatus connectionStatus;
        nn::Result conResult = nn::wlan::Local::GetConnectionStatus(&connectionStatus);
        if( conResult.IsSuccess() && connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            m_ConnectionTime = (endTick - startTick).ToTimeSpan().GetMilliSeconds();

            // Auto の場合もあるので改めてチャネルを取得する
            m_Param.channel = connectionStatus.channel;
        }
    }
    else
    {
        NN_LOG(" - failed : CreateBss\n");
        WlanTest::TPrintResult(result);
        return result;
    }

    //        cs.Unlock();

    return result;
}

nn::Result Master::Close()
{
    nn::Result result;

    for(int j=0; j<sizeof(m_ClientStatus) / sizeof(m_ClientStatus[0]); j++)
    {
        if(m_ClientStatus[j].state == nn::wlan::ConnectionState_Connected)
        {
            nn::wlan::DisconnectClient client;
            client.clientMacAddress = m_ClientStatus[j].clientMacAddress;
            client.reasonCode = nn::wlan::Dot11ReasonCode_UnspecifiedFailure;

            char str[nn::wlan::MacAddress::MacStringSize];
            NN_LOG("Disconnect client : %s, reason : %d\n", client.clientMacAddress.GetString(str), client.reasonCode);
            for(uint8_t i=0; i<m_NumOfDeauth; i++)
            {
                Disconnect(&client);
            }
        }
    }

    // 更新値を次回のマスター起動時に持ち越さないためここで読み込む
    nn::wlan::Local::GetClientStatus(m_ClientStatus, &m_Updated);
    std::memset(m_ClientList, 0x00, sizeof(m_ClientList));
    std::memset(m_ClientStatus, 0x00, sizeof(m_ClientStatus));
    m_Updated = 0x00;

    result = nn::wlan::Local::DestroyBss();
    if (result.IsFailure())
    {
        NN_LOG(" - failed : DestroyBss\n");
        WlanTest::TPrintResult(result);
    }
    else
    {
        //NN_LOG(" - success : DestroyBss\n");
    }

    return result;
}

nn::Result Master::Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize)
{
    nn::Result result;

    std::memcpy(((DIX_FRAME*)data)->Source, GetMacAddress().GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

    //    パケットの中身を表示する
    // for(int i=0; i<30; i++)
    // {
    //    NN_LOG("%02x ", data[i]);
    //       if( i % 8 == 7)
    //           NN_LOG("\n");
    // }
    // NN_LOG("\n");

    nn::wlan::MacAddress destination(((DIX_FRAME*)data)->Destination);
    if(ieInd)//IEデータ
    {
        // アプリから送信可能な IE のサイズ
        const uint16_t IE_SIZE = 254;
        // ヘッダとして EID, Length の定義はあるが IE として送信しない
        const uint16_t IE_HEADER_SIZE = 2 + 4;
        // IE として送信しない EID, Length の 2 byte は payload として埋める
        const uint16_t IE_PAYLOAD_SIZE = IE_SIZE - IE_HEADER_SIZE + 2;
        const uint16_t IE_NUM = 4;
        const uint16_t IE_SUB_TYPE_MIN = 0x15;

        if(size > (IE_PAYLOAD_SIZE * IE_NUM))
        {
            NN_LOG("Can't send because data size too large.\n");
        }

        uint32_t beaconIeIdx = 0;
        uint32_t assocResIeIdx = 0;
        uint32_t offset = 0;
        // data を IE にコピーするために、必要な IE の数を計算(付加できる IE の最大数は 4)
        // 下記ではひとつの IE につき 250 byte 分 を data 領域として使用する
        //uint16_t numIe = nn::math::Min(IE_NUM, static_cast<uint16_t>(ceil(static_cast<f32>(size)/IE_PAYLOAD_SIZE)));
        uint16_t numIe = std::min(IE_NUM, static_cast<uint16_t>(ceil(static_cast<float>(size) / IE_PAYLOAD_SIZE)));

        for(int i=0; i<numIe; i++)
        {
            NintendoIE ie(( i != numIe - 1 ? IE_PAYLOAD_SIZE : size - offset) + IE_HEADER_SIZE);
            ie.SubType = IE_SUB_TYPE_MIN + i;
            std::memcpy(ie.Data, data + offset, ie.GetSize() - IE_HEADER_SIZE);

            // EID, Length の代わりに送信する payload の 2 byte 分をプラス
            nn::Bit8 buf[IE_SIZE + 2];
            ie.GetBinary(buf, IE_SIZE + 2);

            result = nn::wlan::Local::AddIe(&beaconIeIdx, nn::wlan::ManagementFrameType_Beacon, buf + 2, ie.GetSize() - 2); // ElementID, Length分2byte飛ばす
            if (result.IsFailure())
            {
                NN_LOG(" - failed : AddIe( BEACON )\n");
                WlanTest::TPrintResult(result);
            }

            result = nn::wlan::Local::AddIe(&assocResIeIdx, nn::wlan::ManagementFrameType_AssocRes, buf + 2, ie.GetSize() - 2); // ElementID, Length分2byte飛ばす
            if (result.IsFailure())
            {
                NN_LOG(" - failed : AddIe( ASSOC_RESP )\n");
                WlanTest::TPrintResult(result);
            }
            offset += (ie.GetSize() - IE_HEADER_SIZE);
        }
        // result = Sap::SetIes( MANAGEMENT_FRAME_TYPE_BEACON );
        // if (result.IsFailure())
        // {
        //     NN_LOG(" - failed : SetIes( BEACON )\n");
        //     WlanTest::TPrintResult(result);
        // }
        // result = Sap::SetIes( MANAGEMENT_FRAME_TYPE_ASSOC_RESP );
        // if (result.IsFailure())
        // {
        //     NN_LOG(" - failed : SetIes( ASSOC_RESP )\n");
        //     WlanTest::TPrintResult(result);
        // }

        //m_BeaconSentEvent.Wait();

        //result = Sap::RemoveIeFromIesBuffer( MANAGEMENT_FRAME_TYPE_BEACON, IE_NUM ); // 4を指定すると全クリア
        result = nn::wlan::Local::DeleteIe(beaconIeIdx);
        if (result.IsFailure())
        {
            NN_LOG(" - failed : DeleteIe( BEACON )\n");
            WlanTest::TPrintResult(result);
        }
        //result = Sap::RemoveIeFromIesBuffer( MANAGEMENT_FRAME_TYPE_ASSOC_RESP, IE_NUM ); // 4を指定すると全クリア
        result = nn::wlan::Local::DeleteIe(assocResIeIdx);
        if (result.IsFailure())
        {
            NN_LOG(" - failed : DeleteIe( ASSOC_RESP )\n");
            WlanTest::TPrintResult(result);
        }
        // result = Sap::SetIes( MANAGEMENT_FRAME_TYPE_BEACON );
        // if (result.IsFailure())
        // {
        //     NN_LOG(" - failed : SetIes( BEACON )\n");
        //     WlanTest::TPrintResult(result);
        // }
        // result = Sap::SetIes( MANAGEMENT_FRAME_TYPE_ASSOC_RESP );
        // if (result.IsFailure())
        // {
        //     NN_LOG(" - failed : SetIes( ASSOC_RESP )\n");
        //     WlanTest::TPrintResult(result);
        // }
    }
    else // その他宛は通常データとして扱う
    {
        //NN_LOG("PutFrame()\n");

        // for(int i=0; i<30; i++)
        // {
        //     NN_LOG("%02x ", data[i]);
        //     if( i % 8 == 7)
        //         NN_LOG("\n");
        // }
        // NN_LOG("\n\n");

        result = WlanPutFrame(data, size);
        //if(result.IsSuccess())
        //NN_LOG(" - success : PutFrame\n");
        //else
        //NN_LOG(" - failed : PutFrame\n");
    }

    if( result.IsFailure() )
    {
        m_Statistics.SendErrorCount++;
        // NN_LOG("PutFrameRaw failed\n");
        // WlanTest::TPrintResult(result);
        Sleep(nn::TimeSpan::FromMilliSeconds(1));    // 切断時の連続送信失敗を避けるため、スリープする
    }
    else
    {
        if(m_Statistics.SendCount == 0)
        {
            m_Statistics.FirstSendTime = nn::os::GetSystemTick();
        }
        m_Statistics.LastSendTime = nn::os::GetSystemTick();
        m_Statistics.SendCount++;
        m_Statistics.SendSize += size;
        *pSentSize = size;
    }

    return result;
}

void Master::MaintainConnection()
{
    m_RequestStop = false;
    m_IsStop = false;

    nn::os::SystemEventType connectionEvent;
    nn::Result result;

    SetActionFrame();
    Open();

    result = nn::wlan::Local::GetConnectionEvent(&connectionEvent);
    if( result.IsFailure() )
    {
        NN_LOG("GetConnectionEvent failed\n");
        WlanTest::TPrintResult(result);
        goto End;
    }

    while( !m_RequestStop )
    {
        nn::wlan::WlanState state = nn::wlan::WlanState_Stop;
        result = nn::wlan::Local::GetState(&state);
        if( result.IsFailure() )
        {
            NN_LOG("GetState failed\n");
            WlanTest::TPrintResult(result);
            goto End;
        }

        m_Cs.Lock();

        result = nn::wlan::Local::GetClientStatus(m_ClientStatus, &m_Updated);
        if( result.IsFailure() )
        {
            NN_LOG("GetClientStatus failed\n");
            WlanTest::TPrintResult(result);
        }

        for(int i = 0; i < sizeof(m_ClientStatus) / sizeof(m_ClientStatus[0]); i++)
        {
            if( (m_Updated >> i) & 1 )
            {
                if(m_ClientStatus[i].state == nn::wlan::ConnectionState_Connected)
                {
                    m_ClientList[i] = m_ClientStatus[i].clientMacAddress;
                    m_Statistics.ConnectCount++;
                }
                else
                {
                    m_ClientList[i] = nn::wlan::MacAddress::CreateZeroMacAddress();
                    m_Statistics.Disconnected++;
                }

                char adr[nn::wlan::MacAddress::MacStringSize];
                if(m_ClientStatus[i].state == nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("CONNECTED : %s\n", m_ClientStatus[i].clientMacAddress.GetString(adr));
                }
                else if(m_ClientStatus[i].state == nn::wlan::ConnectionState_Disconnected)
                {
                    NN_LOG("DISCONNECTED : %s\n", m_ClientStatus[i].clientMacAddress.GetString(adr));
                }
                NN_LOG("    State       : %d\n", m_ClientStatus[i].state);
                NN_LOG("    Cause       : %d\n", m_ClientStatus[i].cause);
                NN_LOG("    Reason      : %d\n", m_ClientStatus[i].statusReasonCode);
                NN_LOG("    Capability  : 0x%x\n", m_ClientStatus[i].capabilityInfo);
                //NN_LOG("    ListenIntvl : %d\n", m_ClientStatus[i].listenInterval);
                NN_LOG("\n");
                NN_UNUSED(adr); // ワーニング対策
            }
        }
        m_Cs.Unlock();
        TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(100));
    }
End:
    m_RequestStop = true;
    m_IsStop = true;
    nn::os::DestroySystemEvent(&connectionEvent);
}

nn::Result Master::Disconnect( const nn::wlan::DisconnectClient* pClient )
{
    nn::Result result;
    result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, pClient );
    return result;
}

nn::Result Master::Disconnect()
{
    nn::wlan::DisconnectClient client;
    client.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    //#?? Dot11ReasonCode を指定しないといけないため、とりあえず下記を指定する
    client.reasonCode = nn::wlan::Dot11ReasonCode_UnspecifiedFailure;
    return Disconnect( &client );
}

bool Master::IsConnected()
{
    nn::wlan::ConnectionStatus connectionStatus;
    nn::Result result = nn::wlan::Local::GetConnectionStatus(&connectionStatus);

    if( result.IsFailure() )
    {
        return false;
    }

    return connectionStatus.state == nn::wlan::ConnectionState_Connected;
}

void Master::SetActionFrame()
{
    if( IsActionFrameEnabled() )
    {
        NN_LOG("Set Action Frame\n");
        NN_ASSERT(m_ActionFrameSize < sizeof(g_txBuffer), "  - failed : size %u, buf %u\n", m_ActionFrameSize, sizeof(g_txBuffer));
        std::memset(g_txBuffer, 0x00, sizeof(g_txBuffer));
        g_txBuffer[0] = 0x7F; // Category
        g_txBuffer[1] = 0x00; // OUI1
        g_txBuffer[2] = 0x22; // OUI2
        g_txBuffer[3] = 0xAA; // OUI3
        g_txBuffer[4] = nn::wlan::ActionFrameType_Beacon;  // Subtype

        nn::wlan::Local::SetActionFrameWithBeacon(reinterpret_cast<uint8_t*>(g_txBuffer), m_ActionFrameSize);
    }
}

}
