﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "LcsClient.h"

namespace WlanTest {

/*---------------------------------------------------------------------------
　　　　　LcsClient
---------------------------------------------------------------------------*/
LcsClient::LcsClient()
{
    m_Ssid = nn::wlan::Ssid("sap_master");
    m_Bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    m_Channel = 1;
    m_Security.privacyMode = nn::wlan::SecurityMode_Open;
    m_Security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
    std::memset(m_Security.key, 0x00, sizeof(m_Security.key));
    m_Indication = nn::wlan::BeaconIndication_Enable;
    m_EnableDirectBroadcast = false;
    m_EnableClr = false;
    m_TxWait = 5;
    m_IsWaiting = false;
    m_BeaconLostTimeout = 10;
    m_ConnectionTime = 0;
    m_ReceiverHandler = nullptr;

    m_IsIpAddrAssigned = false;
    m_Socket.SetSocketMode(SocketModeClient);
    m_Socket.SetSocketType(SOCKET_TYPE_TCP);
    m_Socket.SetSrcIpAddress("0.0.0.0");
    m_Socket.SetDstIpAddress("169.254.1.1");
    m_Socket.SetSubnetMask("255.255.255.0");
    m_Socket.SetDns1("0.0.0.0");
    m_Socket.SetDns2("0.0.0.0");
    m_Socket.SetGwIpAddress("169.254.1.1");
    m_Socket.SetPort(50222);
}

LcsClient::~LcsClient()
{
}

nn::Result LcsClient::Initialize()
{
    nn::Result result = WlanInitialize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanInitialize()\n");
        return result;
    }

    // 前回の接続処理時間の初期化
    m_ConnectionTime = 0;
    m_Channel = 0;

    ModeChanger::ToLcsClientMode();

    // 送受信する際に Rx エントリが必須なので初期化時に作成する
    CreateRxEntry();

    m_Socket.SetSrcIpAddress("0.0.0.0");
    m_IsIpAddrAssigned = false;
    m_IsInitialized = true;

    return result;
}

nn::Result LcsClient::Finalize()
{
    nn::Result result = DeleteRxEntry();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : DeleteRxEntry()\n");
    }

    result = WlanFinalize();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : LocalNode::WlanFinalize()\n");
    }

    ModeChanger::ToReadyMode();
    m_IsInitialized = false;

    return result;
}

void LcsClient::MaintainConnection()
{
    if( !m_IsInitialized )
    {
        return;
    }

    nn::os::SystemEventType connectionEvent;
    nn::Result result;
    nn::wlan::ConnectionState lastState = nn::wlan::ConnectionState_Disconnected;
    m_ConnectionStatus.state = nn::wlan::ConnectionState_Disconnected;
    m_ConnectionStatus.bssid = nn::wlan::MacAddress::CreateZeroMacAddress();

    result = nn::wlan::Local::GetConnectionEvent(&connectionEvent);
    if( result.IsFailure() )
    {
        NN_LOG("GetConnectionEvent failed\n");
        WlanTest::TPrintResult(result);
        NN_ASSERT(false);
    }

    m_State = State_Disconnected;
    m_RequestStop = false;
    m_IsStop = false;

    while( !m_RequestStop )
    {
        // 接続しているか調べる
        result = nn::wlan::Local::GetConnectionStatus(&m_ConnectionStatus);
        if( result.IsFailure() )
        {
            NN_LOG("GetConnectionStatus failed\n");
            WlanTest::TPrintResult(result);
        }

        m_Cs.Lock();

        TransitState();

        if( m_ConnectionStatus.state != lastState )
        {
            if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                m_Statistics.ConnectCount++;
            }

            if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Idle &&
               lastState == nn::wlan::ConnectionState_Connected )
            {
                m_Statistics.Disconnected++;
                m_State = State_Disconnecting;
            }
        }
        lastState = m_ConnectionStatus.state;

        m_Cs.Unlock();

        TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(100));
    }

    m_Socket.Close();

    //##?
    // 保持しているコネクションステータスを切断にする
    // ここで取得しなければ、切断後も接続中のものが残ってしまう
    result = nn::wlan::Local::GetConnectionStatus(&m_ConnectionStatus);
    if( result.IsFailure() )
    {
        NN_LOG("GetConnectionStatus failed\n");
        WlanTest::TPrintResult(result);
    }

    m_IsStop = true;

    nn::os::DestroySystemEvent(&connectionEvent);
}

void LcsClient::StopMaintainConnection()
{
    if( !m_IsInitialized )
    {
        return;
    }

    if( !m_IsStop )
    {
        m_RequestStop = true;
        nn::os::DestroyThread(&m_MaintainConnectionThread);
        Close();
    }
}

void LcsClient::TransitState()
{
    nn::Result result;

    switch( m_State )
    {
    case State_Disconnected :
        {
            // L2 の接続試行
            Open();

            // 接続できた場合は Connect() を抜けた時点で接続が完了していることが保障されている
            result = nn::wlan::Local::GetConnectionStatus(&m_ConnectionStatus);
            if( result.IsSuccess() )
            {
                if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected )
                {
                    NN_LOG("   L2Connected\n");
                    m_State = State_L2Connected;
                }
                else
                {
                    m_Statistics.ConnectErrorCount++;
                }
            }
        }
        break;

    case State_L2Connected :
        {
            // Master から IP アドレスが割り当てられるのを待つ
            if( m_IsIpAddrAssigned )
            {
                NN_LOG("   IP Address Assigened\n");
                m_State = State_L3Completed;
            }
        }
        break;

    case State_L3Completed :
        {
            // L4 の接続試行
            m_Socket.Open();
            m_State = State_L4Connecting;
        }
        break;

    case State_L4Connecting :
        {
            int ret = m_Socket.Connect();
            if( ret >= 0 )
            {
                NN_LOG("   L4 Connected\n");
                m_State = State_L4Connected;
            }
        }
        break;

    case State_L4Connected :
        {
        }
        break;

    case State_Disconnecting :
        {
            NN_LOG("   Disconnecting\n");
            m_Socket.Close();
            m_IsIpAddrAssigned = false;
            m_Socket.SetSrcIpAddress("0.0.0.0");
            m_State = State_Disconnected;
        }
        break;

    default :
        {
        }
        break;
    }
}

nn::Result LcsClient::Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize)
{
    nn::Result result = nn::ResultSuccess();
    result = m_Socket.Send(data, size, pSentSize);

    if( result.IsFailure() )
    {
        m_Statistics.SendErrorCount++;
        // NN_LOG("PutFrameRaw failed\n");
        // WlanTest::TPrintResult(result);
        Sleep(nn::TimeSpan::FromMicroSeconds(10));    // 切断時の連続送信失敗を避けるため、スリープする
    }
    else
    {
        if(m_Statistics.SendCount == 0)
        {
            m_Statistics.FirstSendTime = nn::os::GetSystemTick();
        }
        m_Statistics.LastSendTime = nn::os::GetSystemTick();
        m_Statistics.SendCount++;
        m_Statistics.SendSize += size;
    }

    return result;
}

nn::Result LcsClient::Receive(uint8_t pOutput[], size_t* pSize, const size_t maxSize)
{
    nn::Result result = nn::ResultSuccess();
    result = m_Socket.Receive(pOutput, pSize, maxSize);

    return result;
}

nn::Result LcsClient::WlanGetFrame( uint32_t rxId, uint8_t pOutput[], size_t* pSize, size_t maxSize, int8_t* pRssi)
{
    NN_UNUSED(rxId);
    NN_UNUSED(pRssi);

    return Receive(pOutput, pSize, maxSize);
}

bool LcsClient::IsConnected()
{
    return m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected;
}

nn::Result LcsClient::RunAssignSrcIpAddressCommand(const string& ipAddress)
{
    m_Socket.SetSrcIpAddress(ipAddress);
    m_IsIpAddrAssigned = true;

    return nn::ResultSuccess();
}

}
