﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Invoker.h"

namespace {

NN_ALIGNAS(4096) char s_InvokeThreadStack[4096 * 10];

}

namespace WlanTest {

/*---------------------------------------------------------------------------
　　　　　Command
---------------------------------------------------------------------------*/

Command::Command()
{
    m_Count = 0;
    m_Number = 1;
}

void Command::Invoke()
{
    m_Count++;
    CommandContent();
}

void Command::CommandContent()
{
}

void Command::SetNumber(uint32_t n)
{
    m_Number = n;
}

bool Command::NeedReinvoke()
{
    if( m_Number == 0 || (m_Number - m_Count) > 0 )
    {
        return true;
    }
    else
    {
        return false;
    }
}


/*!--------------------------------------------------------------------------*
  @brief        ログ表示コマンド

 *---------------------------------------------------------------------------*/
/*
class LogCommand : public Command
{
private:
    char m_String[32];

public:
    virtual void CommandContent()
    {
        NN_LOG("LogCommand : %s\n", m_String);
    }

    virtual size_t GetInstanceSize() const
    {
        return sizeof(LogCommand);
    }

    LogCommand(std::string str)
    {
        std::memcpy(m_String, str.c_str(), str.size() + 1);
    }
};
*/


/*---------------------------------------------------------------------------
　　　　　Invoker
---------------------------------------------------------------------------*/
uint8_t Invoker::m_sId = 0;

Invoker::Invoker() : m_AddCommandEvent(nn::os::EventClearMode_AutoClear)
{
    m_Id = m_sId++;
    NN_ASSERT(m_Id < 10);

    m_Stop = true;
    m_Count = 0;
    m_CommandLimit = 256;
}

Invoker::~Invoker()
{
    Stop();
    Clear();
}

void Invoker::InvokeThreadFunction(void* invoker)
{
    reinterpret_cast<Invoker*>(invoker)->InvokeCommands();
}

nn::Result Invoker::Start(int32_t priority)
{
    if( !m_Stop )
    {
        NN_LOG("  - failed : InvokeThread already running.\n");
        NN_ASSERT(false);
        return WlanTest::ResultBusy();
    }

    m_Stop = false;

    char* ptr = s_InvokeThreadStack + (4096 * m_Id);
    nn::Result result = nn::os::CreateThread(&m_Thread, InvokeThreadFunction, this, ptr, 4096, priority);

    if( result.IsFailure() )
    {
        NN_LOG("  - failed : Cannot create InvokeThread.\n");
        NN_ASSERT(false);
        return WlanTest::ResultFailure();
    }

    static char threadName[] = "WitInvokeThreadX";
    threadName[strlen(threadName) - 1] = '0' + m_Id;
    nn::os::SetThreadNamePointer(&m_Thread, threadName);
    nn::os::StartThread(&m_Thread);

    return result;
}

void Invoker::Stop()
{
    if( m_Stop )
    {
        return;
    }

    m_Stop = true;
    m_AddCommandEvent.Signal();

    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);
}

nn::Result Invoker::AddCommand( const Command& command )
{
    if( m_Count == m_CommandLimit )
    {
        NN_LOG("Command full.\n");
        return WlanTest::ResultOutOfResource();
    }

    void* c = std::malloc(command.GetInstanceSize());
    if( c == NULL )
    {
        NN_LOG("malloc failure.\n");
        return WlanTest::ResultOutOfResource();
    }

    std::memcpy(c, reinterpret_cast<const void*>(&command), command.GetInstanceSize());

    m_Count++;
    m_CommandList.push( (Command*)c );
    m_AddCommandEvent.Signal();

    return nn::ResultSuccess();
}

void Invoker::Clear()
{
    if( !m_Stop )
    {
        NN_LOG("must stop before clear.\n");
        return;
    }

    while( !m_CommandList.empty() )
    {
        Command* c = m_CommandList.front();
        m_CommandList.pop();
        std::free(c);
    }

    m_Count = 0;
}

void Invoker::InvokeCommands()
{
    while( !m_Stop )
    {
        if( !m_CommandList.empty() )
        {
            Command* c = m_CommandList.front();
            c->Invoke();
            m_CommandList.pop();

            if( c->NeedReinvoke() )
            {
                m_CommandList.push(c);
                nn::os::YieldThread();
            }
            else
            {
                std::free(c);
                m_Count--;
            }
        }
        else
        {
            m_AddCommandEvent.Wait();
        }
    }
}

uint32_t Invoker::GetCount()
{
    return m_Count;
}


}
