﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include "Counter.h"

using namespace std;

namespace WlanTest {

/*---------------------------------------------------------------------------
　　　　　Counter
---------------------------------------------------------------------------*/

Counter::Counter()
{
    Clear();
};

void Counter::Count(int64_t val)
{
    m_Count += val;
}

int64_t Counter::GetValue() const
{
    return m_Count;
}

void Counter::Clear()
{
    m_Count = 0;
}




/*---------------------------------------------------------------------------
　　　　　SequenceNumberCounter
---------------------------------------------------------------------------*/

SequenceNumberCounter::SequenceNumberCounter()
{
    count = 0;
    max = 0;
    min = 0xffffffffffffffff; // uint64_tの最大値
    last = 0xffffffffffffffff;
    for(int i=0; i<=16; i++)
    {
        burst[i] = 0;
    }
}

SequenceNumberCounter::~SequenceNumberCounter()
{
}

void SequenceNumberCounter::Count(uint64_t seq)
{
    // 重複チェック
    //if(seq == last)
    //    return;

    // バーストエラーカウント
    uint32_t err;
    if(last == 0xffffffffffffffff)
    {
        err = 0;
    }
    else
    {
        err = seq - last - 1;
    }

    if(err > 0 && err < 16)
    {
        burst[err]++;
    }
    else if(err >= 16)
    {
        burst[16]++;
    }

    if (err > burst[0])
    {
        burst[0] = err;
    }

    last = seq;

    if(seq > max)
        max = seq;

    if(seq < min)
    {
        min = seq;
    }

    count++;
}

double SequenceNumberCounter::GetErrorRate() const
{
    if(GetExpectantCount() == 0)
    {
        return 0.f;
    }
    else
    {
        return 1 - GetRate();
    }
}

double SequenceNumberCounter::GetRate() const
{
    return static_cast<double>(count) / GetExpectantCount();
}

uint64_t SequenceNumberCounter::GetReceiveCount() const
{
    return count;
}

uint64_t SequenceNumberCounter::GetErrorCount() const
{
    return GetExpectantCount() - count;
}

uint64_t SequenceNumberCounter::GetExpectantCount() const
{
    if(last == 0xffffffffffffffff)
    {
        return 0;
    }

    return max - min + 1;
}

uint32_t SequenceNumberCounter::GetBurstErrorCount(uint8_t index) const
{
    if(index <= 16)
    {
        return burst[index];
    }
    else
    {
        return 0;
    }
}

void SequenceNumberCounter::Clear()
{
    count = 0;
    max = 0;
    min = 0xffffffffffffffff; // uint64_tの最大値
    last = 0xffffffffffffffff;
    for(int i=0; i<=16; i++)
    {
        burst[i] = 0;
    }
}



/*---------------------------------------------------------------------------
　　　　　LastSequenceNumberCounter
---------------------------------------------------------------------------*/

LastSequenceNumberCounter::LastSequenceNumberCounter(size_t size)
{
    this->size = size;
    counters = new SequenceNumberCounter [size];
    index = 0;
    last = 0xffffffffffffffff;
}

LastSequenceNumberCounter::~LastSequenceNumberCounter()
{
    delete [] counters;
}

void LastSequenceNumberCounter::Count(uint64_t seq)
{
    // 重複チェック
    if(seq == last)
        return;
    last = seq;

    counters[index].Clear();

    for(int i=0; i<size; i++)
    {
        counters[i].Count(seq);
    }

    index = (index + 1) % size;
}

double LastSequenceNumberCounter::GetErrorRate() const
{
    return counters[index].GetErrorRate();
}

double LastSequenceNumberCounter::GetRate() const
{
    return counters[index].GetRate();
}

uint64_t LastSequenceNumberCounter::GetErrorCount() const
{
    return counters[index].GetErrorCount();
}

uint64_t LastSequenceNumberCounter::GetExpectantCount() const
{
    return counters[index].GetExpectantCount();
}

void LastSequenceNumberCounter::Clear()
{
    for(int i=0; i<size; i++)
    {
        counters[i].Clear();
    }
    index = 0;
}

void LastSequenceNumberCounter::SetSize(size_t size)
{
    this->size = size;
    Clear();
}



/*---------------------------------------------------------------------------
　　　　　CyclicSequenceNumberCounter
---------------------------------------------------------------------------*/

CyclicSequenceNumberCounter::CyclicSequenceNumberCounter()
{
    Clear();
    SetEndNo(0xff); // 1byte
}

CyclicSequenceNumberCounter::~CyclicSequenceNumberCounter()
{
}

void CyclicSequenceNumberCounter::Count(uint64_t seq)
{
    // バーストエラーカウント
    uint32_t err = 0;

    if( first )
    {
        first = false;
        err = 0;
    }
    else if( seq < last )
    {
        // カウンタが一周していた場合
        err = (seq + endNo) - last - startNo;
    }
    else if( seq == last )
    {
        // 再送の場合は何もしない
        return;
    }
    else
    {
        err = seq - last - 1;
    }

    if( err > 0 && err < 16 )
    {
        burst[err]++;
    }
    else if( err >= 16 )
    {
        burst[16]++;
    }

    if( err > burst[0] )
    {
        burst[0] = err;
    }

    last = seq;

    lost += err;
    count++;
}

double CyclicSequenceNumberCounter::GetErrorRate() const
{
    return 1 - GetRate();
}

double CyclicSequenceNumberCounter::GetRate() const
{
    return static_cast<double>(count) / GetExpectantCount();
}

uint64_t CyclicSequenceNumberCounter::GetReceivedCount() const
{
    return count;
}

uint64_t CyclicSequenceNumberCounter::GetErrorCount() const
{
    return lost;
}

uint64_t CyclicSequenceNumberCounter::GetExpectantCount() const
{
    return count + lost;
}

uint32_t CyclicSequenceNumberCounter::GetBurstErrorCount(uint8_t index) const
{
    if( index <= 16 )
    {
        return burst[index];
    }
    else
    {
        return 0;
    }
}

void CyclicSequenceNumberCounter::Clear()
{
    first = true;
    startNo = 0;
    count = 0;
    lost = 0;
    last = 0;
    for(int i=0; i<=16; i++)
    {
        burst[i] = 0;
    }
}



/*---------------------------------------------------------------------------
　　　　　ThroughputMeter
---------------------------------------------------------------------------*/

ThroughputMeter::ThroughputMeter()
{
    totalSize = 0;
    begin = nn::os::GetSystemTick();
    end = begin;
}

ThroughputMeter::~ThroughputMeter()
{
}

void ThroughputMeter::Count(uint64_t size)
{
    nn::os::Tick tick = nn::os::GetSystemTick();
    if( totalSize == 0 )
    {
        begin = tick;
    }
    totalSize += size;
    end = tick;
}

double ThroughputMeter::GetThroughput() const
{
    return static_cast<double>(totalSize) * 8.0f * 1000.0f / (1024.0f * 1024.0f) / static_cast<double>((end - begin).ToTimeSpan().GetMilliSeconds());
}

void ThroughputMeter::Clear()
{
    totalSize = 0;
    begin = nn::os::GetSystemTick();
    end = nn::os::GetSystemTick();
}


/*---------------------------------------------------------------------------
　　　　　LastThroughputMeter
---------------------------------------------------------------------------*/

LastThroughputMeter::LastThroughputMeter(uint64_t size)
{
    SetSize(size);
}

LastThroughputMeter::~LastThroughputMeter()
{
    delete ticks;
    delete sizes;
}

void LastThroughputMeter::Count(uint64_t bytes)
{
    totalSize += bytes;
    if(sizes->GetSize() == size)
    {
        totalSize -= (*sizes)[0];
    }

    sizes->SetValue(bytes);
    //ticks->SetValue(nn::os::Tick::GetSystemCurrent());
    ticks->SetValue(nn::os::GetSystemTick());
}

double LastThroughputMeter::GetThroughput() const
{
    return static_cast<double>(totalSize) * 8.0f * 1000.0f / (1024.0f * 1024.0f) /
            static_cast<double>(((*ticks)[ticks->GetSize() - 1] - (*ticks)[0]).ToTimeSpan().GetMilliSeconds());
}

void LastThroughputMeter::Clear()
{
    ticks->Clear();
    sizes->Clear();
    totalSize = 0;
}

void LastThroughputMeter::SetSize(uint32_t size)
{
    this->size = size;
    if(ticks != NULL)
    {
        delete ticks;
    }
    ticks = new RingBuffer<nn::os::Tick>(size);
    if(sizes != NULL)
    {
        delete sizes;
    }
    sizes = new RingBuffer<uint64_t>(size);
    totalSize = 0;
}


/*---------------------------------------------------------------------------
　　　　　Average
---------------------------------------------------------------------------*/

Average::Average()
{
    Clear();
}

Average::~Average()
{
}

void Average::Count(double value)
{
    total += value;
    minValue = std::min(value, minValue);
    maxValue = std::max(value, maxValue);
    ++count;
}

double Average::GetValue() const
{
    if( count == 0 )
    {
        return 0;
    }

    return total / count;
}

double Average::GetMinValue() const
{
    if( count == 0 )
    {
        return 0;
    }

    return minValue;
}

double Average::GetMaxValue() const
{
    if( count == 0 )
    {
        return 0;
    }

    return maxValue;
}

void Average::Clear()
{
    total = 0;
    count = 0;
    minValue = MaxValue;
    maxValue = MinValue;
}



/*---------------------------------------------------------------------------
　　　　　LastAverage
---------------------------------------------------------------------------*/

LastAverage::LastAverage(int size)
{
    this->size = size;
}

void LastAverage::Count(double value)
{
    Average::Count(value);
    lastValues.push_back(value);

    while( count > size )
    {
        total -= lastValues.front();
        lastValues.pop_front();
        --count;
    }
}

double LastAverage::GetMinValue() const
{
    if( lastValues.empty() )
    {
        return 0;
    }

    double minimum = MaxValue;
    for(auto& v : lastValues)
    {
        minimum = std::min(v, minimum);
    }

    return minimum;
}

double LastAverage::GetMaxValue() const
{
    if( lastValues.empty() )
    {
        return 0;
    }

    double maximum = MinValue;
    for(auto& v : lastValues)
    {
        maximum = std::max(v, maximum);
    }

    return maximum;
}

void LastAverage::SetSize(uint32_t s)
{
    size = s;
    while( count > size )
    {
        total -= lastValues.front();
        lastValues.pop_front();
        --count;
    }
}


/*---------------------------------------------------------------------------
　　　　　LatencyCounter
---------------------------------------------------------------------------*/

const int32_t LatencyCounter::Lifetime_Infinity = 0x7fffffff;
uint32_t LatencyCounter::lifetime;

LatencyCounter::LatencyCounter()
{
    Clear();
}

LatencyCounter::~LatencyCounter()
{
}

void LatencyCounter::Count(int64_t time)
{
    avg.Count(time);
    max = std::max(time, max);
    min = std::min(time, min);

    if( time < 0 )
    {
        // 負の値は < 1ms にカウントする
        ++burst[0];
    }
    else if( time < LastIndex - 1 )
    {
        ++burst[time];
    }
    else
    {
        // burst[LastIndex - 1] には LastIndex - 1 以上の合計値を保存する
        ++burst[LastIndex - 1];
    }
}

int64_t LatencyCounter::GetMinLatency() const
{
    return min;
}

int64_t LatencyCounter::GetMaxLatency() const
{
    return max;
}

double LatencyCounter::GetAvgLatency() const
{
    return avg.GetValue();
}

int64_t LatencyCounter::GetBurstLatencyCount(uint8_t index) const
{
    if( index > LastIndex - 1 )
    {
        index = LastIndex - 1;
    }
    return burst[index];
}

int64_t LatencyCounter::GetBurstLatencyCount(uint8_t from, uint8_t to) const
{
    if( from > to )
    {
        std::swap(from, to);
    }

    if( from > LastIndex - 1 )
    {
        from = LastIndex - 1;
    }

    if( to > LastIndex - 1 )
    {
        to = LastIndex - 1;
    }

    auto sum = 0;
    for(int i=from; i<=to; ++i)
    {
        sum += burst[i];
    }

    return sum;
}

void LatencyCounter::Clear()
{
    min = LastIndex;
    max = 0;
    avg.Clear();
    std::memset(burst, 0, sizeof(burst));
}

uint8_t LatencyCounter::GetLastIndex() const
{
    return LastIndex;
}

int64_t LatencyCounter::GetAlivePackets() const
{
    int64_t sum = 0;
    for(int i=0; i<GetLastIndex() && i<lifetime; ++i)
    {
        sum += burst[i];
    }
    return sum;
}

int64_t LatencyCounter::GetDeadPackets() const
{
    int64_t sum = 0;
    for(int i=lifetime; i<GetLastIndex(); ++i)
    {
        sum += burst[i];
    }
    return sum;
}

uint32_t LatencyCounter::GetLifetime()
{
    return lifetime;
}

void LatencyCounter::SetLifetime(uint32_t time)
{
    lifetime = time;
}


/*---------------------------------------------------------------------------
　　　　　SamplingCountRecorder
---------------------------------------------------------------------------*/

SamplingCountRecorder::SamplingCountRecorder()
{
    Clear();
    recordInterval = 1000; // ms
    recordCount = 1000;
}

SamplingCountRecorder::~SamplingCountRecorder()
{
}

void SamplingCountRecorder::Record(const int64_t& count, bool* isRecorded)
{
    NN_ASSERT(isRecorded);
    nn::os::Tick tick = nn::os::GetSystemTick();
    *isRecorded = false;

    if( lastUpdateTime == nn::os::Tick(0) )
    {
        lastUpdateTime = tick;
    }
    else if( (tick - lastUpdateTime).ToTimeSpan().GetMilliSeconds() >= recordInterval )
    {
        // 規定時間を超えていればそれまでの更新回数を記録する
        if( updatedCount.size() > recordCount )
        {
            updatedCount.pop_front();
        }
        updatedCount.push_back(TimedCount(count, tick));
        max = std::max(max, count);
        min = std::min(min, count);
        lastUpdateTime = tick;
        *isRecorded = true;
    }
}

void SamplingCountRecorder::RecordBySequenceNumber(const int64_t& number)
{
    bool isRecorded = false;
    if( lastUpdateTime == nn::os::Tick(0) )
    {
        Record(0, &isRecorded);
        lastSequenceNumber = number;
    }
    else
    {
        Record(number - lastSequenceNumber, &isRecorded);
        if( isRecorded )
        {
            lastSequenceNumber = number;
        }
    }
}

void SamplingCountRecorder::RecordBySamplingCount(const int64_t& count)
{
    bool isRecorded = false;
    if( lastUpdateTime == nn::os::Tick(0) )
    {
        currentUpdatedCount += count;
        Record(0, &isRecorded);
    }
    else
    {
        currentUpdatedCount += count;
        Record(currentUpdatedCount, &isRecorded);
        if( isRecorded )
        {
            currentUpdatedCount = 0;
        }
    }
}

int64_t SamplingCountRecorder::GetMinCount() const
{
    return min;
}

int64_t SamplingCountRecorder::GetMaxCount() const
{
    return max;
}

const std::deque<SamplingCountRecorder::TimedCount>& SamplingCountRecorder::GetUpdatedCounts() const
{
    return updatedCount;
}

void SamplingCountRecorder::SetRecordInterval(const uint16_t& interval)
{
    recordInterval = interval;
}
void SamplingCountRecorder::SetRecordCount(const uint16_t& count)
{
    recordCount = count;
}

uint16_t SamplingCountRecorder::GetRecordInterval() const
{
    return recordInterval;
}

uint16_t SamplingCountRecorder::GetRecordCount() const
{
    return recordCount;
}

void SamplingCountRecorder::Clear()
{
    updatedCount.clear();
    currentUpdatedCount = 0;
    min = 0xffffffff;
    max = 0;
    lastSequenceNumber = 0;
    lastUpdateTime -= lastUpdateTime;
}

}
