﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/btm/btm_Api.h>
#include <nn/btm/system/btm_SystemApi.h>

#include "ConfigurationMenu.h"
#include "Program.h"

namespace{

    nn::btm::BdAddress DummyAddress = {{0}};

}

namespace WlanTest {

/*---------------------------------------------------------------------------
  BtFlightModeItem
---------------------------------------------------------------------------*/

BtFlightModeItem::BtFlightModeItem()
{
    GenerateItems();
}

BtFlightModeItem::~BtFlightModeItem()
{
}

string BtFlightModeItem::GetTitleName()
{
    return "Bt Flight Mode";
}

int BtFlightModeItem::Execute()
{
    bool isEnabled = GetValue();
    Application::SetBtFlightMode(isEnabled);
    NN_LOG(" ### SetBtFlightMode(%s) : No Result\n", (isEnabled ? "true" : "false"));
    return 0;
}

int BtFlightModeItem::Cancel()
{
    return 0;
}

int BtFlightModeItem::Reset()
{
    Application::SetBtFlightMode(false);
    NN_LOG(" ### SetBtFlightMode(false) : No Result\n");
    return 0;
}

void BtFlightModeItem::GenerateItems()
{
    bool isEnabled = Application::IsBtFlightModeEnabled();
    Add(NamedValue<bool>("On", true), true == isEnabled);
    Add(NamedValue<bool>("Off", false), false == isEnabled);
}

/*---------------------------------------------------------------------------
  LlrNotifyItem
---------------------------------------------------------------------------*/

LlrNotifyItem::LlrNotifyItem()
{
}

LlrNotifyItem::~LlrNotifyItem()
{
}

string LlrNotifyItem::GetTitleName()
{
    return "LLR Notify";
}

void LlrNotifyItem::Update()
{
    // ペアリング済みのデバイス情報を取得
    nn::btm::DeviceInfoList infoList;
    nn::btm::GetDeviceInfo(&infoList);

    // 接続済みのデバイス情報を取得
    nn::btm::DeviceConditionList conditionList;
    nn::btm::GetConnectedDeviceCondition(&conditionList);

    Clear();

    bool isAdded = false;
    for(int i=0; i<infoList.deviceCount; ++i)
    {
        bool isConnected = false;
        const auto& pairingAddress = infoList.device[i].bdAddress;
        for(int j=0; j<conditionList.deviceCount; ++j)
        {
            const auto& connectedAddress = conditionList.device[j].bdAddress;
            if( pairingAddress == connectedAddress )
            {
                isConnected = true;
                break;
            }
        }

        // ペアリング済み && 非接続状態であれば選択肢に表示する
        if( !isConnected )
        {
            isAdded = true;
            Add(NamedValue<nn::btm::BdAddress>(ToString(pairingAddress), pairingAddress));
        }
    }

    if( !isAdded )
    {
        Add(NamedValue<nn::btm::BdAddress>("Not Available", DummyAddress));
    }
}

int LlrNotifyItem::Execute()
{
    int rtn = 0;

    if( Size() <= 1 && GetValue() == DummyAddress )
    {
        rtn = -1;
    }
    else
    {
        auto address = GetValue();
        nn::Result result = nn::btm::LlrNotify(&address);
        rtn = result.GetDescription();
        NN_LOG(" ### LlrNotify(%s) : %d\n", ToString(address).c_str(), rtn);
    }

    return rtn;
}

int LlrNotifyItem::Cancel()
{
    return 0;
}

int LlrNotifyItem::Reset()
{
    return 0;
}

void LlrNotifyItem::GenerateItems()
{
}

/*---------------------------------------------------------------------------
  ContinuousLlrNotifyItem
---------------------------------------------------------------------------*/

ContinuousLlrNotifyItem::ContinuousLlrNotifyItem() :
    m_IsRunning(false),
    m_RequestStop(true)
{
}

ContinuousLlrNotifyItem::~ContinuousLlrNotifyItem()
{
    Cancel();
}

string ContinuousLlrNotifyItem::GetTitleName()
{
    return "Continuous LLR Notify";
}

int ContinuousLlrNotifyItem::Execute()
{
    int rtn = 0;

    if( !m_IsRunning )
    {
        m_IsRunning = true;
        m_RequestStop = false;
        ContinuousLlrNotify();
        m_IsRunning = false;
    }

    return rtn;
}

int ContinuousLlrNotifyItem::Cancel()
{
    if( m_IsRunning )
    {
        m_RequestStop = true;
    }
    return 0;
}

int ContinuousLlrNotifyItem::Reset()
{
    Cancel();
    return 0;
}

void ContinuousLlrNotifyItem::GenerateItems()
{
}

void ContinuousLlrNotifyItem::ContinuousLlrNotify()
{
    auto address = GetValue();
    auto last = nn::os::Tick(0);
    while( !m_RequestStop )
    {
        auto cur = nn::os::GetSystemTick();

        // 一回の LLR Scan は約 4 秒
        if( (cur - last).ToTimeSpan().GetMilliSeconds() > 3850 )
        {
            auto result = nn::btm::LlrNotify(&address);
            auto description = result.GetDescription();
            if( description == 0 )
            {
                last = cur;
            }
            NN_LOG(" ### LlrNotify(%s) : %d\n", ToString(address).c_str(), description);
        }
        Sleep(nn::TimeSpan::FromMilliSeconds(100));
    }
}

/*---------------------------------------------------------------------------
  StartGamepadPairingItem
---------------------------------------------------------------------------*/

StartGamepadPairingItem::StartGamepadPairingItem()
{
    GenerateItems();
}

StartGamepadPairingItem::~StartGamepadPairingItem()
{
}

string StartGamepadPairingItem::GetTitleName()
{
    return "Start Gamepad Pairing";
}

int StartGamepadPairingItem::Execute()
{
    nn::Result result = nn::btm::system::StartGamepadPairing();
    int rtn = result.GetDescription();
    NN_LOG(" ### StartGamepadPairing() : %d\n", rtn);
    return rtn;
}

int StartGamepadPairingItem::Cancel()
{
    return 0;
}

int StartGamepadPairingItem::Reset()
{
    nn::btm::system::CancelGamepadPairing();
    NN_LOG(" ### CancelGamepadPairing() : No Result\n");
    return 0;
}

void StartGamepadPairingItem::GenerateItems()
{
    Add("No Args");
}

/*---------------------------------------------------------------------------
  CancelGamepadPairingItem
---------------------------------------------------------------------------*/

CancelGamepadPairingItem::CancelGamepadPairingItem()
{
    GenerateItems();
}

CancelGamepadPairingItem::~CancelGamepadPairingItem()
{
}

string CancelGamepadPairingItem::GetTitleName()
{
    return "Cancel Gamepad Pairing";
}

int CancelGamepadPairingItem::Execute()
{
    nn::btm::system::CancelGamepadPairing();
    NN_LOG(" ### CancelGamepadPairing() : No Result\n");
    return 0;
}

int CancelGamepadPairingItem::Cancel()
{
    return 0;
}

int CancelGamepadPairingItem::Reset()
{
    return 0;
}

void CancelGamepadPairingItem::GenerateItems()
{
    Add("No Args");
}

/*---------------------------------------------------------------------------
  EnableSlotSavingItem
---------------------------------------------------------------------------*/

EnableSlotSavingItem::EnableSlotSavingItem()
{
    GenerateItems();
}

EnableSlotSavingItem::~EnableSlotSavingItem()
{
}

string EnableSlotSavingItem::GetTitleName()
{
    return "Enable Slot Saving";
}

int EnableSlotSavingItem::Execute()
{
    bool isEnabled = GetValue();
    nn::btm::EnableSlotSaving(isEnabled);
    NN_LOG(" ### EnableSlotSaving(%s) : No Result\n", (isEnabled ? "true" : "false"));
    return 0;
}

int EnableSlotSavingItem::Cancel()
{
    return 0;
}

int EnableSlotSavingItem::Reset()
{
    nn::btm::EnableSlotSaving(false);
    NN_LOG(" ### EnableSlotSaving(false) : No Result\n");
    return 0;
}

void EnableSlotSavingItem::GenerateItems()
{
    Add(NamedValue<bool>("On", true), true);
    Add(NamedValue<bool>("Off", false));
}

/*---------------------------------------------------------------------------
  BtResetItem
---------------------------------------------------------------------------*/

BtResetItem::BtResetItem()
{
    GenerateItems();
}

BtResetItem::~BtResetItem()
{
}

string BtResetItem::GetTitleName()
{
    return "Bt Reset";
}

int BtResetItem::Execute()
{
    NN_LOG(" ### Bluetooth Reset\n");

    Application::SetBtFlightMode(false);
    NN_LOG(" ### SetBtFlightMode(false) : No Result\n");

    nn::btm::system::CancelGamepadPairing();
    NN_LOG(" ### CancelGamepadPairing() : No Result\n");

    nn::btm::EnableSlotSaving(false);
    NN_LOG(" ### EnableSlotSaving(false) : No Result\n");

    return 0;
}

int BtResetItem::Cancel()
{
    return 0;
}

int BtResetItem::Reset()
{
    return 0;
}

void BtResetItem::GenerateItems()
{
    Add("No Args");
}

/*---------------------------------------------------------------------------
  ConfigurationPage
---------------------------------------------------------------------------*/

ConfigurationPage::ConfigurationPage()
    : m_CommandResult(0),
      m_CommandState(CommandState_Ready)
{
    GenerateComponents();
}

ConfigurationPage::~ConfigurationPage()
{
    if( m_CommandState != CommandState_Ready )
    {
        Stop();
        nn::os::DestroyThread(&m_ExecuteThread);
        m_CommandState = CommandState_Ready;
    }
}

void ConfigurationPage::UpdateCommandResult()
{
    Color color;
    string text;
    switch( m_CommandState )
    {
    case CommandState_Ready :
        {
            text = m_ResultLabel.Text;
            color = m_ResultLabel.TextColor;
        }
        break;

    case CommandState_Executing :
        {
            text = "";
            color = ToColor(WHITE);
        }
        break;

    case CommandState_Executed :
        {
            if( m_CommandResult == 0 )
            {
                text = "OK";
                color = ToColor(GREEN);
            }
            else
            {
                ostringstream oss;
                oss << "NG(" << static_cast<int>(m_CommandResult) << ")";
                text = oss.str();
                color = ToColor(RED);
            }
            nn::os::DestroyThread(&m_ExecuteThread);
            m_CommandState = CommandState_Ready;
            m_ActiveIndicator.Inactivate();
            m_ResultLabel.X = 80 - Display::GetInstance().GetFixedWidth() * 0.5f * text.size();
        }
        break;

    default :
        {
            text = "";
            color = ToColor(WHITE);
        }
        break;
    }

    m_ResultLabel.Text = text;
    m_ResultLabel.TextColor = color;
}

void ConfigurationPage::ShowImpl(Display& display)
{
    UpdateCommandResult();

    int index = -1;
    if( m_CommandState == CommandState_Executing )
    {
        index = m_Selector.GetCursorIndex();
    }

    InvokeInput();

    // コマンド実行中に他のアイテムを選択できないようにする
    if( index != -1 )
    {
        m_Selector.SetCursorIndex(index);
    }

    InvokeShow(display);
}

void ConfigurationPage::UpdateItems()
{
    // アイテムに更新する必要があるものをまとめて更新する
    m_LlrNotifyItem.Update();
    m_ContinuousLlrNotifyItem.Update();
}

void ConfigurationPage::InputPad(Pad& pad)
{
    if( pad.IsTrigger(Button::A) )
    {
        switch( m_CommandState )
        {
        case CommandState_Ready:
            {
                Start();
                m_ActiveIndicator.Activate();
            }
            break;

        case CommandState_Executing:
            {
                Stop();
            }
            break;

        default:
            {
            }
            break;
        }
    }
}

void ConfigurationPage::GenerateComponents()
{
    uint32_t h = Display::GetInstance().GetLineHeight();
    uint32_t w = Display::GetInstance().GetFixedWidth();
    uint32_t x = DISPLAY_CONTENT_START_X;
    uint32_t y = 0; //DISPLAY_CONTENT_START_Y;

    m_Selector.SetSideIndicator(false);

    Add(m_Selector);
    m_Selector.X = 150;
    m_Selector.Y = y;
    m_Selector.Width = 3000;
    m_Selector.Height = h * 2;//DISPLAY_CONTENT_HEIGHT;

    m_Selector.Register(m_BtFlightModeItem.GetTitleName(), &m_BtFlightModeItem);
    m_Selector.Register(m_EnableSlotSavingItem.GetTitleName(), &m_EnableSlotSavingItem);
    m_Selector.Register(m_StartGamepadPairingItem.GetTitleName(), &m_StartGamepadPairingItem);
    m_Selector.Register(m_CancelGamepadPairingItem.GetTitleName(), &m_CancelGamepadPairingItem);
    m_Selector.Register(m_LlrNotifyItem.GetTitleName(), &m_LlrNotifyItem);
    m_Selector.Register(m_ContinuousLlrNotifyItem.GetTitleName(), &m_ContinuousLlrNotifyItem);
    m_Selector.Register(m_BtResetItem.GetTitleName(), &m_BtResetItem);

    Add(m_ResultLabel);
    m_ResultLabel.X = 0;
    m_ResultLabel.Y = 0;
    m_ResultLabel.Text = "Command";

    Add(m_ActiveIndicator);
    m_ActiveIndicator.X = 68;
    m_ActiveIndicator.Y = 0;
    m_ActiveIndicator.Inactivate();
}

void ConfigurationPage::ExecuteCommand()
{
    nn::os::Tick tick = nn::os::GetSystemTick();
    if( m_Selector.GetSelectingItem() == &m_BtResetItem )
    {
        for(int i=0; i<m_Selector.GetSize(); ++i)
        {
            m_Selector.Reset(i);
        }
    }
    else
    {
        m_CommandResult = m_Selector.Execute();
    }
    uint64_t executedTimeMs = (nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds();

    // アニメーションのため少し待機する
    if( executedTimeMs < 500 )
    {
        Sleep(nn::TimeSpan::FromMilliSeconds(500 - executedTimeMs));
    }

    m_CommandState = CommandState_Executed;
}

void ConfigurationPage::ExecuteThreadFunction(void* configurationPage)
{
    reinterpret_cast<ConfigurationPage*>(configurationPage)->ExecuteCommand();
}

void ConfigurationPage::Start(int32_t priority)
{
    static NN_ALIGNAS(4096) char s_ExecuteThreadStack[4096];

    if( m_CommandState != CommandState_Ready )
    {
        NN_ASSERT(false);
        return;
    }

    m_CommandState = CommandState_Executing;
    nn::Result result = nn::os::CreateThread(&m_ExecuteThread, ExecuteThreadFunction, this, s_ExecuteThreadStack, sizeof(s_ExecuteThreadStack), priority);
    if( result.IsFailure() )
    {
        NN_LOG("  - failed : Cannot create ExecuteThread.\n");
        NN_ASSERT(false);
        return;
    }

    nn::os::SetThreadNamePointer(&m_ExecuteThread, "WitExecuteThread");
    nn::os::StartThread(&m_ExecuteThread);
}

void ConfigurationPage::Stop()
{
    if( m_CommandState == CommandState_Executing )
    {
        m_Selector.Cancel();
    }
}

}    // WlanTest
