﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Client.h"

namespace WlanTest {

const char* ClientParam::ITEM_STR_CHANNEL               = "CHANNEL";
const char* ClientParam::ITEM_STR_SSID                  = "SSID";
const char* ClientParam::ITEM_STR_SECURITY_MODE         = "SECURITY_MODE";
const char* ClientParam::ITEM_STR_SECURITY_KEY          = "SECURITY_KEY";
const char* ClientParam::ITEM_STR_BEACON_LOST_TIMEOUT   = "BEACON_LOST_TIMEOUT";

/*---------------------------------------------------------------------------
　　　　　Client
---------------------------------------------------------------------------*/
Client::Client() : m_Cs(false, 0)
{
    m_Ssid = nn::wlan::Ssid("sap_master");
    m_Bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    m_Channel = 1;
    m_Security.privacyMode = nn::wlan::SecurityMode_Open;
    m_Security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
    std::memset(m_Security.key, 0x00, sizeof(m_Security.key));
    m_Indication = nn::wlan::BeaconIndication_Enable;
    m_EnableDirectBroadcast = false;
    m_EnableClr = false;
    m_TxWait = 5;
    m_IsWaiting = false;
    m_BeaconLostTimeout = 10;
    m_ConnectionTime = 0;

    // SapTxQueueParam param;
    // param.eCWmin = 4;
    // param.eCWmax = 10;
    // param.aifsn = 2;
    // param.queueType = TX_QUEUE_1; // toriaezu
    // m_TxQueueParam[0] = param;
    // m_TxQueueParam[1] = param;
    // m_TxQueueParam[0].queueType = TX_QUEUE_1;
    // m_TxQueueParam[1].queueType = TX_QUEUE_2;

    //m_InformationElementReceiver = new InformationElementReceiver();
    m_InformationElementReceiveHandler = new ReceiveEventHandler(Node)(*this, &Node::OnReceive);
}

Client::~Client()
{
    delete m_InformationElementReceiveHandler;
}

nn::Result Client::Initialize()
{
    nn::Result result = WlanInitialize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanInitialize()\n");
        return result;
    }

    // 前回の接続処理時間の初期化
    m_ConnectionTime = 0;
    m_Channel = 0;

    ModeChanger::ToLocalClientMode();

    result = nn::wlan::Socket::EnableTsfTimerFunction();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : EnableTsfTimerFunction()\n");
        return result;
    }

    // 送受信する際に Rx エントリが必須なので初期化時に作成する
    CreateRxEntry();

    m_IsInitialized = true;

    return result;
}

nn::Result Client::Finalize()
{
    nn::Result result = DeleteRxEntry();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : DeleteRxEntry()\n");
    }

    result = nn::wlan::Socket::DisableTsfTimerFunction();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : DisableTsfTimerFunction()\n");
    }

    result = WlanFinalize();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : LocalNode::WlanFinalize()\n");
    }

    ModeChanger::ToReadyMode();
    m_IsInitialized = false;

    return result;
}

nn::Result Client::Open()
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    char buf[nn::wlan::Ssid::SsidHexStringLengthMax] = {0};
    NN_LOG(" === Connect Params ===============================\n");
    NN_LOG("   Ssid              : %s\n", m_Ssid.GetHexString(buf));
    NN_LOG("   Bssid             : %s\n", m_Bssid.GetString(buf));
    NN_LOG("   Channel           : %d\n", m_Channel);
    NN_LOG("   Security mode     : %d\n", m_Security.groupPrivacyMode);
    NN_LOG("   Security key      : %s\n", reinterpret_cast<char*>(m_Security.key));
    NN_LOG("   Auto keep alive   : %d\n", true);
    NN_LOG("   Indication        : %d\n", m_Indication);
    NN_LOG("   Bmiss timeout     : %d\n", m_BeaconLostTimeout);
    NN_LOG(" ===============================================\n");

    nn::Result result;
    nn::os::Tick startTick = nn::os::GetSystemTick();
    result = nn::wlan::Local::Connect(m_Ssid, m_Bssid, m_Channel, m_Security, true, m_Indication, m_BeaconLostTimeout);
    nn::os::Tick endTick = nn::os::GetSystemTick();

    if( result.IsSuccess() )
    {
        // connect() の時間を計測
        nn::wlan::ConnectionStatus connectionStatus;
        nn::Result conResult = nn::wlan::Local::GetConnectionStatus(&connectionStatus);
        if( conResult.IsSuccess() && connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            m_ConnectionTime = (endTick - startTick).ToTimeSpan().GetMilliSeconds();

            // Auto の場合もあるので改めてチャネルを取得する
            m_Channel = connectionStatus.channel;
        }
    }
    else
    {
        NN_LOG(" - failed : Connect\n");
        WlanTest::TPrintResult(result);
        return result;
    }

    return result;
}

nn::Result Client::Close()
{
    DBG_VLOG("");
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    nn::Result result;

    DBG_VLOG("");
    result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, NULL);
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : Disconnect\n");
        WlanTest::TPrintResult(result);
        return result;
    }
    DBG_VLOG("");

    return result;
}

void Client::MaintainConnection()
{
    if( !m_IsInitialized )
    {
        return;
    }

    nn::os::SystemEventType connectionEvent;
    nn::Result result;
    nn::wlan::ConnectionState lastState = nn::wlan::ConnectionState_Disconnected;
    m_ConnectionStatus.state = nn::wlan::ConnectionState_Disconnected;
    m_ConnectionStatus.bssid = nn::wlan::MacAddress::CreateZeroMacAddress();

    result = nn::wlan::Local::GetConnectionEvent(&connectionEvent);
    if( result.IsFailure() )
    {
        NN_LOG("GetConnectionEvent failed\n");
        WlanTest::TPrintResult(result);
        goto End;
    }

    m_RequestStop = false;
    m_IsStop = false;

    while( !m_RequestStop )
    {
        // 接続しているか調べる
        result = nn::wlan::Local::GetConnectionStatus(&m_ConnectionStatus);

        m_Cs.Lock();

        // 接続していなければ接続する
        if( m_ConnectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            Open();

            // 接続できた場合は Connect() を抜けた時点で接続が完了していることが保障されている
            result = nn::wlan::Local::GetConnectionStatus(&m_ConnectionStatus);
            if( result.IsSuccess() && m_ConnectionStatus.state != nn::wlan::ConnectionState_Connected )
            {
                m_Statistics.ConnectErrorCount++;
            }
        }

        // 接続しているか調べる
        result = nn::wlan::Local::GetConnectionStatus(&m_ConnectionStatus);

        if( result.IsFailure() )
        {
            NN_LOG("GetConnectionStatus failed\n");
            WlanTest::TPrintResult(result);
        }

        if( m_ConnectionStatus.state != lastState )
        {
            if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                NN_LOG("   Connected!!\n");
                m_Statistics.ConnectCount++;
            }

            if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Idle &&
               lastState == nn::wlan::ConnectionState_Connected )
            {
                NN_LOG("   Disconnected..\n");
                m_Statistics.Disconnected++;
            }
        }
        lastState = m_ConnectionStatus.state;

        m_Cs.Unlock();

        TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(100));
    }
End:

    //##?
    // 保持しているコネクションステータスを切断にする
    // ここで取得しなければ、切断後も接続中のものが残ってしまう
    result = nn::wlan::Local::GetConnectionStatus(&m_ConnectionStatus);
    if( result.IsFailure() )
    {
        NN_LOG("GetConnectionStatus failed\n");
        WlanTest::TPrintResult(result);
    }

    m_IsStop = true;

    nn::os::DestroySystemEvent(&connectionEvent);
}

bool Client::IsConnected()
{
    return m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected;
}

}
