﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <map>
#include "Delegate.h"
#include "Value.h"
#include "Util.h"
#include "WlanCommon.h"

namespace WlanTest {

#if 0

class CompleteMeasureEventArgs : public EventArgs
{
public:
    MeasureChannelResult Result;
};

/*!--------------------------------------------------------------------------*
  @brief        イベントハンドラインタフェース
-----------------------------------------------------------------------------*/
typedef IDelegate<CompleteMeasureEventArgs> ICompleteMeasureEventHandler;
#define CompleteMeasureEventHandler(a) Delegate<a,CompleteMeasureEventArgs>


/*---------------------------------------------------------------------------
           NoiseMeasurer
---------------------------------------------------------------------------*/
class NoiseMeasurer
{
protected:
    MeasureChannelParam m_Param;
    MeasureChannelResult m_Result;
    ICompleteMeasureEventHandler* m_CompleteMeasureEventHandler;

    NoiseMeasurer();
    virtual ~NoiseMeasurer();

public:
    virtual nn::Result Measure();
    virtual nn::Result WlanMeasure() = 0;
    virtual void Clear();

    virtual void SetCcaMode(uint16_t ccaMode)      { m_Param.ccaMode = ccaMode; };
    virtual void SetEdThreshold(uint16_t th)       { m_Param.edThreshold = th; };
    virtual void SetMeasureTime(uint16_t time)     { m_Param.measureTime = time; };
    virtual void SetChannelList(uint16_t list)     { m_Param.channelList = list; };
    virtual void AddChannel(uint16_t ch)           { m_Param.channelList |= 0x1 << (ch - 1); };
    virtual void RemoveChannel(uint16_t ch)        { m_Param.channelList &= ~(0x1 << (ch - 1)); };
    virtual void GetBusyRatio(uint8_t ratio[14])
    {
        std::memcpy(ratio, m_Result.channelBusyRatio, 14);
    }
    virtual uint8_t GetBusyRatio(uint8_t ch)
    {
        return m_Result.channelBusyRatio[ch - 1];
    }

    virtual void SetCompleteMeasureEventHandler(ICompleteMeasureEventHandler& handler)
    {
        m_CompleteMeasureEventHandler = &handler;
    }

    virtual uint8_t GetMinBusyRatioChannel()
    {
        uint8_t ratio = 100;
        uint8_t minCh = 1;
        uint8_t i = 1;
        uint16_t list = m_Param.channelList;
        while(list != 0)
        {
            if(list & 0x1 == 1)
            {
                if(ratio > m_Result.channelBusyRatio[i - 1])
                {
                    ratio = m_Result.channelBusyRatio[i - 1];
                    minCh = i;
                }
            }
            list >>= 1;
            i++;
        }

        return minCh;
    }
};


/*---------------------------------------------------------------------------
           LocalMoiseMeasurer
---------------------------------------------------------------------------*/
class LocalNoiseMeasurer : public NoiseMeasurer
{
public:

    virtual nn::Result WlanMeasure();
};

#endif

/*---------------------------------------------------------------------------
           SecurityConfig
---------------------------------------------------------------------------*/
struct SecurityConfig
{
    bool isWpaSupported;
    bool isWpa2Supported;
    bool isWpsSupported;

    SecurityConfig()
    {
        isWpaSupported = false;
        isWpa2Supported = false;
        isWpsSupported = false;
    }

    SecurityConfig(bool wpa, bool wpa2, bool wps)
    {
        isWpaSupported = wpa;
        isWpa2Supported = wpa2;
        isWpsSupported = wps;
    }
};

/*---------------------------------------------------------------------------
           ScanStatistics
---------------------------------------------------------------------------*/
struct SearchStatistics
{
    // 更新間隔
    static nn::TimeSpan        updateInterval;
    static nn::os::Tick        lastUpdateTime;

    nn::os::Tick               firstSendTime;
    nn::os::Tick               lastSendTime;

    // アクションフレーム送信に関する統計情報
    uint32_t                   sendSize;
    uint64_t                   sendCount;
    uint64_t                   sendErrorCount;
    Average                    sendExecutionTime;

    // スキャンに関する統計情報
    uint32_t                   scanCount;
    uint32_t                   scanErrorCount;
    Average                    scanExecutionTime;

    // 一定間隔の統計情報
    uint32_t                   lastSendSize;
    uint64_t                   lastSendCount;
    uint64_t                   lastSendErrorCount;
    uint32_t                   lastScanCount;
    uint32_t                   lastScanErrorCount;

    // 一定間隔の統計情報の処理するための一時変数
    uint32_t                   workSendSize;
    uint64_t                   workSendCount;
    uint64_t                   workSendErrorCount;
    uint32_t                   workScanCount;
    uint32_t                   workScanErrorCount;

    void Clear()
    {
        firstSendTime = nn::os::Tick(0);
        lastSendTime = nn::os::Tick(0);

        sendSize = 0;
        sendCount = 0;
        sendErrorCount = 0;
        sendExecutionTime.Clear();

        scanCount = 0;
        scanErrorCount = 0;
        scanExecutionTime.Clear();

        lastSendSize = 0;
        lastSendCount = 0;
        lastSendErrorCount = 0;
        lastScanCount = 0;
        lastScanErrorCount = 0;

        workSendSize = 0;
        workSendCount = 0;
        workSendErrorCount = 0;
        workScanCount = 0;
        workScanErrorCount = 0;
    }
};


/*---------------------------------------------------------------------------
           AccessPoint
---------------------------------------------------------------------------*/
class AccessPoint
{
private:

    nn::wlan::MacAddress  m_Bssid;
    nn::wlan::Ssid        m_Ssid;
    int16_t               m_Channel;
    int32_t               m_Rssi;
    nn::wlan::LinkLevel   m_LinkLevel;
    SecurityConfig        m_SecurityConfig;
    uint16_t              m_BeaconInterval;
    nn::Bit16             m_Capability;

public:

    AccessPoint();
    AccessPoint(nn::wlan::MacAddress bssid, nn::wlan::Ssid ssid, int16_t ch, int32_t rssi,
                nn::wlan::LinkLevel linkLevel, SecurityConfig security, uint16_t interval, nn::Bit16 capability);

    nn::wlan::MacAddress GetBssid() { return m_Bssid; }
    nn::wlan::Ssid GetSsid() { return m_Ssid; }
    int16_t GetChannel() { return m_Channel; }
    SecurityConfig GetSecurityConfig() { return m_SecurityConfig; }
    int32_t GetRssi() { return m_Rssi; }
    nn::wlan::LinkLevel GetLinkLevel() { return m_LinkLevel; }
    uint16_t GetBeaconInterval() { return m_BeaconInterval; }
    nn::Bit16 GetCapability() { return m_Capability; }
};

/*---------------------------------------------------------------------------
           AccessPointList
---------------------------------------------------------------------------*/
typedef multimap<int16_t, AccessPoint*> AccessPointList;

/*---------------------------------------------------------------------------
           Scanner
---------------------------------------------------------------------------*/
class Scanner
{
protected:

    nn::wlan::ScanParameters          m_Param;
    uint32_t                          m_BufferSize;
    uint8_t*                          m_pBuffer;
    nn::wlan::BeaconScanResultReader* m_pReader;

private:

    static const uint8_t SSID_MAX = 0xff;
    map<uint64_t, AccessPoint*>       m_ApList;
    AccessPointList                   m_ApListSortedByRssi;
    AccessPointList::reverse_iterator m_Iterator;

    nn::wlan::Ssid                    m_SsidList[SSID_MAX];

    virtual nn::Result WlanStartScan() = 0;
    virtual nn::Result WlanStopScan() = 0;

public :

    virtual void StartScan();
    void StopScan();
    AccessPoint* GetFirstAccessPoint();
    AccessPoint* GetNextAccessPoint();
    void Clear();
    virtual int16_t SearchChannel();

    Scanner();
    virtual ~Scanner();

    virtual void SetBufferSize(const uint32_t size);
    virtual void SetMaxBuffer();
    virtual void SetEnableActiveScan(const bool active);
    virtual void SetScanType(const nn::wlan::ScanType scanType);
    virtual void SetChannel(const int16_t chList[nn::wlan::WirelessChannelsCountMax], const uint8_t count);
    virtual void SetScanTime(const int32_t time);
    virtual void SetHomeChannelTime(const int32_t time);
    virtual void SetSsid(const char* ssid);
    virtual void SetSsid(const string& ssid);
    virtual void SetSsid(const nn::wlan::Ssid& ssid);
    virtual void SetSsidList(const nn::wlan::Ssid* ssidList, const uint8_t count);
    virtual void SetBssid(const nn::wlan::MacAddress bssid);

    virtual uint32_t GetApCount();
    virtual nn::wlan::BeaconScanResultReader& GetScanResult();
    virtual nn::wlan::BeaconDescriptionReader GetFirstDescription();
    virtual nn::wlan::BeaconDescriptionReader GetNextDescription();

    nn::wlan::ScanParameters GetScanParam();

    void PrintResult();

private:

    Scanner(const Scanner& scanner);
    Scanner& operator=(const Scanner& scanner);

};


/*---------------------------------------------------------------------------
           InfraScanner
---------------------------------------------------------------------------*/
class InfraScanner : public Scanner
{
private:

    virtual nn::Result WlanStartScan();
    virtual nn::Result WlanStopScan();

};


/*---------------------------------------------------------------------------
           LocalScanner
---------------------------------------------------------------------------*/
class LocalScanner : public Scanner
{
protected:

    uint32_t m_BeaconIeIdx;
    bool     m_AddIe;
    Oui      m_Oui;
    nn::wlan::ScanIeMatchInfo m_MatchInfo;

public:

    LocalScanner();
    virtual nn::Result WlanStartScan();
    virtual nn::Result WlanStopScan();
    virtual void EnableAddIe(const bool b);
    virtual nn::Result SetOui(const Oui oui);
    virtual nn::Result AddIe(const nn::Bit8 body[], const size_t length);
    virtual nn::Result RemoveIe();

};

/*---------------------------------------------------------------------------
           AfTxScanner
---------------------------------------------------------------------------*/
class AfTxScanner : public Scanner
{
protected:

    bool m_IsStopped;
    bool m_IsAfEnabled;
    bool m_IsScanEnabled;
    ActionFrameParams m_AfParam;
    uint64_t m_ScanWait;
    SearchStatistics m_Stats;
    map<int16_t, uint64_t> m_SequenceNumber;

public:

    AfTxScanner();
    virtual void StartScan();
    virtual nn::Result WlanStartScan();
    virtual nn::Result WlanStopScan();

    void SetAfParam(const ActionFrameParams &afParam);
    ActionFrameParams GetAfParam();

    void SetIsAfEnabled(const bool isEnabled);
    bool IsAfEnabled();

    void SetIsScanEnabled(const bool isEnabled);
    bool IsScanEnabled();

    SearchStatistics GetStatistics();

    void ClearStatistics();

protected:

    virtual void TransmitActionFrame();
    void GeneratePacket(uint8_t* buf, const size_t bufSize, const uint64_t sequenceNumber, const int16_t channel, const uint32_t length);

};


}
