﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>

#include "SceneManager.h"

const int BufferCount = 2;

//------------------------------------------------------------------------------
//  コマンド生成
//------------------------------------------------------------------------------
struct MakeCommandUserData
{
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* pWriter;
    SceneManager* pManager;
    int frame;
};

void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    MakeCommandUserData* pData = reinterpret_cast<MakeCommandUserData*>(pUserData);
    Task::GraphicTools tools = {
            pGraphicsFramework,
            bufferIndex,
            pData->pPrimitiveRenderer,
            pData->pWriter
    };

    pData->pManager->Draw(&tools);
}

void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    MakeCommandUserData* pData = reinterpret_cast<MakeCommandUserData*>(pUserData);
    pData->pManager->Update();
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // デバッグ設定書き込み
    const bool isEnabled = true;
    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
    nn::nifm::Initialize();
    nn::nifm::SetWirelessCommunicationEnabledForTest(false);
    // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておく
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // TIME ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );

    // フレームワーク初期化
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetBufferCount(BufferCount);
    fwInfo.SetSwapChainBufferCount(BufferCount);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, (BufferCount * 8 + 8) * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, (BufferCount * 8 + 8) * 1024 * 1024);
    fwInfo.SetRootCommandBufferCommandMemorySize(8 * 1024 * 1024);
    nns::gfx::GraphicsFramework gfw;
    gfw.Initialize(fwInfo);

    // デバッグフォント初期化
    nns::gfx::GraphicsFramework::DebugFontTextWriter writer;
    gfw.InitializeDebugFontTextWriter(&writer, 1024 * 8, BufferCount);

    // プリミティブレンダラー初期化
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer;
    {
        nns::gfx::PrimitiveRenderer::RendererInfo rendererInfo;
        rendererInfo.SetDefault();
        rendererInfo.SetAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nullptr);
        rendererInfo.SetAdditionalBufferSize(1024 * 4);
        rendererInfo.SetDrawCallCountMax(1024 * 4);
        rendererInfo.SetViewFunctionCallCountMax(1024 * 4);

        rendererInfo.SetMultiBufferQuantity(BufferCount);

        pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(gfw.GetDevice(), rendererInfo);
        pPrimitiveRenderer->SetScreenWidth(gfw.GetDisplayWidth());
        pPrimitiveRenderer->SetScreenHeight(gfw.GetDisplayHeight());
    }

    // GraphicsFramework によるフレーム処理用の設定
    MakeCommandUserData userData;
    userData.pPrimitiveRenderer = pPrimitiveRenderer;
    userData.pWriter = &writer.object;
    userData.frame = 0;
    SceneManager sceneManager;
    userData.pManager = &sceneManager;

    gfw.SetCalculateCallback(CalculateCallback, &userData);
    gfw.SetMakeCommandCallback(MakeCommandCallback, &userData);
    gfw.SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_DeferredExecution);
    gfw.SetPresentInterval(1);

    sceneManager.Initialize();
    // 毎フレームのレンダリング
    while( sceneManager.IsExit() != true )
    {
        gfw.ProcessFrame();
    }
    sceneManager.Finalize();

    gfw.QueueFinish();
    // プリミティブレンダラー終了
    nns::gfx::PrimitiveRenderer::DestroyRenderer(pPrimitiveRenderer, gfw.GetDevice(), nns::gfx::GraphicsFramework::DefaultFreeFunction, nullptr);
    // デバッグフォント終了
    gfw.FinalizeDebugFontTextWriter(&writer);
    gfw.Finalize();
    // TIME ライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
}
