﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "FsSdCard.h"

bool FsSdCard::Initialize(const char* filePath) NN_NOEXCEPT
{
    nn::Result result;
    bool ret = true;
    m_IsMounted = true;

    NN_ABORT_UNLESS_NOT_NULL(filePath);
    strcpy(m_filePath, filePath);

    // SD カードをマウント名 "sd" でマウント
    result = nn::fs::MountSdCardForDebug("sd");
    if( nn::fs::ResultSdCardAccessFailed::Includes(result) )
    {
        NN_LOG("SD card not found.");
        m_IsMounted = false;
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // ファイルが存在しない場合は作成
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, m_filePath);
        if( nn::fs::ResultPathNotFound().Includes(result) )
        {
            NN_LOG("Create \"%s\"\n", m_filePath);
            result = nn::fs::CreateFile(m_filePath, 0);
            if( nn::fs::ResultPathAlreadyExists::Includes(result)
                  || nn::fs::ResultTargetLocked::Includes(result) )
            {
                // 対象ファイルが既に存在している。エラーにする。
                NN_LOG("The file already exists.\n");
                ret = false;
            }
            else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
            {
                // 空き領域が不足
                NN_LOG("Usable space not enough.\n");
                ret = false;
            }
        }
    }

    if( ret == false )
    {
        // アンマウント
        nn::fs::Unmount("sd");
    }
    return ret;
}

void FsSdCard::Finalize() NN_NOEXCEPT
{
    if( m_IsMounted == true )
    {
        // アンマウント
        nn::fs::Unmount("sd");
    }
}

bool FsSdCard::Append(const void* buf, size_t length) NN_NOEXCEPT
{
    nn::Result result;
    bool ret = true;

    // ファイル末尾に追記
    {
        nn::fs::FileHandle fileHandle;
        result = nn::fs::OpenFile(&fileHandle, m_filePath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            NN_LOG("Can't find the file.\n");
            return false;
        }
        else if( nn::fs::ResultTargetLocked::Includes(result) )
        {
            NN_LOG("The file is already opened.\n");
            return false;
        }

        int64_t fileSize = 0;
        (void)nn::fs::GetFileSize(&fileSize, fileHandle);

        result = nn::fs::WriteFile(fileHandle, fileSize, buf, length, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // 空き領域が不足
            NN_LOG("Usable space not enough.\n");
            ret = false;
        }
        nn::fs::CloseFile(fileHandle);
    }

    return ret;
}

