﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Npad.h"

#include <nn/nn_Log.h>
#include <nn/os.h>


bool Npad::m_IsIrConfigurationEnabled = false;
bool Npad::m_IsEnabled = true;

const nn::hid::NpadIdType Npad::m_NpadIds[Npad::NpadIdCountMax] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::Handheld
};

Npad::Npad()
{
    m_HoldDetectionDelay = nn::TimeSpan::FromMilliSeconds(300);
    m_HoldDetectionInterval = nn::TimeSpan::FromMilliSeconds(30);
    Clear();
}

Npad::~Npad()
{
}

Npad& Npad::GetInstance()
{
    static Npad pad;
    static bool first = true;

    if( first )
    {
        if( pad.m_IsEnabled )
        {
            pad.Clear();

            nn::hid::InitializeNpad();

            // 使用する操作形態を設定
            nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask |
                                              nn::hid::NpadStyleJoyLeft::Mask |
                                              nn::hid::NpadStyleJoyRight::Mask|
                                              nn::hid::NpadStyleJoyDual::Mask |
                                              nn::hid::NpadStyleHandheld::Mask);

            // 使用するNpadId の設定
            nn::hid::SetSupportedNpadIdType(pad.m_NpadIds, NpadIdCountMax);

        }
        first = false;
    }

    return pad;
};

void Npad::Finalize()
{
}

void Npad::UpdatePadState()
{
    if( !m_IsEnabled )
    {
        return;
    }

    static nn::hid::NpadFullKeyState previousBuf;
    static nn::hid::NpadFullKeyState currentBuf;

    Npad& pad = GetInstance();
    nn::hid::NpadButtonSet buttons;

    buttons.Reset();

    // すべての入力の論理和を入力とする
    for(int i=0; i<NpadIdCountMax; ++i)
    {
        //現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(pad.m_NpadIds[i]);

        // フルキー操作が有効な場合
        if( style.Test<nn::hid::NpadStyleFullKey>() == true )
        {
            nn::hid::NpadFullKeyState currentFullKeyBuf;
            nn::hid::GetNpadState(&currentFullKeyBuf, pad.m_NpadIds[i]);
            buttons |= currentFullKeyBuf.buttons;
        }

        // Joy-Con 操作が有効な場合
        if( style.Test<nn::hid::NpadStyleJoyDual>() == true )
        {
            nn::hid::NpadJoyDualState currentJoyDualBuf;
            nn::hid::GetNpadState(&currentJoyDualBuf, pad.m_NpadIds[i]);
            buttons |= currentJoyDualBuf.buttons;
        }

        if( style.Test<nn::hid::NpadStyleJoyLeft>() == true )
        {
            nn::hid::NpadJoyLeftState currentJoyLeftBuf;
            nn::hid::GetNpadState(&currentJoyLeftBuf, pad.m_NpadIds[i]);
            buttons |= currentJoyLeftBuf.buttons;
        }

        if( style.Test<nn::hid::NpadStyleJoyRight>() == true )
        {
            nn::hid::NpadJoyRightState currentJoyRightBuf;
            nn::hid::GetNpadState(&currentJoyRightBuf, pad.m_NpadIds[i]);
            buttons |= currentJoyRightBuf.buttons;
        }

        // 携帯機コントローラー操作が有効な場合
        if(style.Test<nn::hid::NpadStyleHandheld>() == true )
        {
            nn::hid::NpadHandheldState currentHandheldBuf;
            nn::hid::GetNpadState(&currentHandheldBuf, pad.m_NpadIds[i]);
            buttons |= currentHandheldBuf.buttons;
        }
    }

    currentBuf.buttons = buttons;

    nn::hid::NpadButtonSet currentButtons = currentBuf.buttons;
    nn::hid::NpadButtonSet previousButtons = previousBuf.buttons;

    pad.m_CurrentState.trigger = (~previousButtons) & currentButtons;
    pad.m_CurrentState.hold = currentButtons;
    pad.m_CurrentState.release = previousButtons & (~currentButtons);

    if( pad.m_CurrentState.trigger.IsAnyOn() )
    {
        pad.m_PressTime = nn::os::GetSystemTick().ToTimeSpan();
        pad.m_PreviousDetectionTime = 0;
    }

    // 一つ前の入力を保存
    previousBuf = currentBuf;
}

bool Npad::IsTrigger(Button button)
{
    if( m_CurrentState.trigger.Test(ToNpadButton(button)) )
    {
        return true;
    }

    return false;
}

bool Npad::IsHold(Button button)
{
    if( m_CurrentState.hold.Test((ToNpadButton(button))) )
    {
        nn::TimeSpan now = nn::os::GetSystemTick().ToTimeSpan();

        if( ((now - m_PressTime) > m_HoldDetectionDelay) && (m_PreviousDetectionTime == 0) )
        {
            m_PreviousDetectionTime = now;

            return true;
        }
        else if( ((now - m_PreviousDetectionTime) > m_HoldDetectionInterval) && (m_PreviousDetectionTime != 0) )
        {
            m_PreviousDetectionTime = now;
            return true;
        }
    }
    return false;
}

bool Npad::IsRelease(Button button)
{
    if( m_CurrentState.release.Test((ToNpadButton(button))) )
    {
        return true;
    }

    return false;
}

void Npad::Clear()
{
    m_CurrentState.trigger.Reset();
    m_CurrentState.hold.Reset();
    m_CurrentState.release.Reset();

    m_PreviousDetectionTime -= m_PreviousDetectionTime;
}

void Npad::DisconnectNpad(const nn::hid::NpadIdType& id)
{
    nn::hid::DisconnectNpad(id);
}

void Npad::DisconnectAllNpads()
{
    Npad& pad = GetInstance();
    for(int i=0; i<NpadIdCountMax; ++i)
    {
        nn::hid::DisconnectNpad(pad.m_NpadIds[i]);
    }
}

void Npad::EnablePad()
{
    m_IsEnabled = true;
}

void Npad::DisablePad()
{
    m_IsEnabled = false;
}

int Npad::ToNpadButton(Npad::Button button)
{
    switch(button)
    {
    case A        : return 0;
    case B        : return 1;
    case X        : return 2;
    case Y        : return 3;
    case SL       : return 4;
    case SR       : return 5;
    case L        : return 6;
    case R        : return 7;
    case ZL       : return 8;
    case ZR       : return 9;
    case START    :
    case PLUS     : return 10;
    case SELECT   :
    case MINUS    : return 11;
    case LEFT     : return 12;
    case UP       : return 13;
    case RIGHT    : return 14;
    case DOWN     : return 15;
    case SL_LEFT  : return 16;
    case SL_UP    : return 17;
    case SL_RIGHT : return 18;
    case SL_DOWN  : return 19;
    case SR_LEFT  : return 20;
    case SR_UP    : return 21;
    case SR_RIGHT : return 22;
    case SR_DOWN  : return 23;
    default       : return 63;
    }
}

