﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <cstring>
#include <nn/hid.h>
#include <nn/wlan/wlan_Result.h>
#include "../common.h"

namespace {

// スキャンバッファ
const size_t  buffSize = 100 * 1024; // 100KB
NN_ALIGNAS(4096) char  g_scanBuffer[ buffSize ];

const size_t ThreadStackSize = 4096;              // スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackProcess[ ThreadStackSize * 4 ];
nn::os::ThreadType  g_ThreadProcess;
bool g_exitThread;

nn::os::SystemEventType g_connectionEvent;
nn::wlan::ConnectionStatus g_connectionStatus;

int TargetApCount;
struct TargetApInfo {
    nn::wlan::Ssid ssid;
    nn::wlan::Security security;
    int16_t channel;
};
TargetApInfo g_ApInfoList[20];

}

void InfraScan()
{
    nn::Result result;

    nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Passive,
            {36, 40, 44, 48},
            0,  // all channel scan
            120,
            0,
            NULL,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };

    /*
    nn::wlan::Ssid pSsidList[3];
    pSsidList[0].Set(WlanTest::localMasterSsid);
    pSsidList[1].Set("WLAN_TEST_WPA2_AES");
    pSsidList[2].Set("montana-wg1400hp");
    scanParam.scanType = nn::wlan::ScanType_Active;
    scanParam.ssidCount = 3;
    scanParam.ssidList = pSsidList;
    */

    NN_LOG("Do scan\n");
    nn::wlan::Infra::StartScan(g_scanBuffer, buffSize, scanParam);
    NN_LOG("Scan finished.\n");
    nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);
    NN_LOG("Found %d APs\n", resultReader.GetCount());
    WlanTest::DumpScanResultWithReader(g_scanBuffer);
}

void InfraConnect(int apPattern)
{
    nn::Result result;

    NN_LOG("Try to connect...\n");
    nn::wlan::Infra::Connect(g_ApInfoList[apPattern].ssid,
                             nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                             g_ApInfoList[apPattern].channel,
                             g_ApInfoList[apPattern].security,
                             false);
}

void InfraDisconnect()
{
    // Disconnects from AP
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());
}

void EventMonitor(void* arg)
{
    NN_UNUSED(arg);

    nn::wlan::Infra::GetConnectionEvent(&g_connectionEvent);

    while( !g_exitThread )
    {
        if( nn::os::TryWaitSystemEvent(&g_connectionEvent) == true )
        {
            nn::wlan::Infra::GetConnectionStatus(&g_connectionStatus);
            WlanTest::PrintConnectionStatus(&g_connectionStatus);
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

void InitializeTargetApInfo()
{
    int i = 0;
    /* AP */
    g_ApInfoList[i].ssid.Set("WLAN_TEST_WEP128_OPEN");
    g_ApInfoList[i].security.privacyMode = nn::wlan::SecurityMode_Wep128Open;
    g_ApInfoList[i].security.groupPrivacyMode = nn::wlan::SecurityMode_Wep128Open;
    g_ApInfoList[i].security.keyIdx = 0;
    //std::strcpy(reinterpret_cast<char*>(g_ApInfoList[i].security.key), "30313233343536373839616263");  // ASCII string "0123456789abc"
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[i].security.key), "30313233343536373839616264");  // ASCII string "0123456789abd"
    g_ApInfoList[i].channel = -1;
    i++;

    /* AP */
    g_ApInfoList[i].ssid.Set("WLAN_TEST_WPA_TKIP");
    g_ApInfoList[i].security.privacyMode = nn::wlan::SecurityMode_WpaTkip;
    g_ApInfoList[i].security.groupPrivacyMode = nn::wlan::SecurityMode_WpaTkip;
    g_ApInfoList[i].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[i].security.key), "012345678A");
    g_ApInfoList[i].channel = 6;
    i++;

    /* AP */
    g_ApInfoList[i].ssid.Set("AP3-WPA-PSK_TKIP");
    g_ApInfoList[i].security.privacyMode = nn::wlan::SecurityMode_WpaTkip;
    g_ApInfoList[i].security.groupPrivacyMode = nn::wlan::SecurityMode_WpaTkip;
    g_ApInfoList[i].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[i].security.key), "0123456aaab");
    g_ApInfoList[i].channel = -1;  // 48ch
    i++;

    /* AP */
    g_ApInfoList[i].ssid.Set("WLAN_TEST_OPEN_HIDDEN");
    g_ApInfoList[i].security.privacyMode = nn::wlan::SecurityMode_Open;
    g_ApInfoList[i].security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
    g_ApInfoList[i].security.keyIdx = 0;
    g_ApInfoList[i].channel = -1;
    i++;

    /* AP */
    g_ApInfoList[i].ssid.Set("WLAN_TEST_WPA2_AES");
    g_ApInfoList[i].security.privacyMode = nn::wlan::SecurityMode_Wpa2Aes;
    g_ApInfoList[i].security.groupPrivacyMode = nn::wlan::SecurityMode_Wpa2Aes;
    g_ApInfoList[i].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[i].security.key), "012345678Z");
    g_ApInfoList[i].channel = -1;
    i++;

    /* AP4 */
    g_ApInfoList[i].ssid.Set("kurose_wps_test");
    g_ApInfoList[i].security.privacyMode = nn::wlan::SecurityMode_Wpa2Aes;
    g_ApInfoList[i].security.groupPrivacyMode = nn::wlan::SecurityMode_Wpa2Aes;
    g_ApInfoList[i].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[i].security.key), "Shi2iTaiZen");
    g_ApInfoList[i].channel = 11;
    i++;

    /* AP */
    g_ApInfoList[i].ssid.Set("AP2-WPA2_AES");
    g_ApInfoList[i].security.privacyMode = nn::wlan::SecurityMode_Wpa2Aes;
    g_ApInfoList[i].security.groupPrivacyMode = nn::wlan::SecurityMode_Wpa2Aes;
    g_ApInfoList[i].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[i].security.key), "AbcdefghijklmnopqrstuvwxyZA");
    g_ApInfoList[i].channel = -1; // 6ch
    i++;

    /* AP */
    g_ApInfoList[i].ssid.Set("WPA-PSK");
    g_ApInfoList[i].security.privacyMode = nn::wlan::SecurityMode_WpaAes;
    g_ApInfoList[i].security.groupPrivacyMode = nn::wlan::SecurityMode_WpaAes;
    g_ApInfoList[i].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[i].security.key), "Passphrase");
    g_ApInfoList[i].channel = -1;  // 11ch
    i++;

    TargetApCount = i;
}

void InitializeWlanInfra()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
    char str[256];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::GetFwVersion(str, sizeof(str)));
    NN_LOG("FW ver:\n%s\n", str);
}

void FinalizeWlanInfra()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
}

void RequestSleep()
{
    NN_LOG("%s\n", __FUNCTION__);
    nn::Result result;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
    result = nn::wlan::Infra::RequestSleep();
    if( result.IsSuccess() )
    {
        NN_LOG("wlan sleep is accepted\n");
    }
    else
    {
        NN_LOG("sleep failure: Description[%d]\n", result.GetDescription());
    }
}

void RequestWakeUp()
{
    NN_LOG("%s\n", __FUNCTION__);
    nn::Result result;
    result = nn::wlan::Infra::RequestWakeUp();
    if( result.IsSuccess() )
    {
        NN_LOG("wlan wake up is accepted\n");
    }
    else
    {
        NN_LOG("wake up failure: Description[%d]\n", result.GetDescription());
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
}

void ConnectionRepeat(int apPattern)
{
    g_exitThread = true;
    nn::os::WaitThread( &g_ThreadProcess );
    nn::os::DestroyThread( &g_ThreadProcess );

    int i = 0;
    int j = 0;
    while( 1 )
    {
        InfraConnect(apPattern);

        nn::os::WaitSystemEvent(&g_connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&g_connectionStatus);
        if( g_connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("Connection success[%d]\n", ++i);
            j = 0;
            WlanTest::PrintConnectionStatus(&g_connectionStatus);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

            while( 1 )
            {
                NN_LOG("Try to disconnect\n");
                InfraDisconnect();
                nn::os::WaitSystemEvent(&g_connectionEvent);
                nn::wlan::Infra::GetConnectionStatus(&g_connectionStatus);
                if( g_connectionStatus.state != nn::wlan::ConnectionState_Connected )
                {
                    WlanTest::PrintConnectionStatus(&g_connectionStatus);
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        }
        else
        {
            NN_LOG("Connection failed[%d]. Retry 3 seconds later.\n", ++j);
            // CauseOfInfoが想定内かどうかチェック
            if( g_connectionStatus.cause == nn::wlan::CauseOfInfo_DisconnectReq )
            {
                NN_ABORT("Unexpected CauseOfInfo!!\n");
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
    }

}

void InfraConnectWithWps(nn::wlan::WpsMethod method)
{
    nn::Result result;

    NN_LOG("Try to connect with WPS...\n");
    if( method == nn::wlan::WpsMethod_Pin )
    {
        char pin[nn::wlan::WpsPinLength + 1];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::GenerateWpsPin(pin, sizeof(pin)));
        NN_LOG("\nPIN CODE\n%s\n\n", pin);
        NN_LOG("wait 30 seconds\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(30));
        NN_LOG("start WPS PIN connection\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, pin, sizeof(pin), 10));
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pbc, NULL, 0, 10));
    }

}

void PrintMenu()
{
    NN_LOG("A : Scan\n");
    NN_LOG("B : Disconnect\n");
    NN_LOG("X : Connect\n");
    NN_LOG("Y : Connect repeat\n");
    NN_LOG("UP : Wake up\n");
    NN_LOG("DOWN : Sleep\n");
    NN_LOG("L/R : Change target AP info\n");
    NN_LOG("ZL : WPS PBC\n");
    NN_LOG("ZR : WPS PIN\n");
    NN_LOG("SELECT : end test\n");
}

extern "C" void nnMain()
{
    NN_LOG("\n\n Start InfraControl\n\n");

    WlanTest::SystemInitialize();
    TargetApCount = 0;
    InitializeTargetApInfo();

    nn::hid::InitializeDebugPad();
    int32_t prevPadSamplingNumber = 0;
    nn::hid::DebugPadState debugPadState[nn::hid::DebugPadStateCountMax];

    PrintMenu();

    InitializeWlanInfra();

    // イベント監視スレッド起動
    g_exitThread = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread( &g_ThreadProcess, EventMonitor, NULL, g_ThreadStackProcess, ThreadStackSize, nn::os::DefaultThreadPriority - 1 )
    );
    nn::os::StartThread( &g_ThreadProcess );

    int apPattern = 0;
    while( 1 )
    {
        // DebugPad の入力を取得します。
        // debugPad の状態を表す構造体を用意します。
        nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);

        // 直前から現在までの Pad 入力の更新回数を取得します。
        int32_t padSamplingCount = debugPadState[0].samplingNumber - prevPadSamplingNumber;
        if (padSamplingCount >= nn::hid::DebugPadStateCountMax)
        {
            padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
        }
        nn::hid::DebugPadButtonSet padButtonDown(debugPadState[0].buttons & ~debugPadState[padSamplingCount].buttons);

        // 現在までの Pad 入力の更新回数を取得します。
        prevPadSamplingNumber = debugPadState[0].samplingNumber;

        if( padButtonDown.Test<nn::hid::DebugPadButton::A>() )
        {
            // Scan
            InfraScan();
            PrintMenu();
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::B>() )
        {
            InfraDisconnect();
            PrintMenu();
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::X>() )
        {
            InfraConnect(apPattern);
            PrintMenu();
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::Y>() )
        {
            ConnectionRepeat(apPattern);
            PrintMenu();
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::Up>() )
        {
            RequestWakeUp();
            PrintMenu();
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::Down>() )
        {
            RequestSleep();
            PrintMenu();
        }
        else if (padButtonDown.Test<nn::hid::DebugPadButton::L>())
        {
            apPattern++;
            if (apPattern > TargetApCount - 1)
            {
                apPattern = 0;
            }
            NN_LOG("TargetSSID:%s\n", g_ApInfoList[apPattern].ssid.GetSsidData());
        }
        else if (padButtonDown.Test<nn::hid::DebugPadButton::R>())
        {
            apPattern--;
            if (apPattern < 0)
            {
                apPattern = TargetApCount - 1;
            }
            NN_LOG("TargetSSID:%s\n", g_ApInfoList[apPattern].ssid.GetSsidData());
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::ZL>() )
        {
            InfraConnectWithWps(nn::wlan::WpsMethod_Pbc);
            PrintMenu();
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::ZR>() )
        {
            InfraConnectWithWps(nn::wlan::WpsMethod_Pin);
            PrintMenu();
        }
        else if(padButtonDown.Test<nn::hid::DebugPadButton::Select>())
        {
            NN_LOG("leave the loop.\n");
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    g_exitThread = true;
    nn::os::WaitThread( &g_ThreadProcess );
    nn::os::DestroyThread( &g_ThreadProcess );

    FinalizeWlanInfra();

    WlanTest::SystemFinalize();

    NN_LOG("\n\n End InfraControl \n\n");
} //NOLINT(impl/function_size)

