﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Network.h"
#include <nn/bsdsocket/cfg/cfg.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include <nn/nn_Log.h>
#include <map>


namespace
{

static const std::map<nn::wlan::LinkLevel, std::string> g_LevelNameMap = {
    { nn::wlan::LinkLevel_0, "Worst" },
    { nn::wlan::LinkLevel_1, "Bad" },
    { nn::wlan::LinkLevel_2, "Better" },
    { nn::wlan::LinkLevel_3, "Best" },
};

}


namespace ApConnectivityTest
{

// コンストラクタ
Network::WlanConnection::WlanConnection(nn::wlan::WpsMethod method, nn::bsdsocket::cfg::IfSettings bsdSettings) :
        m_WpsMethod(method), m_BsdSettings(bsdSettings)
{
    nn::nifm::SetWirelessCommunicationEnabled(false);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    nn::wlan::InitializeInfraManager();
    nn::wlan::Infra::OpenMode();
}


// デストラクタ
Network::WlanConnection::~WlanConnection()
{
    Disconnect();

    nn::wlan::Infra::CloseMode();
    nn::wlan::FinalizeInfraManager();

    nn::nifm::SetWirelessCommunicationEnabled(true);
}


// 接続
bool Network::WlanConnection::Connect()
{
    bool resultBool = false;
    nn::Result result;

    // PIN モード
    if (m_WpsMethod == nn::wlan::WpsMethod_Pin)
    {
        NN_LOG("Start WPS Pin...\n");

        char pin[nn::wlan::WpsPinLength + 1];
        nn::wlan::Infra::GenerateWpsPin(pin, sizeof(pin));
        NN_LOG("Pin: %s\n", pin);

        NN_LOG("Wait 30 seconds... (Set client PIN to AP and start WPS PIN authentication)\n", pin);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(30));

        result = nn::wlan::Infra::ConnectWithWps(m_WpsMethod, pin, sizeof(pin));
    }
    // Push Button Connection モード
    else if (m_WpsMethod == nn::wlan::WpsMethod_Pbc)
    {
        NN_LOG("Start WPS PBC...\n");

        result = nn::wlan::Infra::ConnectWithWps(m_WpsMethod, nullptr, 0);
    }

    nn::wlan::ConnectionStatus status;
    nn::wlan::Infra::GetConnectionStatus(&status);

    if (result.IsSuccess() && status.state == nn::wlan::ConnectionState_Connected)
    {
        NN_LOG("Success WPS\n\n");

        Network::GetInstance().m_IsNifmConnection = false;

        char ssidString[nn::wlan::Ssid::SsidHexStringLengthMax + 1] = {};
        status.ssid.GetHexString(ssidString);

        char macString[nn::wlan::MacAddress::MacStringSize + 1] = {};
        status.bssid.GetString(macString);

        nn::wlan::LinkLevel linkLevel;
        nn::wlan::Infra::GetLinkLevel(&linkLevel);

        NN_LOG("SSID      : %s\n", ssidString);
        NN_LOG("BSSID     : %s\n", macString);
        NN_LOG("LinkLevel : %d (%s)\n", linkLevel, g_LevelNameMap.at(linkLevel).c_str());
        NN_LOG("Channel   : %d\n", status.channel);

        NN_LOG("\n");
        NN_LOG("Interface up...\n\n");

        result = nn::bsdsocket::cfg::SetIfUp("wl0", &m_BsdSettings);

        if (result.IsSuccess())
        {
            nn::bsdsocket::cfg::IfState state;
            nn::bsdsocket::cfg::GetIfState("wl0", &state);

            NN_LOG("IP              : %d.%d.%d.%d\n",
                state.addr.S_addr & 0xff,
                (state.addr.S_addr >> 8) & 0xff,
                (state.addr.S_addr >> 16) & 0xff,
                (state.addr.S_addr >> 24) & 0xff
            );

            NN_LOG("Subnet Mask     : %d.%d.%d.%d\n",
                state.subnetMask.S_addr & 0xff,
                (state.subnetMask.S_addr >> 8) & 0xff,
                (state.subnetMask.S_addr >> 16) & 0xff,
                (state.subnetMask.S_addr >> 24) & 0xff
            );

            NN_LOG("Default Gateway : %d.%d.%d.%d\n",
                state.gatewayAddr.S_addr & 0xff,
                (state.gatewayAddr.S_addr >> 8) & 0xff,
                (state.gatewayAddr.S_addr >> 16) & 0xff,
                (state.gatewayAddr.S_addr >> 24) & 0xff
            );

            NN_LOG("Primary DNS     : %d.%d.%d.%d\n",
                state.dnsAddrs[0].S_addr & 0xff,
                (state.dnsAddrs[0].S_addr >> 8) & 0xff,
                (state.dnsAddrs[0].S_addr >> 16) & 0xff,
                (state.dnsAddrs[0].S_addr >> 24) & 0xff
            );

            NN_LOG("Secondary DNS   : %d.%d.%d.%d\n",
                state.dnsAddrs[1].S_addr & 0xff,
                (state.dnsAddrs[1].S_addr >> 8) & 0xff,
                (state.dnsAddrs[1].S_addr >> 16) & 0xff,
                (state.dnsAddrs[1].S_addr >> 24) & 0xff
            );

            NN_LOG("\n\n");

            resultBool = true;
        }
    }
    else
    {
        NN_LOG("Failed WPS\n");
    }

    return resultBool;
}


// 切断
void Network::WlanConnection::Disconnect()
{
    nn::bsdsocket::cfg::SetIfDown("wl0");

    nn::wlan::Infra::Disconnect();
}

}
