﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Network.h"
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiScan.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include <nn/nn_Log.h>
#include <nn/os/os_MultipleWait.h>


namespace
{
const size_t g_NifmWatchThreadStackSize = 0x10000;
}


namespace ApConnectivityTest
{

// コンストラクタ
Network::NifmConnection::NifmConnection(const nn::nifm::NetworkProfileData& profileData, bool mixedMode, bool permanentProfile) :
        m_pNetworkConnection(nullptr),
        m_pTemporaryNetworkProfile(nullptr),
        m_ProfileData(profileData),
        m_MixedMode(mixedMode),
        m_PermanentProfile(permanentProfile),
        m_NifmWatchThreadExitEvent(nn::os::EventClearMode_AutoClear)
{
    nn::nifm::SetWirelessCommunicationEnabled(true);

    auto stack = m_NifmWatchThreadStack = new char[g_NifmWatchThreadStackSize + nn::os::ThreadStackAlignment];

    stack += nn::os::ThreadStackAlignment - 1;
    stack -= reinterpret_cast<uintptr_t>(stack) % nn::os::ThreadStackAlignment;

    void(NifmConnection::*pTheadFunc)() = &NifmConnection::NifmWatchThread;
    nn::os::CreateThread(
            &m_NifmWatchThread,
            *reinterpret_cast<nn::os::ThreadFunction*>(&pTheadFunc), this,
            stack, g_NifmWatchThreadStackSize, nn::os::DefaultThreadPriority);
    nn::os::SetThreadName(&m_NifmWatchThread, "NifmWatchThread");
}


// デストラクタ
Network::NifmConnection::~NifmConnection()
{
    Disconnect();

    nn::os::DestroyThread(&m_NifmWatchThread);
    delete[] m_NifmWatchThreadStack;
}


// 接続
bool Network::NifmConnection::Connect()
{
    // 既存のプロファイルを削除しておく
    {
        int profilesCount = 0;
        nn::nifm::EnumerateNetworkProfiles(reinterpret_cast<nn::nifm::NetworkProfileBasicInfo*>(1), &profilesCount, 0,
            nn::nifm::NetworkProfileType_SsidList |
            nn::nifm::NetworkProfileType_Temporary |
            nn::nifm::NetworkProfileType_User);
        std::vector<nn::nifm::NetworkProfileBasicInfo> profiles(profilesCount);
        nn::nifm::EnumerateNetworkProfiles(profiles.data(), &profilesCount, static_cast<int>(profiles.size()),
            nn::nifm::NetworkProfileType_SsidList |
            nn::nifm::NetworkProfileType_Temporary |
            nn::nifm::NetworkProfileType_User);

        for (auto &profile : profiles)
        {
            nn::nifm::RemoveNetworkProfile(profile.id);
        }
    }

    // Mixed Mode の場合先にスキャンしておく
    int mixedScore = 0;
    if (m_MixedMode)
    {
        int authenticationScore[] = {
            0,  // Authentication_Invalid
            0,  // Authentication_Open
            0,  // Authentication_Shared
            0,  // Authentication_Wpa
            1,  // Authentication_WpaPsk
            0,  // Authentication_Wpa2
            2,  // Authentication_Wpa2Psk
        };

        int encryptionScore[] = {
            0,  // Encryption_Invalid
            0,  // Encryption_None
            0,  // Encryption_Wep
            1,  // Encryption_Tkip
            3,  // Encryption_Aes
        };

        nn::nifm::Scan();
        nn::nifm::AccessPointData apDatas[nn::nifm::AccessPointCountMax];
        int apCount;
        nn::nifm::GetScanData(apDatas, &apCount, sizeof(apDatas) / sizeof(*apDatas));

        apCount = std::min(apCount, nn::nifm::AccessPointCountMax);
        for (int i = 0; i < apCount; ++i)
        {
            if (apDatas[i].ssid.length == m_ProfileData.wirelessSetting.ssidConfig.ssid.length &&
                    std::memcmp(apDatas[i].ssid.hex, m_ProfileData.wirelessSetting.ssidConfig.ssid.hex, apDatas[i].ssid.length))
            {
                if (authenticationScore[apDatas[i].authentication] * encryptionScore[apDatas[i].encryption] > mixedScore)
                {
                    m_ProfileData.wirelessSetting.security.authEncryption.authentication = apDatas[i].authentication;
                    m_ProfileData.wirelessSetting.security.authEncryption.encryption = apDatas[i].encryption;
                }
            }
        }
    }

    // プロファイルを作成
    nn::nifm::TemporaryNetworkProfile* pTemporaryNetworkProfile = nullptr;
    if (m_PermanentProfile)
    {
        NN_LOG("Saving Network Profile\n");
        nn::nifm::SetNetworkProfile(&m_ProfileUuid, m_ProfileData);
    }
    else
    {
        pTemporaryNetworkProfile = new nn::nifm::TemporaryNetworkProfile(m_ProfileData);
        m_ProfileUuid = pTemporaryNetworkProfile->GetId();
    }

    // 接続試行
    auto pNetworkConnection = new nn::nifm::NetworkConnection(nn::os::EventClearMode_AutoClear);

    nn::nifm::RequestHandle requestHandle = pNetworkConnection->GetRequestHandle();
    nn::nifm::SetRequestNetworkProfileId(requestHandle, m_ProfileUuid);

    nn::nifm::SetRequestConnectionConfirmationOption(requestHandle, nn::nifm::ConnectionConfirmationOption_Prohibited);

    pNetworkConnection->SubmitRequestAndWait();
    auto succeed = pNetworkConnection->IsAvailable();

    if (succeed)
    {
        m_pNetworkConnection = pNetworkConnection;
        m_pTemporaryNetworkProfile = pTemporaryNetworkProfile;

        // 監視スレッドを開始
        m_NifmWatchThreadExitEvent.Clear();
        nn::os::StartThread(&m_NifmWatchThread);
    }
    // 失敗したら接続要求取り消しなど
    else
    {
        auto result = pNetworkConnection->GetResult();
        NN_LOG("nifm : %08X\n", result.GetInnerValueForDebug());

        delete pNetworkConnection;
        if (pTemporaryNetworkProfile)
        {
            delete pTemporaryNetworkProfile;
        }
    }

    return succeed;
}


// 切断
void Network::NifmConnection::Disconnect()
{
    // ネットワーク利用要求を取り下げる
    if (m_pNetworkConnection)
    {
        // 監視スレッドを終了
        m_NifmWatchThreadExitEvent.Signal();
        nn::os::WaitThread(&m_NifmWatchThread);

        m_pNetworkConnection->CancelRequest();
        delete m_pNetworkConnection;
        m_pNetworkConnection = nullptr;

        delete m_pTemporaryNetworkProfile;
        m_pTemporaryNetworkProfile = nullptr;
    }
}


// ネットワーク状態監視スレッド
void Network::NifmConnection::NifmWatchThread()
{
    nn::os::TimerEvent retryTimerEvent(nn::os::EventClearMode_AutoClear);
    retryTimerEvent.Signal();

    auto& nifmEvent = m_pNetworkConnection->GetSystemEvent();

    nn::os::MultiWaitHolderType exitEventWaitHolder;
    nn::os::InitializeMultiWaitHolder(&exitEventWaitHolder, m_NifmWatchThreadExitEvent.GetBase());
    nn::os::SetMultiWaitHolderUserData(&exitEventWaitHolder, 0);

    nn::os::MultiWaitHolderType nifmEventWaitHolder;
    nn::os::InitializeMultiWaitHolder(&nifmEventWaitHolder, nifmEvent.GetBase());
    nn::os::SetMultiWaitHolderUserData(&nifmEventWaitHolder, 1);

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::LinkMultiWaitHolder(&multiWait, &nifmEventWaitHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &exitEventWaitHolder);

    for (uintptr_t userData; (userData = nn::os::WaitAny(&multiWait)->userData); )
    {
        nifmEvent.Wait();

        if (m_pNetworkConnection->IsRequestOnHold())
        {
            NN_LOG("Connecting...\n");
            continue;
        }
        else if (!m_pNetworkConnection->IsAvailable())
        {
            NN_LOG("Disconnected\n");

            auto result = m_pNetworkConnection->GetResult();
            NN_LOG("nifm : %08X\n", result.GetInnerValueForDebug());

            retryTimerEvent.Wait();

            nn::nifm::RequestHandle requestHandle = m_pNetworkConnection->GetRequestHandle();
            nn::nifm::SetRequestNetworkProfileId(requestHandle, m_ProfileUuid);

            m_pNetworkConnection->SubmitRequest();

            retryTimerEvent.StartOneShot(nn::TimeSpan::FromSeconds(1));
        }
        else
        {
            NN_LOG("Connected\n");
        }
    }

    m_pNetworkConnection->CancelRequest();

    nn::os::UnlinkAllMultiWaitHolder(&multiWait);
    nn::os::FinalizeMultiWait(&multiWait);

    nn::os::FinalizeMultiWaitHolder(&nifmEventWaitHolder);
    nn::os::FinalizeMultiWaitHolder(&exitEventWaitHolder);
}

}
