﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "WebSimple_Pad.hpp"

#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>

// Button と DebugPad の対応関係
const int SamplePad::DebugPadMap[Button_Count] =
{
    nn::hid::DebugPadButton::A::Index,
    nn::hid::DebugPadButton::B::Index,
    nn::hid::DebugPadButton::X::Index,
    nn::hid::DebugPadButton::Y::Index,
    nn::hid::DebugPadButton::L::Index,
    nn::hid::DebugPadButton::R::Index,
    nn::hid::DebugPadButton::ZR::Index,
    nn::hid::DebugPadButton::Left::Index,
    nn::hid::DebugPadButton::Right::Index,
    nn::hid::DebugPadButton::Up::Index,
    nn::hid::DebugPadButton::Down::Index,
    nn::hid::DebugPadButton::Start::Index,
};

// Button と NPad の対応関係
const int SamplePad::NpadMap[Button_Count] =
{
    nn::hid::NpadButton::A::Index,
    nn::hid::NpadButton::B::Index,
    nn::hid::NpadButton::X::Index,
    nn::hid::NpadButton::Y::Index,
    nn::hid::NpadButton::L::Index,
    nn::hid::NpadButton::R::Index,
    nn::hid::NpadButton::ZR::Index,
    nn::hid::NpadButton::Left::Index,
    nn::hid::NpadButton::Right::Index,
    nn::hid::NpadButton::Up::Index,
    nn::hid::NpadButton::Down::Index,
    nn::hid::NpadButton::Plus::Index,
};

SamplePad::SamplePad() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();
    nn::hid::InitializeDebugPad();

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA = nn::hid::KeyboardKey::A::Index;
    map.buttonB = nn::hid::KeyboardKey::B::Index;
    map.buttonX = nn::hid::KeyboardKey::X::Index;
    map.buttonY = nn::hid::KeyboardKey::Y::Index;
    map.buttonL = nn::hid::KeyboardKey::L::Index;
    map.buttonR = nn::hid::KeyboardKey::R::Index;
    map.buttonZR = nn::hid::KeyboardKey::Z::Index;
    map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);

    const nn::hid::NpadIdType npadIds[2] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::Handheld
    };
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    ResetCurrentButtons();
    CopyStateCurrentToPrevious();
}

SamplePad::~SamplePad() NN_NOEXCEPT
{
}

void SamplePad::Update() NN_NOEXCEPT
{
    CopyStateCurrentToPrevious();
    ResetCurrentButtons();

    const nn::hid::NpadStyleSet styleNo1 = nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1);
    const nn::hid::NpadStyleSet styleHandheld =
        nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld);

    if( styleNo1.Test<nn::hid::NpadStyleFullKey>() )
    {
        nn::hid::GetNpadState(&m_NpadFullKeyCurrentState, nn::hid::NpadId::No1);
    }
    else if( styleHandheld.Test<nn::hid::NpadStyleHandheld>() )
    {
        nn::hid::GetNpadState(&m_NpadHandheldCurrentState, nn::hid::NpadId::Handheld);
    }

    nn::hid::GetDebugPadState(&m_DebugPadCurrentState);
}

bool SamplePad::IsHold(Button button) NN_NOEXCEPT
{
    bool isHold = false;

    isHold |= m_DebugPadCurrentState.buttons.Test(DebugPadMap[button]);
    isHold |= m_NpadFullKeyCurrentState.buttons.Test(NpadMap[button]);
    isHold |= m_NpadHandheldCurrentState.buttons.Test(NpadMap[button]);

    return isHold;
}

bool SamplePad::IsTrigger(Button button) NN_NOEXCEPT
{
    bool isTrigger = false;

    isTrigger |= m_DebugPadCurrentState.buttons.Test(DebugPadMap[button])
        && !m_DebugPadPreviousState.buttons.Test(DebugPadMap[button]);
    isTrigger |= m_NpadFullKeyCurrentState.buttons.Test(NpadMap[button])
        && !m_NpadFullKeyPreviousState.buttons.Test(NpadMap[button]);
    isTrigger |= m_NpadHandheldCurrentState.buttons.Test(NpadMap[button])
        && !m_NpadHandheldPreviousState.buttons.Test(NpadMap[button]);

    return isTrigger;
}

void SamplePad::ResetCurrentButtons() NN_NOEXCEPT
{
    m_DebugPadCurrentState.buttons.Reset();
    m_NpadFullKeyCurrentState.buttons.Reset();
    m_NpadHandheldCurrentState.buttons.Reset();
}

void SamplePad::CopyStateCurrentToPrevious() NN_NOEXCEPT
{
    m_DebugPadPreviousState = m_DebugPadCurrentState;
    m_NpadFullKeyPreviousState = m_NpadFullKeyCurrentState;
    m_NpadHandheldPreviousState = m_NpadHandheldCurrentState;
}
