﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UtilConvertSjis.cpp,PageSampleUtilConvertSjis}
 *
 * @brief
 *  Shift_JIS(コードページ 932) と Unicode の相互変換のサンプルプログラム
 */

/**
 * @page PageSampleUtilConvertSjis 文字コード変換
 * @tableofcontents
 *
 * @brief
 *  文字コード変換のサンプルプログラムの解説です。
 *
 * @section PageSampleUtilConvertSjis_SectionBrief 概要
 *  ここでは、Shift_JIS と Unicode の各符号化方式の相互変換のサンプルプログラムの説明をします。
 *  文字コード変換ライブラリの使い方については、
 *  @ref nn::util "ユーティリティライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleUtilConvertSjis_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/UtilConvertSjis Samples/Sources/Applications/UtilConvertSjis @endlink 以下にあります。
 *
 * @section PageSampleUtilConvertSjis_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleUtilConvertSjis_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleUtilConvertSjis_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleUtilConvertSjis_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleUtilConvertSjis_SectionDetail 解説
 *
 * @subsection PageSampleUtilConvertSjis_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  SjisEncoding.cpp
 *  @includelineno SjisEncoding.cpp
 *
 *  SjisEncoding.h
 *  @includelineno SjisEncoding.h
 *
 *  UtilConvertSjis.cpp
 *  @includelineno UtilConvertSjis.cpp
 *
 * 上記ソースコードの他に SjisToUtf16Table.h と Utf16ToSjisTable.h がサンプルに含まれますが、これらのヘッダファイルには文字コードの変換で用いる配列が定義されているのみで行数も比較的多いので引用を省略します。
 *
 * @subsection PageSampleUtilConvertSjis_SectionSampleDetail サンプルプログラムの解説
 *
 * Shift_JIS 文字列と UTF-16 文字列や UTF-8 文字列を相互変換します。
 * 本サンプルでは厳密な Shift_JIS ではなく、 Shift_JIS を拡張した Windows コードページ 932 で扱われている符号化文字集合を変換可能としています。
 * 特に Shift_JIS と UTF-16 の変換においては、変換が成功する場合 Windows の WideCharToMultiByte() 関数と MultiByteToWideChar() 関数と同等の変換結果が得られるようになっています。
 * 変換失敗時の挙動についてはこの限りではありません。
 *
 * Shift_JIS と UTF-16 の変換においては変換テーブルを持ちそれを利用し変換を行っています。
 * Shift_JIS と UTF-8 の変換においては間に UTF-16 と UTF-8 の変換を挟むことで実現しています。
 * UTF-16 と UTF-8 の変換には nn::util::ConvertStringUtf16NativeToUtf8() 関数と nn::util::ConvertStringUtf8ToUtf16Native() 関数を利用しています。
 *
 * このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  UtilConvertSjis_OutputExample.txt
 */

#include <stdint.h>
#include <cstdio>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_CharacterEncoding.h>

#include "SjisEncoding.h"

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    // あいうえお (Shift_JIS)
    unsigned char pSjisSrc[11] = { 0x82u, 0xA0u,
                                   0x82u, 0xA2u,
                                   0x82u, 0xA4u,
                                   0x82u, 0xA6u,
                                   0x82u, 0xA8u,
                                   '\0'};
    // あいうえお (UTF-8)
    unsigned char pUtf8Src[16] = { 0xE3u, 0x81u, 0x82u,
                                   0xE3u, 0x81u, 0x84u,
                                   0xE3u, 0x81u, 0x86u,
                                   0xE3u, 0x81u, 0x88u,
                                   0xE3u, 0x81u, 0x8Au,
                                   '\0' };
    // あいうえお (UTF-16)
    uint16_t pUtf16Src[6] = { 0x3042,
                              0x3044,
                              0x3046,
                              0x3048,
                              0x304A,
                              '\0' };

    unsigned char pSjisDst[11];
    unsigned char pUtf8Dst[16];
    uint16_t pUtf16Dst[6];

    // Shift_JIS から UTF-16 への変換
    NN_LOG("--- Shift_JIS to UTF-16 ---\nBefore: ");
    for (int i = 0; i < 5; ++i)
    {
        NN_LOG("0x%02X%02X%s", pSjisSrc[i * 2], pSjisSrc[i * 2 + 1], (i == 4) ? "\nAfter:  " : " ");
    }
    auto retVal = ConvertStringSjisToUtf16(pUtf16Dst, 6, reinterpret_cast<char*>(pSjisSrc));
    NN_ASSERT(retVal == nn::util::CharacterEncodingResult_Success, "Convert failed.");
    for (int i = 0; i < 5; ++i)
    {
        NN_LOG("0x%04X%c", pUtf16Dst[i], (i == 4) ? '\n' : ' ');
    }

    // UTF-16 から Shift_JIS への変換
    NN_LOG("--- UTF-16 to Shift_JIS ---\nBefore: ");
    for (int i = 0; i < 5; ++i)
    {
        NN_LOG("0x%04X%s", pUtf16Src[i], (i == 4) ? "\nAfter:  " : " ");
    }
    retVal = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pSjisDst), 11, pUtf16Src);
    NN_ASSERT(retVal == nn::util::CharacterEncodingResult_Success, "Convert failed.");
    for (int i = 0; i < 5; ++i)
    {
        NN_LOG("0x%02X%02X%c", pSjisDst[i * 2], pSjisDst[i * 2 + 1], (i == 4) ? '\n' : ' ');
    }

    // Shift_JIS から UTF-8 への変換
    NN_LOG("--- Shift_JIS to UTF-8 ---\nBefore: ");
    for (int i = 0; i < 5; ++i)
    {
        NN_LOG("0x%02X%02X%s", pSjisSrc[i * 2], pSjisSrc[i * 2 + 1], (i == 4) ? "\nAfter:  " : " ");
    }
    retVal = ConvertStringSjisToUtf16(pUtf16Dst, 6, reinterpret_cast<char*>(pSjisSrc));
    NN_ASSERT(retVal == nn::util::CharacterEncodingResult_Success, "Convert failed.");
    pUtf16Dst[5] = '\0';
    retVal = nn::util::ConvertStringUtf16NativeToUtf8(reinterpret_cast<char*>(pUtf8Dst), 16, pUtf16Dst);
    NN_ASSERT(retVal == nn::util::CharacterEncodingResult_Success, "Convert failed.");
    for (int i = 0; i < 5; ++i)
    {
        NN_LOG("0x%02X%02X%02X%c", pUtf8Dst[i * 2], pUtf8Dst[i * 2 + 1], pUtf8Dst[i * 2 + 2], (i == 4) ? '\n' : ' ');
    }

    // UTF-8 から Shift_JIS への変換
    NN_LOG("--- UTF-8 to Shift_JIS ---\nBefore: ");
    for (int i = 0; i < 5; ++i)
    {
        NN_LOG("0x%02X%02X%02X%s", pUtf8Src[i * 2], pUtf8Src[i * 2 + 1], pUtf8Src[i * 2 + 2], (i == 4) ? "\nAfter:  " : " ");
    }
    retVal = nn::util::ConvertStringUtf8ToUtf16Native(pUtf16Dst, 6, reinterpret_cast<char*>(pUtf8Src));
    NN_ASSERT(retVal == nn::util::CharacterEncodingResult_Success, "Convert failed.");
    pUtf16Dst[5] = '\0';
    retVal = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pSjisDst), 10, pUtf16Dst);
    NN_ASSERT(retVal == nn::util::CharacterEncodingResult_Success, "Convert failed.");
    for (int i = 0; i < 5; ++i)
    {
        NN_LOG("0x%02X%02X%c", pSjisDst[i * 2], pSjisDst[i * 2 + 1], (i == 4) ? '\n' : ' ');
    }
}
