﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "UserMigration_ClientSequence.h"

#include <cstring>

#include <nn/account/account_Api.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_SelectorForSystemServices.h>
#include <nn/migration/migration_UserMigrationApi.h>
#include <nn/migration/migration_Result.h>
#include <nn/os/os_Tick.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/util/util_Optional.h>

#include "UserMigration.h"

namespace {
class SimpleClientAsyncSceneBase
    : public ClientSequence::SceneBase
{
protected:
    typedef ClientSequence::SceneBase Base;

private:
    nn::migration::UserMigrationClient& m_Client;
    nn::migration::AsyncContext(nn::migration::UserMigrationClient::*m_AsyncFunction)();
    char m_Label[128];
    bool m_IsSignaled;

    static void SignalHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SimpleClientAsyncSceneBase*>(data);
        obj.m_IsSignaled = true;
    }

    static bool HandleResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Client] HandleResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SimpleClientAsyncSceneBase*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        auto& components = *obj.GetComponentsPtr();
        components.ClearOption();
        components.AddOption("OK", obj.GetEndHandler());
        return false;
    }

protected:
    virtual void (*GetEndHandler() NN_NOEXCEPT) (void*,size_t)  = 0;

public:
    SimpleClientAsyncSceneBase(
        const char* label,
        nn::migration::UserMigrationClient& client,
        nn::migration::AsyncContext (nn::migration::UserMigrationClient::*asyncFunction)(),
        SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : Base(dialogCaller)
        , m_Client(client)
        , m_AsyncFunction(asyncFunction)
        , m_IsSignaled(false)
    {
        std::strncpy(m_Label, label, sizeof(m_Label));

        auto& components = *GetComponentsPtr();
        components.Reset(m_Label, nullptr, this);
        components.AddOption("Start", SignalHandler);
    }
    virtual void Update() NN_NOEXCEPT final NN_OVERRIDE
    {
        if (!m_IsSignaled)
        {
            return;
        }
        m_IsSignaled = false;

        NNS_MIGRATION_LOG_LN("[Client] Start async function");
        SequenceBase::AsyncTask async = {(m_Client.*m_AsyncFunction)(), HandleResult, this, sizeof(*this)};
        PushAsyncProgress("Processing", std::move(async));
    }
};

class SceneStart final
    : public ClientSequence::SceneBase
{
private:
    typedef ClientSequence::SceneBase Base;
public:
    SceneStart() NN_NOEXCEPT
    {
        GetComponentsPtr()->Reset("Starting client", nullptr);
        EndScene(Base::SceneKind::CreateClient);
    }
};
class SceneEnd final
    : public ClientSequence::SceneBase
{
private:
    ClientSequence& m_Parent;

    static void Handler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneEnd*>(data);
        obj.m_Parent.RequestClose();
        obj.GetComponentsPtr()->ClearOption();
    }

public:
    explicit SceneEnd(ClientSequence& parent) NN_NOEXCEPT
        : m_Parent(parent)
    {
        auto& components = *GetComponentsPtr();
        components.Reset("Client sequence ended", nullptr, this);
        components.AddOption("Dispose client object", Handler);
    }
};

class SceneCreateClient final
    : public ClientSequence::SceneBase
{
private:
    typedef ClientSequence::SceneBase Base;
    nn::migration::UserMigrationClient& m_Client;
    void* m_Buffer;
    size_t m_BUfferSize;
    bool m_IsNewClient;
    bool m_IsSignaled;
    bool m_IsSkipping;
    bool m_IsAborting;

    static void SignalHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneCreateClient*>(data);
        obj.m_IsSignaled = true;
    }

    static void SignalWithSkipHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneCreateClient*>(data);
        obj.m_IsSkipping = true;
    }

    static void AbortHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneCreateClient*>(data);
        obj.m_IsAborting = true;
    }

public:
    SceneCreateClient(nn::migration::UserMigrationClient* pClient, void* buffer, size_t bufferSize, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : Base(dialogCaller)
        , m_Client(*pClient)
        , m_Buffer(buffer)
        , m_BUfferSize(bufferSize)
        , m_IsSignaled(false)
        , m_IsSkipping(false)
        , m_IsAborting(false)
    {
        auto state = nn::migration::GetLastUserMigrationState();
        NN_ABORT_UNLESS(false
            || state == nn::migration::LastUserMigrationState::Unresumable
            || state == nn::migration::LastUserMigrationState::ClientResumable
            || state == nn::migration::LastUserMigrationState::ClientResumeSuggested);
        m_IsNewClient = (state == nn::migration::LastUserMigrationState::Unresumable);

        if (m_IsNewClient)
        {
            auto& components = *GetComponentsPtr();
            components.Reset("Client will be newly created", nullptr, this);
            components.AddOption("Create new client", SignalHandler);
        }
        else
        {
            auto& components = *GetComponentsPtr();
            components.Reset(
                "Suspended client is found",
                state == nn::migration::LastUserMigrationState::ClientResumeSuggested
                    ? "state: LastUserMigrationState::ClientResumeSuggested"
                    : "state: LastUserMigrationState::ClientResumable",
                this);
            components.AddOption("Resume client", SignalHandler);
            components.AddOption("Resume client (skip login)", SignalWithSkipHandler);
            components.AddOption("Abort ongoing immigration", AbortHandler);
        }
    }
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        if (!m_IsSignaled)
        {
            return;
        }
        m_IsSignaled = false;

        if (m_IsAborting)
        {
            NNS_MIGRATION_LOG_LN("[Client] Aborting client");
            NN_SDK_ASSERT(!m_IsNewClient);
            auto r = nn::migration::CleanupUserMigration(m_Buffer, m_BUfferSize);
            if (!r.IsSuccess())
            {
                char str[128];
                nn::util::SNPrintf(str, sizeof(str), "nn::migration::CleanupUserMigration() failed with %03d-%04d(0x%08lx)\n", r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
                PushError(str);
                return;
            }
            PushError("This is not an error, but ongoing immigration is aborted.");
            return;
        }

        if (m_IsNewClient)
        {
            NNS_MIGRATION_LOG_LN("[Client] Creating client");
            nn::settings::system::SerialNumber sn;
            nn::settings::system::GetSerialNumber(&sn);
            nn::migration::UserMigrationClientProfile profile = {};
            std::strncpy(reinterpret_cast<char*>(profile.data), sn.string, sizeof(profile.data));
            m_Client = nn::migration::CreateUserMigrationClient(profile, m_Buffer, m_BUfferSize);
        }
        else
        {
            NNS_MIGRATION_LOG_LN("[Client] Resumimg client");
            m_Client = nn::migration::ResumeUserMigrationClient(m_Buffer, m_BUfferSize);
        }

        bool skipping = m_IsSkipping;
        NNS_MIGRATION_LOG_LN("[Client] Skip login: %s\n", skipping ? "true" : "false");
        m_IsSkipping = false;
        if (!skipping)
        {
            NNS_MIGRATION_LOG_LN("[Client] Opening psel");
            auto sessionId = m_Client.CreateLoginSession();
            auto r = nn::account::ShowUiToIntroduceExternalNetworkServiceAccountForRegistration(sessionId);
            if (!r.IsSuccess())
            {
                if (nn::account::ResultCancelledByUser::Includes(r))
                {
                    PushError("User selection is canceled.");
                    return;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(r);
            }
        }

        auto nsaId = m_Client.GetNetworkServiceAccountId();
        nn::account::Nickname nickname;
        m_Client.GetUserNickname(&nickname);
        char description[128];
        nn::util::SNPrintf(description, sizeof(description), "\"%s\"(%08lx)", nickname.name, nsaId.id);

        NNS_MIGRATION_LOG_LN("[Client] User: %s", description);

        auto& components = *GetComponentsPtr();
        components.Reset("User to immigrate", description, this);
        components.AddOption("OK", Base::EndHandler<decltype(*this), Base::SceneKind::Prepare>);
    }
};

class ScenePrepare final
    : public SimpleClientAsyncSceneBase
{
private:
    nn::migration::UserMigrationClient& m_Client;
protected:
    virtual void (*GetEndHandler() NN_NOEXCEPT) (void* data, size_t dataSize)
    {
        if (m_Client.IsConnectionRequired())
        {
            return &Base::EndHandler<decltype(*this), Base::SceneKind::Connect>;
        }
        else
        {
            return &Base::EndHandler<decltype(*this), Base::SceneKind::Complete>;
        }
    }

public:
    ScenePrepare(nn::migration::UserMigrationClient& client, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : SimpleClientAsyncSceneBase("Prepare migration", client, &nn::migration::UserMigrationClient::PrepareAsync, dialogCaller)
        , m_Client(client)
    {
    }
};

class SceneConnect final
    : public ClientSequence::SceneBase
{
private:
    typedef ClientSequence::SceneBase Base;
    nn::migration::UserMigrationClient& m_Client;
    bool m_IsSignaled;
    bool m_IsAborting;
    bool m_IsScanStarted;
    bool m_IsSelected;

    nn::migration::UserMigrationServerInfo m_Servers[4];
    nn::migration::UserMigrationServerInfo m_SelectedServer;

    static void SignalHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneConnect*>(data);
        obj.m_IsSignaled = true;
    }
    static void AbortHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneConnect*>(data);
        obj.m_IsAborting = true;
    }
    static void ScanStartHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneConnect*>(data);
        obj.m_IsScanStarted = true;
    }
    static void SelectHandler(int index, void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneConnect*>(data);
        obj.m_IsSelected = true;
        obj.m_SelectedServer = obj.m_Servers[index];
    }

    static bool HandleScanResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Client] HandleScanResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneConnect*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        auto& components = *obj.GetComponentsPtr();
        components.Reset("Select server to connect", "Scan result:", &obj);
        auto count = obj.m_Client.ListServers(obj.m_Servers, std::extent<decltype(obj.m_Servers)>::value);
        for (size_t i = 0; i < count; ++ i)
        {
            // NOTE ここで得られるサーバー情報はセキュアではないので内容のチェックをするべき
            const auto& server = obj.m_Servers[i];
            char label[128];
            nn::util::SNPrintf(label, sizeof(label), "Server[%d] Device \"%s\"", i, server.profile.data);

            NNS_MIGRATION_LOG_LN("[Client] %s", label);
            components.AddOption(label, SelectHandler);
        }
        components.AddOption("Rescan", ScanStartHandler);
        return false;
    }

    static bool HandleConnectResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Client] HandleConnectResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneConnect*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        auto shortfall = obj.m_Client.GetStorageShortfall();
        if (shortfall != 0)
        {
            char message[128];
            nn::util::SNPrintf(message, sizeof(message), "Insufficient storage: shortfall=%lld", shortfall);
            obj.GetComponentsPtr()->ClearOption();
            obj.PushError(message);
            return true;
        }

        // NOTE ここで得られるサーバー情報はセキュアではないので内容のチェックをするべき
        char title[128];
        nn::util::SNPrintf(title, sizeof(title), "Connected with \"%s\"", obj.m_SelectedServer.profile.data);

        auto uid = obj.m_Client.GetImmigrantUid();
        char description[128];
        nn::util::SNPrintf(description, sizeof(description), "Uid: %016llx_%016llx", uid._data[0], uid._data[1]);

        auto& components = *obj.GetComponentsPtr();
        components.Reset(title, description, &obj);
        components.AddOption("OK", Base::EndHandler<decltype(obj), Base::SceneKind::Transfer>);
        return false;
    }

public:
    SceneConnect(nn::migration::UserMigrationClient& client, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : Base(dialogCaller)
        , m_Client(client)
        , m_IsSignaled(false)
        , m_IsAborting(false)
        , m_IsScanStarted(false)
        , m_IsSelected(false)
    {
        auto& components = *GetComponentsPtr();
        components.Reset("Scan migration servers", nullptr, this);
        components.AddOption("Start", ScanStartHandler);
    }
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        if (!m_IsSignaled)
        {
            return;
        }
        m_IsSignaled = false;

        if (m_IsAborting)
        {
            PushError("Server selection is canceled.");
            return;
        }
        else if (m_IsScanStarted)
        {
            NNS_MIGRATION_LOG_LN("[Client] Start scanning servers");
            m_IsScanStarted = false;
            SequenceBase::AsyncTask async = {m_Client.ScanServersAsync(), HandleScanResult, this, sizeof(*this)};
            PushAsyncProgress("Scanning", std::move(async));
            return;
        }
        else  if (m_IsSelected)
        {
            NNS_MIGRATION_LOG_LN("[Client] Start connecting to server device \"%s\"", m_SelectedServer.profile.data);
            m_IsSelected = false;
            SequenceBase::AsyncTask async = {m_Client.ConnectAsync(m_SelectedServer.id), HandleConnectResult, this, sizeof(*this)};

            PushAsyncProgress("Connecting", std::move(async));
            return;
        }
    }
};

class SceneTransfer final
    : public ClientSequence::SceneBase
{
private:
    typedef ClientSequence::SceneBase Base;
    nn::migration::UserMigrationClient& m_Client;
    bool m_IsSignaled;
    bool m_IsSuspending;
    bool m_IsTransferStarted;
    bool m_IsTransferAllStarted;

    nn::migration::TransferInfo m_Total;
    std::atomic<nn::migration::TransferInfo> m_Current;

    static void SignalHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneTransfer*>(data);
        obj.m_IsSignaled = true;
    }
    static void SuspendHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneTransfer*>(data);
        obj.m_IsSuspending = true;
    }
    static void TransferHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneTransfer*>(data);
        obj.m_IsTransferStarted = true;
    }
    static void TransferAllHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneTransfer*>(data);
        obj.m_IsTransferAllStarted = true;
    }

    static void UpdateComponents(SceneTransfer& obj) NN_NOEXCEPT
    {
        auto current = obj.m_Current.load();

        char description[128];
        nn::util::SNPrintf(
            description, sizeof(description), "Progress: %u of %u (%llu of %llu bytes)",
            current.count, obj.m_Total.count, current.sizeInBytes, obj.m_Total.sizeInBytes);

        auto& components = *obj.GetComponentsPtr();
        if (current.count < obj.m_Total.count)
        {
            components.Reset("Transfer next?", description, &obj);
            components.AddOption("Transfer next", TransferHandler);
            components.AddOption("Transfer all", TransferAllHandler);
            components.AddOption("Skip all", Base::EndHandler<decltype(obj), Base::SceneKind::Complete>);
        }
        else
        {
            components.Reset("Transfer done", description, &obj);
            components.AddOption("Complete transfer", Base::EndHandler<decltype(obj), Base::SceneKind::Complete>);
        }
        components.AddOption("Suspend", SuspendHandler);
        components.AddOption("DEBUG COMMANDS", Base::EndHandler<decltype(obj), Base::SceneKind::Debug>);
    }

    static bool HandleTransferResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Client] HandleTransferResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneTransfer*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        obj.m_Current.store(obj.m_Client.GetCurrentTransferInfo());
        UpdateComponents(obj);
        return false;
    }
    static bool HandleSuspendResult(void* data, size_t dataSize, nn::Result r)  NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Client] HandleSuspendResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneTransfer*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        auto& components = *obj.GetComponentsPtr();
        components.ClearOption();
        obj.PushError("Transfer is canceled.");
        return true;
    }

public:
    SceneTransfer(nn::migration::UserMigrationClient& client, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : Base(dialogCaller)
        , m_Client(client)
        , m_IsSignaled(false)
        , m_IsSuspending(false)
        , m_IsTransferStarted(false)
        , m_IsTransferAllStarted(false)
    {
        auto shortfall = m_Client.GetStorageShortfall();
        if (shortfall > 0)
        {
            PushError("Storage capacity not enough");
            return;
        }

        m_Total = m_Client.GetTotalTransferInfo();
        m_Current.store(m_Client.GetCurrentTransferInfo());
        UpdateComponents(*this);
    }
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_FUNCTION_LOCAL_STATIC(nn::util::optional<nn::TimeSpan>, s_LastPrinted);
        if (!(s_LastPrinted && (nn::os::GetSystemTick().ToTimeSpan() - *s_LastPrinted).GetMilliSeconds() < 500))
        {
            auto current0 = m_Current.load();
            auto current = m_Client.GetCurrentTransferInfo();
            if (!(current.count == current0.count && current.sizeInBytes == current0.sizeInBytes))
            {
                s_LastPrinted = nn::os::GetSystemTick().ToTimeSpan();
                m_Current.store(current);
                NNS_MIGRATION_LOG_LN("[Client] Progress: %u of %u (%llu of %llu bytes)",
                    current.count, m_Total.count, current.sizeInBytes, m_Total.sizeInBytes);
            }
        }

        if (!m_IsSignaled)
        {
            return;
        }
        m_IsSignaled = false;

        if (m_IsSuspending)
        {
            m_IsSuspending = false;

            NNS_MIGRATION_LOG_LN("[Client] Client will be suspended");
            SequenceBase::AsyncTask async = {m_Client.SuspendAsync(), HandleSuspendResult, this, sizeof(*this)};
            PushAsyncProgress("Suspending", std::move(async));
            return;
        }
        else if (m_IsTransferStarted)
        {
            m_IsTransferStarted = false;

            auto current = m_Current.load();
            NNS_MIGRATION_LOG_LN(
                "[Client] Transfer next: Progress: %u of %u (%llu of %llu bytes)",
                current.count, m_Total.count, current.sizeInBytes, m_Total.sizeInBytes);
            SequenceBase::AsyncTask async = {m_Client.TransferNextAsync(), HandleTransferResult, this, sizeof(*this)};
            PushAsyncProgress("Transferring", std::move(async));
            return;
        }
        else if (m_IsTransferAllStarted)
        {
            m_IsTransferAllStarted = false;

            auto f = [this](void) -> nn::Result
            {
                auto current = m_Client.GetCurrentTransferInfo();
                while (current.count < m_Total.count)
                {
                    auto async = m_Client.TransferNextAsync();
                    nn::os::SystemEvent e;
                    async.GetSystemEvent(&e);
                    e.Wait();
                    NN_RESULT_DO(async.GetResult());

                    current = m_Client.GetCurrentTransferInfo();
                }

                m_Current.store(current);
                NNS_MIGRATION_LOG_LN("[Client] Progress: %u of %u (%llu of %llu bytes)",
                    current.count, m_Total.count, current.sizeInBytes, m_Total.sizeInBytes);
                NN_RESULT_SUCCESS;
            };

            SequenceBase::SyncTask sync = {std::move(f), HandleTransferResult, this, sizeof(*this)};
            PushSyncProgress("Transferring", std::move(sync));
            return;
        }
    }
};

class SceneComplete final
    : public SimpleClientAsyncSceneBase
{
protected:
    virtual void (*GetEndHandler() NN_NOEXCEPT) (void* data, size_t dataSize)
    {
        return Base::EndHandler<decltype(*this), Base::SceneKind::End>;
    }
public:
    SceneComplete(nn::migration::UserMigrationClient& client, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : SimpleClientAsyncSceneBase("Finalizing immigration", client, &nn::migration::UserMigrationClient::CompleteAsync, dialogCaller)
    {
        static nn::ncm::ApplicationId appIds[1024];
        auto actualSize = client.GetCurrentRelatedApplications(appIds, std::extent<decltype(appIds)>::value);

        NNS_MIGRATION_LOG_LN("[Client] Related Apps");
        for (size_t i = 0; i < actualSize; ++ i)
        {
            NNS_MIGRATION_LOG_LN("  #%d : %016llx", i, appIds[i].value);
        }
    }
};

class SceneDebug final
    : public ClientSequence::SceneBase
{
protected:
    typedef ClientSequence::SceneBase Base;
    nn::migration::UserMigrationClient& m_Client;

    bool m_IsSignaled;
    int m_Selected;

    static void SelectionHandler(int index, void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneDebug*>(data);

        obj.m_IsSignaled = true;
        obj.m_Selected = index;
    }
    static bool HandleAsyncResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        return reinterpret_cast<SceneDebug*>(data)->Base::HandleResult(r);
    }

public:
    SceneDebug(nn::migration::UserMigrationClient& client, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : Base(dialogCaller)
        , m_Client(client)
        , m_IsSignaled(false)
    {

        auto& components = *GetComponentsPtr();
        components.Reset("DEBUG COMMANDS", nullptr, this);
        components.AddOption("Send \"sync:InFinalization\"", SelectionHandler);
        components.AddOption("Send \"suspend\"", SelectionHandler);
        components.AddOption("End", Base::EndHandler<decltype(*this), Base::SceneKind::End>);
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        if (!m_IsSignaled)
        {
            return;
        }
        m_IsSignaled = false;

        switch (m_Selected)
        {
        case 0: // Send sync:InFinalization command
            {
                SequenceBase::AsyncTask async = {m_Client.DebugSynchronizeStateInFinalization(), HandleAsyncResult, this, sizeof(*this)};
                PushAsyncProgress("Waiting", std::move(async));
            }
            break;
        case 1: // Send suspend command
            {
                SequenceBase::AsyncTask async = {m_Client.SuspendAsync(), HandleAsyncResult, this, sizeof(*this)};
                PushAsyncProgress("Waiting", std::move(async));
            }
            break;
        case 2: // End
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
};

Mutex g_BufferLock = {NN_OS_MUTEX_INITIALIZER(false)};
std::aligned_storage<nn::migration::RequiredWorkBufferSizeForUserMigrationOperation, nn::os::MemoryPageSize>::type g_Buffer;

} // ~namespace <anonymous>

ClientSequence::ClientSequence(Window& window) NN_NOEXCEPT
    : SequenceBase(window)
    , m_CurrentScene(SceneKind::Start)
    , m_NextScene(SceneKind::Start)
    , m_IsLockAcquired(false)
{
    m_Scene.reset(new SceneStart());
    NotifyCurrentSceneUpdate();

    NNS_MIGRATION_LOG_LN("[Server] sequense created");
}
ClientSequence::~ClientSequence() NN_NOEXCEPT
{
    if (m_IsLockAcquired)
    {
        m_Client = decltype(m_Client)();
        g_BufferLock.Unlock();
    }

}
void ClientSequence::OnEntering(SceneKind next) NN_NOEXCEPT
{
    switch (next)
    {
    case SceneKind::Start:
        NN_ABORT("");

    case SceneKind::End:
        m_Scene.reset(new SceneEnd(*this));
        NNS_MIGRATION_LOG_LN("[Client] Scene entered: End");
        break;

    case SceneKind::CreateClient:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Start);
        {
            NN_ABORT_UNLESS(g_BufferLock.TryLock());
            m_IsLockAcquired = true;
            m_Scene.reset(new SceneCreateClient(&m_Client, &g_Buffer, sizeof(g_Buffer), GetDialogCallerRef()));
        }
        NNS_MIGRATION_LOG_LN("[Client] Scene entered: CreateServer");
        break;

    case SceneKind::Prepare:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::CreateClient);
        m_Scene.reset(new ScenePrepare(m_Client, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Client] Scene entered: Prepare");
        break;

    case SceneKind::Connect:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Prepare);
        m_Scene.reset(new SceneConnect(m_Client, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Client] Scene entered: Connect");
        break;

    case SceneKind::Transfer:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Connect || m_CurrentScene == SceneKind::Transfer);
        m_Scene.reset(new SceneTransfer(m_Client, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Client] Scene entered: Transfer");
        break;

    case SceneKind::Complete:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Prepare || m_CurrentScene == SceneKind::Transfer);
        m_Scene.reset(new SceneComplete(m_Client, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Client] Scene entered: Complete");
        break;

    case SceneKind::Debug:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Prepare || m_CurrentScene == SceneKind::Transfer);
        m_Scene.reset(new SceneDebug(m_Client, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Client] Scene entered: Debug");
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    m_CurrentScene = next;
}
const Window::Components* ClientSequence::GetCurrentSceneComponentsPtr() const NN_NOEXCEPT
{
    return static_cast<const SceneBase*>(m_Scene.get())->GetComponentsPtr();
}
void ClientSequence::UpdateImpl() NN_NOEXCEPT
{
    if (IsErrorOccurred())
    {
        if (m_CurrentScene != SceneKind::End)
        {
            NNS_MIGRATION_LOG_LN("[Client] ClientSequence detects error");
        }
        m_NextScene = SceneKind::End;
    }

    if (m_NextScene != m_CurrentScene)
    {
        OnEntering(m_NextScene);
        NotifyCurrentSceneUpdate();
    }

    m_Scene->Update();
    if (m_Scene->IsEnded())
    {
        m_NextScene = m_Scene->GetNextScene();
    }
}
