﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "UserMigration.h"
#include "UserMigration_ClientSequence.h"
#include "UserMigration_ServerSequence.h"

#include <nn/migration/migration_UserMigrationApi.h>
#include <nn/settings/system/settings_SystemApplication.h>

namespace {
class SceneStart final
    : public RootSequence::SceneBase
{
private:
    typedef RootSequence::SceneBase Base;

    static void SelectionHandler(int index, void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);

        auto& obj = *reinterpret_cast<SceneStart*>(data);
        switch (index)
        {
        case 0:
            obj.EndScene(Base::SceneKind::ServerSequence);
            break;
        case 1:
            obj.EndScene(Base::SceneKind::ClientSequence);
            break;
        case 2:
            {
                nn::settings::system::AppletLaunchFlagSet value;
                nn::settings::system::GetAppletLaunchFlags(&value);
                value.Flip<nn::settings::system::AppletLaunchFlag::Migration>();
                nn::settings::system::SetAppletLaunchFlags(value);
                obj.Reset();
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void Reset() NN_NOEXCEPT
    {
        auto& components = *GetComponentsPtr();

        auto state = nn::migration::GetLastUserMigrationState();
        switch (state)
        {
        case nn::migration::LastUserMigrationState::Unresumable:
            {
                components.Reset("Starting user migration", nullptr, this);
                components.AddOption("Start as server", SelectionHandler);
                components.AddOption("Start as client", SelectionHandler);

                nn::settings::system::AppletLaunchFlagSet value;
                nn::settings::system::GetAppletLaunchFlags(&value);
                char settingsStr[64];
                nn::util::SNPrintf(
                    settingsStr, sizeof(settingsStr),
                    "Migration launch flag: %s",
                    value.Test<nn::settings::system::AppletLaunchFlag::Migration>() ? "Enabled" : "Disabled");
                components.AddOption(settingsStr, SelectionHandler);
            }
            break;
        case nn::migration::LastUserMigrationState::ServerResumable:
        case nn::migration::LastUserMigrationState::ServerResumeSuggested:
            components.Reset("Moving to server subsequence", nullptr);
            EndScene(Base::SceneKind::ServerSequence);
            break;
        case nn::migration::LastUserMigrationState::ClientResumable:
        case nn::migration::LastUserMigrationState::ClientResumeSuggested:
            components.Reset("Moving to client subsequence", nullptr);
            EndScene(Base::SceneKind::ClientSequence);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

public:
    SceneStart() NN_NOEXCEPT
    {
        Reset();
    }
};
} // ~namespace <anonymous>

RootSequence::RootSequence(Window& window) NN_NOEXCEPT
    : SequenceBase(window)
    , m_Window(window)
    , m_CurrentScene(SceneKind::Start)
    , m_NextScene(SceneKind::Start)
{
    m_Scene.reset(new SceneStart());
    NotifyCurrentSceneUpdate();

    NNS_MIGRATION_LOG_LN("[Root] sequense created");
}
RootSequence::~RootSequence() NN_NOEXCEPT
{
}
void RootSequence::OnEntering(SceneKind next) NN_NOEXCEPT
{
    switch (next)
    {
    case SceneKind::Start:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::ClientSequence || m_CurrentScene == SceneKind::ServerSequence);
        m_Scene.reset(new SceneStart());
        NNS_MIGRATION_LOG_LN("[Root] Scene entered: Start");
        break;

    case SceneKind::ClientSequence:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Start);
        m_SubSequence.reset(new ClientSequence(m_Window));
        NNS_MIGRATION_LOG_LN("[Root] Scene entered: ClientSequence");
        break;

    case SceneKind::ServerSequence:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Start);
        m_SubSequence.reset(new ServerSequence(m_Window));
        NNS_MIGRATION_LOG_LN("[Root] Scene entered: ServerSequence");
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    m_CurrentScene = next;
}
const Window::Components* RootSequence::GetCurrentSceneComponentsPtr() const NN_NOEXCEPT
{
    return static_cast<const SceneBase*>(m_Scene.get())->GetComponentsPtr();
}
void RootSequence::UpdateImpl() NN_NOEXCEPT
{
    if (m_SubSequence)
    {
        m_SubSequence.reset();
        m_NextScene = SceneKind::Start;
        NNS_MIGRATION_LOG_LN("[Root] Exit subsequence");
    }

    if (m_NextScene != m_CurrentScene)
    {
        OnEntering(m_NextScene);
        NotifyCurrentSceneUpdate();
    }

    m_Scene->Update();
    if (m_Scene->IsEnded())
    {
        m_NextScene = m_Scene->GetNextScene();
    }
}
SequenceBase* RootSequence::GetSubSequence() NN_NOEXCEPT
{
    return m_SubSequence.get();
}
