﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d.h>

#include "Ui2dUserShader.h"

// ユーザーピクセルシェーダーで使用するコンスタントバッファ
struct UserShaderConstantBufferForPixelShader
{
    float   color[4];
    float   time;
    float   distortion;
};

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------

//  シェーダーのアニメーション制御のためのタイマー
static float    g_ShaderTimer = 0.0f;

//-----------------------------------------------------------------------------
//  ユーザーシェーダー情報取得コールバック
//-----------------------------------------------------------------------------
static bool GetUserShaderInformationFromUserDataCallback(
    nn::ui2d::UserShaderInformation& userShaderInformation,
    const nn::ui2d::ResExtUserDataList* pExtUserDataList,
    void* pUserData)
{
    NN_UNUSED(pUserData);

    bool    isValidShaderNameSet = false;

    // ペインに設定したユーザー拡張データからユーザーシェーダーの情報を取得して
    // userShaderInfomation に設定します。
    // 下記のユーザー拡張データの解釈はサンプル仕様のため
    // 名前などはアプリケーションが自由に決定できます。
    if (pExtUserDataList != NULL)
    {
        // ユーザーシェーダー名。
        // ArchiveShader-XXXXX.bnsh の「XXXXX」部分。
        // アーカイブシェーダーの仕様により長さは nn::ui2d::ArchiveShaderNameMax-1 文字分。
        const nn::ui2d::ResExtUserData* pShaderName = GetExtUserData(
            pExtUserDataList,
            "UserShaderName");
        if (pShaderName != NULL)
        {
            isValidShaderNameSet = userShaderInformation.SetShaderName(pShaderName->GetString());
        }

        // ユーザー頂点シェーダーで拡張したコンスタントバッファのサイズ
        const nn::ui2d::ResExtUserData* pVertexShaderConstantBufferSize
            = GetExtUserData(pExtUserDataList, "UserVertexShaderConstantBufferSize");

        if (pVertexShaderConstantBufferSize != NULL)
        {
            userShaderInformation.vertexShaderConstantBufferExtendSize
                = pVertexShaderConstantBufferSize->GetIntArray()[0];
        }

        // ユーザージオメトリシェーダーで拡張したコンスタントバッファのサイズ
        const nn::ui2d::ResExtUserData* pGeometryShaderConstantBufferSize
            = GetExtUserData(pExtUserDataList, "UserGeometryShaderConstantBufferSize");

        if (pGeometryShaderConstantBufferSize != NULL)
        {
            userShaderInformation.geometryShaderConstantBufferExtendSize
                = pGeometryShaderConstantBufferSize->GetIntArray()[0];
        }

        // ユーザーピクセルシェーダーで拡張したコンスタントバッファのサイズ
        const nn::ui2d::ResExtUserData* pPixelShaderConstantBufferSize
            = GetExtUserData(pExtUserDataList, "UserPixelShaderConstantBufferSize");
        if (pPixelShaderConstantBufferSize != NULL)
        {
            userShaderInformation.pixelShaderConstantBufferExtendSize
                = pPixelShaderConstantBufferSize->GetIntArray()[0];
        }
    }

    // 戻り値でユーザーシェーダーの使用可否が決定されます。
    return isValidShaderNameSet;
}

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeUserShader()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Ui2dUserShader.arc", nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの初期化
    g_pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    {
        void* pFont = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool    result = g_pFont->SetResource(pDevice, pFont);
        NN_ASSERT(result);
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

        g_pArcResourceAccessor->RegisterFont("sample.bffnt", g_pFont);
    }

    // レイアウトの初期化
    nn::ui2d::BuildResultInformation buildResult;
    {
        g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

        nn::ui2d::Layout::BuildOption    opt;
        opt.SetDefault();
        // ユーザーシェーダー情報を取得するコールバックを設定します。
        opt.pGetUserShaderInformationFromUserDataCallback = GetUserShaderInformationFromUserDataCallback;
        buildResult.SetDefault();

        g_pLayout->BuildWithName(&buildResult, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, "UserShader.bflyt");
    }

    // グラフィックスリソースの設定
    InitializeGraphicsResource();

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    g_pDrawInfo = AllocAndConstruct<nn::ui2d::DrawInfo>();
    NN_ASSERT_NOT_NULL(g_pDrawInfo);
    g_pUi2dConstantBuffer = AllocAndConstruct<nn::font::GpuBuffer>();
    NN_ASSERT_NOT_NULL(g_pUi2dConstantBuffer);
    InitializeUi2dBuffers(*g_pDrawInfo, buildResult, g_pUi2dConstantBuffer);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo->SetGraphicsResource(g_pGraphicsResource);
        g_pDrawInfo->SetProjectionMtx(projection);
        g_pDrawInfo->SetViewMtx(view);
    }

    nn::ui2d::Animator* pAnimator = g_pLayout->CreateGroupAnimator(pDevice, "ExtUserDataAnim");
    pAnimator->PlayAuto(1.0f);

    // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

    // ユーザーシェーダー更新コールバックの設定
    g_pDrawInfo->SetUpdateUserShaderCallback(UpdateUserShaderConstantBuffers, NULL);
}

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeUserShader()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pUi2dConstantBuffer->Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    DestructAndFree<nn::font::GpuBuffer>(g_pUi2dConstantBuffer);
    g_pUi2dConstantBuffer = NULL;
    DestructAndFree<nn::ui2d::DrawInfo>(g_pDrawInfo);
    g_pDrawInfo = NULL;

    FinalizeGraphicsResource();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);
    g_pLayout = NULL;

    {
        g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);

        void* pFontResource = g_pFont->RemoveResource(pDevice);
        Ui2dDeallocateFunction(pFontResource, NULL);
        g_pFont->Finalize(pDevice);
        DestructAndFree<nn::font::ResFont>(g_pFont);
        g_pFont = NULL;
    }

    {
        g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        g_pArcResourceAccessor->Detach();
        g_pArcResourceAccessor->Finalize(pDevice);

        Ui2dDeallocateFunction(g_pLayoutArchiveBinary, NULL);
        g_pLayoutArchiveBinary = NULL;

        DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);
        g_pArcResourceAccessor = NULL;
    }
}

//------------------------------------------------------------------------------
// ユーザーシェーダーのコンスタントバッファ更新処理
//------------------------------------------------------------------------------
void UpdateUserShaderConstantBuffers(const nn::ui2d::DrawInfo& drawInfo, nn::ui2d::Pane* pPane, void* pUserData)
{
    NN_UNUSED(pUserData);

    // ユーザーシェーダーを使用するように設定したペインからマテリアルを取得し
    // ユーザーシェーダーのコンスタントバッファに必要な情報を書き込む。
    // ユーザーシェーダーのコンスタントバッファは前フレームに書き込まれた値は保持されないため
    // すべてのパラメータを毎フレーム書き込む必要がある。

    if (std::strcmp(pPane->GetName(), "UserShader") == 0)
    {
        nn::ui2d::Material* pMaterial = pPane->GetMaterial(0);

        // ユーザーシェーダーで拡張したコンスタントバッファの領域へのポインタを取得して値を書き込む。
        UserShaderConstantBufferForPixelShader* pConstantBufferForUserPixelShader
            = static_cast<UserShaderConstantBufferForPixelShader*>(pMaterial->GetConstantBufferForUserPixelShader(drawInfo));
        if (pConstantBufferForUserPixelShader != NULL)
        {
            pConstantBufferForUserPixelShader->color[0] = 0.5f;
            pConstantBufferForUserPixelShader->color[1] = 0.75f;
            pConstantBufferForUserPixelShader->color[2] = 1.0f;
            pConstantBufferForUserPixelShader->color[3] = 1.0f;

            // ユーザー拡張データアニメーションから値を取得してコンスタントバッファへ設定する。
            const nn::ui2d::ResExtUserData*   pTimeUserData = pPane->FindExtUserDataByName("Time");
            pConstantBufferForUserPixelShader->time = pTimeUserData->GetFloatArray()[0];
            const nn::ui2d::ResExtUserData*   pDistortionUserData = pPane->FindExtUserDataByName("Distortion");
            pConstantBufferForUserPixelShader->distortion = pDistortionUserData->GetFloatArray()[0];
        }
    }

    if (std::strcmp(pPane->GetName(), "picture_vi") == 0)
    {
        nn::ui2d::Material* pMaterial = pPane->GetMaterial(0);

        // ユーザーシェーダーで拡張したコンスタントバッファの領域へのポインタを取得して値を書き込む。
        UserShaderConstantBufferForPixelShader* pConstantBufferForUserPixelShader
            = static_cast<UserShaderConstantBufferForPixelShader*>(pMaterial->GetConstantBufferForUserPixelShader(drawInfo));
        if (pConstantBufferForUserPixelShader != NULL)
        {
            pConstantBufferForUserPixelShader->color[0] = 0.5f;
            pConstantBufferForUserPixelShader->color[1] = 0.75f;
            pConstantBufferForUserPixelShader->color[2] = 1.0f;
            pConstantBufferForUserPixelShader->color[3] = 1.0f;

            // ユーザー拡張データアニメーションから値を取得してコンスタントバッファへ設定する。
            const nn::ui2d::ResExtUserData*   pTimeUserData = pPane->FindExtUserDataByName("Time");
            pConstantBufferForUserPixelShader->time = pTimeUserData->GetFloatArray()[0];
            const nn::ui2d::ResExtUserData*   pDistortionUserData = pPane->FindExtUserDataByName("Distortion");
            pConstantBufferForUserPixelShader->distortion = pDistortionUserData->GetFloatArray()[0];
        }
    }

    if (std::strcmp(pPane->GetName(), "UserGeometryShader") == 0)
    {
        nn::ui2d::Material* pMaterial = pPane->GetMaterial(0);

        // ユーザーシェーダーで拡張したコンスタントバッファの領域へのポインタを取得して値を書き込む。
        UserShaderConstantBufferForPixelShader* pConstantBufferForUserPixelShader
            = static_cast<UserShaderConstantBufferForPixelShader*>(pMaterial->GetConstantBufferForUserPixelShader(drawInfo));
        if (pConstantBufferForUserPixelShader != NULL)
        {
            pConstantBufferForUserPixelShader->color[0] = 0.5f;
            pConstantBufferForUserPixelShader->color[1] = 0.75f;
            pConstantBufferForUserPixelShader->color[2] = 0.0f;
            pConstantBufferForUserPixelShader->color[3] = 1.0f;

            // ユーザー拡張データアニメーションから値を取得してコンスタントバッファへ設定する。
            const nn::ui2d::ResExtUserData*   pTimeUserData = pPane->FindExtUserDataByName("Time");
            pConstantBufferForUserPixelShader->time = pTimeUserData->GetFloatArray()[0];
            pConstantBufferForUserPixelShader->distortion = 0.0f;
        }

        float*  pConstantBufferForUserGeometryShader
            = static_cast<float*>(pMaterial->GetConstantBufferForUserGeometryShader(drawInfo));

        if (pConstantBufferForUserGeometryShader != NULL)
        {
            pConstantBufferForUserGeometryShader[0] = g_ShaderTimer;
        }
    }
}

//------------------------------------------------------------------------------
// 計算処理
//------------------------------------------------------------------------------

void CalculateUserShader()
{
    //  シェーダーアニメーション用のタイマーを更新する。
    g_ShaderTimer += 0.02f;
    if (g_ShaderTimer > 1.0f)
    {
        g_ShaderTimer -= 1.0f;
    }
}
