﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d.h>
#include "Ui2dCannonButton.h"
#include <nns/hid.h>
#include <nn/hid/hid_Mouse.h>
#include "InputController.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------

// 大砲を撃つアニメータ
static nn::ui2d::GroupAnimator* g_pFireAnimator = NULL;

// 大砲の弾を変更するアニメータ
static nn::ui2d::GroupAnimator* g_pBallChangeAnimator = NULL;

// ボタングループ
static nn::ui2d::ButtonGroup* g_pButtonGroup;

// ボタンを区別するためのタグ
enum ButtonTag {
    ButtonTag_Fire,
    ButtonTag_BallChange
};

// タッチもしくはクリックされているか
static bool g_Hold;

//------------------------------------------------------------------------------
//  ボタンが押されたときのコールバック
//------------------------------------------------------------------------------
static void LayoutButtonCallback(nn::ui2d::AnimButton* pButton, nn::ui2d::AnimButton::State prevState, nn::ui2d::AnimButton::State nextState, void* pParam)
{
    NN_UNUSED(pParam);
    switch (pButton->GetTag())
    {
    case ButtonTag_Fire:
        // 発射ボタンが押されたとき
        if (prevState == nn::ui2d::AnimButton::State_OnIdle && nextState == nn::ui2d::AnimButton::State_Down)
        {
            // 弾丸を切り替えるボタンの状態を見て、弾丸のテクスチャパターンアニメを設定する
            nn::ui2d::CheckButton* checkButton = nn::font::DynamicCast<nn::ui2d::CheckButton*>(g_pButtonGroup->FindButtonByTag(ButtonTag_BallChange));
            g_pBallChangeAnimator->StopAt(checkButton->IsChecked() ? 1.f : 0.f);
            // 発射アニメを先頭から再生
            g_pFireAnimator->PlayAuto(1.f);
        }
        break;
    default:
        // 何もしない
        break;
    }
}

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeCannonButton()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/CannonButton_00.arc", nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの初期化
    g_pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    {
        void* pFont = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool    result = g_pFont->SetResource(pDevice, pFont);
        NN_ASSERT(result);
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

        g_pArcResourceAccessor->RegisterFont("sample.bffnt", g_pFont);
    }

    // レイアウトの初期化
    nn::ui2d::BuildResultInformation buildResult;
    {
        g_pButtonGroup = AllocAndConstruct<nn::ui2d::ButtonGroup>();

        // ctrlモジュールで用意されているボタンを作成するためのクラスを作成。ボタングループを引数として指定する。
        nn::ui2d::DefaultControlCreator controlCreater(g_pButtonGroup);

        g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

        nn::ui2d::Layout::BuildOption    opt;
        opt.SetDefault();
        buildResult.SetDefault();

        g_pLayout->BuildWithName(&buildResult, pDevice, g_pArcResourceAccessor, &controlCreater, NULL, opt, "CannonButton_00.bflyt");
    }
    {
        // コールバックで区別しやすいように、ボタンにタグをつけておく
        // ボタンの名前は、部品ペインの名前と同じになる
        g_pButtonGroup->FindButtonByName("L_Btn_00")->SetTag(ButtonTag_Fire);
        g_pButtonGroup->FindButtonByName("L_Btn_01")->SetTag(ButtonTag_BallChange);
        // ボタンコールバックを設定する
        g_pButtonGroup->SetStateChangeCallbackAll(LayoutButtonCallback, NULL);
    }

    // グラフィックスリソースの設定
    InitializeGraphicsResource();

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    g_pDrawInfo = AllocAndConstruct<nn::ui2d::DrawInfo>();
    NN_ASSERT_NOT_NULL(g_pDrawInfo);
    g_pUi2dConstantBuffer = AllocAndConstruct<nn::font::GpuBuffer>();
    NN_ASSERT_NOT_NULL(g_pUi2dConstantBuffer);
    InitializeUi2dBuffers(*g_pDrawInfo, buildResult, g_pUi2dConstantBuffer);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo->SetGraphicsResource(g_pGraphicsResource);
        g_pDrawInfo->SetProjectionMtx(projection);
        g_pDrawInfo->SetViewMtx(view);
    }

    // アニメの作成
    {
        // "Fire"という名前のアニメタグのアニメーションを作成、アニメタグに関連付けられた1つ目のグループに属する
        // ペインにバインドする
        g_pFireAnimator = g_pLayout->CreateGroupAnimator(pDevice, "Fire");
        // 同様に"BallChange"という名前のアニメタグのアニメーションを作成する
        g_pBallChangeAnimator = g_pLayout->CreateGroupAnimator(pDevice, "BallChange");
        // アニメーションは、デフォルトの状態ではフレーム0でStopした状態になっている。
        // もしそれ以外の状態にしておきたい場合は、Stopメソッド等を呼び出す必要がある。
    }

    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

    g_Hold = false;
}

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeCannonButton()
{
    // ボタンの破棄
    g_pButtonGroup->FreeAll();

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pUi2dConstantBuffer->Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    DestructAndFree<nn::font::GpuBuffer>(g_pUi2dConstantBuffer);
    g_pUi2dConstantBuffer = NULL;
    DestructAndFree<nn::ui2d::DrawInfo>(g_pDrawInfo);
    g_pDrawInfo = NULL;

    FinalizeGraphicsResource();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);
    DestructAndFree<nn::ui2d::ButtonGroup>(g_pButtonGroup);
    g_pButtonGroup = NULL;
    g_pLayout = NULL;
    {
        g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);

        void* pFontResource = g_pFont->RemoveResource(pDevice);
        Ui2dDeallocateFunction(pFontResource, NULL);
        g_pFont->Finalize(pDevice);
        DestructAndFree<nn::font::ResFont>(g_pFont);
        g_pFont = NULL;
    }

    {
        g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        g_pArcResourceAccessor->Detach();
        g_pArcResourceAccessor->Finalize(pDevice);

        Ui2dDeallocateFunction(g_pLayoutArchiveBinary, NULL);
        g_pLayoutArchiveBinary = NULL;

        DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);
        g_pArcResourceAccessor = NULL;
    }
}

//------------------------------------------------------------------------------
// 計算処理
//------------------------------------------------------------------------------
void CalculateCannonButton()
{
    float x = 0.0f; // タッチの X 座標
    float y = 0.0f; // タッチの Y 座標
    bool hold = false; // タッチされているか
    bool trigger = false; // タッチされた瞬間か

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // マウス状態の取得
    x = static_cast<float>(GetMouseState().x);
    y = static_cast<float>(GetMouseState().y);
    if (GetMouseState().buttons.Test(nn::hid::MouseButton::Left::Index))
    {
        hold = true;
    }
#else

    // タッチ状態の取得
    nns::hid::TouchScreen* pTouchScreen = reinterpret_cast<nns::hid::TouchScreen*>(GetControllerManager().GetController(nns::hid::ControllerId_TouchScreen, 0));
    const std::vector<nns::hid::TouchScreen::TouchState>& states = pTouchScreen->GetTouchStates();
    for (size_t i = 0; i < states.size(); i++)
    {
        // 複数のタッチがあった場合は最初の 1 つだけを処理する
        const nns::hid::TouchScreen::TouchState& state = states[i];
        x = state.position.x;
        y = state.position.y;
        hold = true;
        break;
    }

#endif

    // タッチされた瞬間を判定
    if (!g_Hold && hold)
    {
        trigger = true;
    }
    g_Hold = hold;

    // ボタンの当たり判定を更新
    // ユーザーがボタンを押したフレームにアニメーションを開始するためには、
    // この処理はレイアウトのAnimateメソッドを呼び出す前に行う必要がある。
    {
        // タッチが返す位置は、(0,0)-(1280,720)の座標系なので、レイアウトの座標系に直す
        nn::util::Float2 pos;
        nn::font::Rectangle layoutRect = g_pLayout->GetLayoutRect();
        pos.x = layoutRect.GetWidth() / 2.0f * (2.0f * x / 1280.0f - 1.0f);
        pos.y = layoutRect.GetHeight() / 2.0f * (2.0f * y / 720.0f - 1.0f);
        g_pButtonGroup->Update(&pos, trigger);
    }
}
