﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{Ui2dDemo/Main.cpp,PageSampleUi2dDemo}
 *
 * @brief
 *  Ui2d ライブラリの使い方を示したサンプルプログラム
 */

/**
 * @page PageSampleUi2dDemo Ui2dDemo
 * @tableofcontents
 *
 * @image html Applications\Ui2dDemo\ui2ddemo.png
 *
 * @brief
 *  Ui2d ライブラリの使い方を示したサンプルプログラムです。
 *
 * @section PageSampleUi2dDemo_SectionBrief 概要
 *  ここでは、Ui2dDemo の説明をします。
 *
 * @section PageSampleUi2dDemo_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/Ui2dDemo Samples/Sources/Applications/Ui2dDemo @endlink 以下にあります。
 *
 * @section PageSampleUi2dDemo_SectionNecessaryEnvironment 必要な環境
 *  特にありません。
 *
 * @section PageSampleUi2dDemo_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、マウス、タッチパネルを利用して、以下の操作が可能です。
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> 十字ボタン上下、タッチパネルまたはマウスで項目を選択 </td><td> サンプルを選択します。 </td></tr>
 * <tr><td> A ボタン、マウス左ボタン押し </td><td> サンプルを実行します。 </td></tr>
 * <tr><td> B ボタン、マウス右ボタン押し </td><td> メニューに戻ります。 </td></tr>
 * </table>
 *
 * @section PageSampleUi2dDemo_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleUi2dDemo_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、メニューから選び実行するサンプルを切り替えることができます。
 *
 * @section PageSampleUi2dDemo_SectionDetail 解説
 *  各サンプルは初期化処理、メインループ処理、終了処理を行うだけのシンプルな流れになっています。
 *
 * メニューから選択または g_SampleMode の値を変更、または引数に 0 から始まる番号を入力することで、以下のサンプルを確認できます。
 *
 * - SampleMode_Simple              最も単純なサンプルです。
 * - SampleMode_MultiArcResource    複数のアーカイブリソースやリソースアクセサを扱うサンプルです。
 * - SampleMode_TextureChange       ペインのテクスチャを自ら読み込んだテクスチャに変更します。
 * - SampleMode_UserShader          カスタマイズしたシェーダを Ui2d ライブラリに読み込ませるサンプルです。
 * - SampleMode_FontChange          フォントをプログラム上で差し替えるサンプルです。
 * - SampleMode_ComplexFont         複合フォントを読み込むサンプルです。
 * - SampleMode_CannonButton        ボタンを扱うサンプルです。
 * - SampleMode_CapturePane         キャプチャテクスチャを扱うサンプルです。
 */
#include <cstdlib>

#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>
#include <nn/nn_Assert.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

#include <nn/ui2d.h>
#include <nn/font.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>

#include "Ui2dCommon.h"
#include "Ui2dSimple.h"
#include "Ui2dMultiArcResource.h"
#include "Ui2dTextureChange.h"
#include "Ui2dUserShader.h"
#include "Ui2dFontChange.h"
#include "Ui2dComplexFont.h"
#include "Ui2dCannonButton.h"
#include "Ui2dCapturePane.h"
#include "Ui2dAlignmentPane.h"
#include "Ui2dMenu.h"
#include "InputController.h"

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <filesystem>
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#endif

//------------------------------------------------------------------------------
//  定数の宣言
//------------------------------------------------------------------------------

// アプリケーションヒープとして確保されるメモリ領域の大きさ(byte)
static const size_t                         ApplicationHeapSize = 128 * 1024 * 1024;

const uint16_t* commandMenuNames[] =
{
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("Simple")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("MultiArcResource")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("TextureChange")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("UserShader")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("FontChange")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("ComplexFont")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("ConnonButton")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("CapturePane")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("AlignmentPane")),
};

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------

// サンプルの動作モードです。
static SampleMode                           g_SampleMode = SampleMode_CommandMenu;

SampleMode                                  g_NextSampleMode = g_SampleMode;

// グラフィックスフレームワーク
nns::gfx::GraphicsFramework                 g_GfxFramework;

// PrimitiveRenderer
nns::gfx::PrimitiveRenderer::Renderer*   g_pPrimitiveRenderer;

// ui2d レイアウト
nn::ui2d::Layout*                           g_pLayout = NULL;

// ui2d アーカイブリソースアクセッサ
nn::ui2d::ArcResourceAccessor*              g_pArcResourceAccessor = NULL;

// レイアウトアーカイブバイナリデータ
void*                                       g_pLayoutArchiveBinary = NULL;

// レイアウトで使用するフォントリソース
nn::font::ResFont*                          g_pFont = NULL;

// ui2d が描画に使用するグラフィックスリソース
nn::ui2d::GraphicsResource*                 g_pGraphicsResource;

// GraphicsResource に利用するメモリプールのオフセット
static ptrdiff_t                            g_OffsetForMemoryPoolOfGraphicsResource = 0;

// GraphicsResource に外部のメモリプールを利用するかどうか
static const bool                           g_IsUseExternalMemoryPoolForGraphicsResource = false;

// ui2d 描画情報
nn::ui2d::DrawInfo*                         g_pDrawInfo;

// レイアウト・フォントの描画に使用するコンスタントバッファ
nn::font::GpuBuffer*                        g_pUi2dConstantBuffer;

static ptrdiff_t                            g_OffsetToUi2dDrawMemoryPool;

// アプリケーションヒープ
static nn::util::BytePtr                    g_pMemoryHeap(NULL);
static nn::mem::StandardAllocator           g_ApplicationHeapHandle;

// ui2d のコンスタントバッファのダブルバッファリング用インデックス
static uint32_t                             g_Ui2dConstantBufferIndex   = 0;
// メモリアロケート回数のカウンター
static int                                  g_AllocationCount           = 0;

// ファイルシステムのメタデータキャッシュバッファ
static char*                                g_MountRomCacheBuffer = NULL;

static int                                  g_LoopCount = 0;

static bool                                 g_OneTimeSample = false;

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する
//------------------------------------------------------------------------------
void* Ui2dAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    void* ptr = g_ApplicationHeapHandle.Allocate(size, alignment);
    if (ptr != NULL)
    {
        g_AllocationCount++;
    }
    return ptr;
}

//------------------------------------------------------------------------------
void Ui2dDeallocateFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);

    if (ptr == NULL)
    {
        return;
    }
    g_AllocationCount--;
    g_ApplicationHeapHandle.Free(ptr);
}

//------------------------------------------------------------------------------
bool RegisterSlotForTexture(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
bool RegisterSlotForSampler(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
void UnregisterSlotForTexture(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
void UnregisterSlotForSampler(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        const int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }

}



//------------------------------------------------------------------------------
// メモリを確保してファイルを読み込みます
//------------------------------------------------------------------------------
void* ReadFileWithAllocate(const char* pFileName, size_t alignment)
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;
    int64_t fileSize;

    result = nn::fs::OpenFile(&fileHandle, pFileName, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    void* binary = Ui2dAllocateFunction(static_cast<size_t>(fileSize), alignment, NULL);
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, binary, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::fs::CloseFile(fileHandle);

    return binary;
}

//------------------------------------------------------------------------------
// あらかじめ確保されたバッファにファイルを読み込みます
//------------------------------------------------------------------------------
void* ReadFileToBuffer(size_t* pFileSize, const char* pFileName, void* pBuffer, ptrdiff_t offset)
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;

    result = nn::fs::OpenFile(&fileHandle, pFileName, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    int64_t size;
    result = nn::fs::GetFileSize(&size, fileHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *pFileSize = static_cast<size_t>(size);

    void* binary = nn::util::BytePtr(pBuffer).Advance(offset).Get();
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, binary, static_cast<size_t>(*pFileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::fs::CloseFile(fileHandle);

    return binary;
}

//-----------------------------------------------------------------------------
//  PrimitiveRenderer 初期化処理
//------------------------------------------------------------------------------
void InitializePrimitiveRenderer(nn::gfx::Device* pDevice)
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);

    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(Ui2dAllocateFunction, NULL);

    info.SetMultiBufferQuantity(nn::ui2d::ConstantBufferCount);

    // PrimitiveRendererのインスタンス
    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(pDevice, info);

    g_pPrimitiveRenderer->SetScreenWidth(g_GfxFramework.GetDisplayWidth());
    g_pPrimitiveRenderer->SetScreenHeight(g_GfxFramework.GetDisplayHeight());
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 破棄処理
//------------------------------------------------------------------------------
void FinalizePrimitiveRenderer(nn::gfx::Device* pDevice)
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);
    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pPrimitiveRenderer, pDevice, Ui2dDeallocateFunction, NULL);
    g_pPrimitiveRenderer = NULL;
}

//------------------------------------------------------------------------------
// ui2d 描画のためのバッファ関連初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dBuffers(nn::ui2d::DrawInfo& drawInfo, const nn::ui2d::BuildResultInformation& buildResult, nn::font::GpuBuffer* pUi2dConstantBuffer)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // バッファの再セットアップ
    // ui2d 描画に使用する各種 gfx バッファのためのメモリプールを作成
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);

    // BuildWithName でレイアウトデータをセットアップした際に BuildResultInformation に収集した各種バッファのサイズを満たすメモリプールを作成する。
    const size_t oneConstantBufferSize = (buildResult.requiredUi2dConstantBufferSize + buildResult.requiredFontConstantBufferSize);
    // ダブルバッファリング分のサイズが総サイズとなる
    const size_t constantBufferTotalSize = oneConstantBufferSize * nn::ui2d::ConstantBufferCount;

    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo);
    const size_t memoryPoolSize = nn::util::align_up(constantBufferTotalSize, memoryPoolGranularity);
    const size_t memoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);

    const nns::gfx::GraphicsFramework::MemoryPoolType constantBufferType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;

    nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(constantBufferType);
    g_OffsetToUi2dDrawMemoryPool = g_GfxFramework.AllocatePoolMemory(constantBufferType, memoryPoolSize, memoryPoolAlignment);

    // ui2d 用のコンスタントバッファを作成する。
    {
        nn::font::GpuBuffer::InitializeArg  arg;

        arg.SetGpuAccessFlag(static_cast<nn::gfx::GpuAccess>(nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer));
        arg.SetBufferSize(oneConstantBufferSize); // シングルバッファ分のサイズを指定する
        arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
        arg.SetMemoryPool(pMemoryPool);
        arg.SetMemoryPoolOffset(g_OffsetToUi2dDrawMemoryPool);
        arg.SetAllocator(Ui2dAllocateFunction, NULL);
        arg.SetAllocateSyncFlag(false);

        pUi2dConstantBuffer->Initialize(pDevice, arg);
    }

    // 別々のバッファを指定することも可能です。
    drawInfo.SetUi2dConstantBuffer(pUi2dConstantBuffer);
    drawInfo.SetFontConstantBuffer(pUi2dConstantBuffer);
}

//------------------------------------------------------------------------------
// レイアウトデータセットアップ処理
//------------------------------------------------------------------------------
void BuildLayoutData(
    nn::ui2d::BuildResultInformation& totalRequiredBufferInfomation,
    nn::ui2d::Layout* pLayout,
    nn::ui2d::Layout::BuildOption& option,
    nn::ui2d::ResourceAccessor* pResourceAccessor,
    const char* pFileName)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトデータをセットアップして、必要なバッファサイズを収集します。
    nn::ui2d::BuildResultInformation   buildResultInformation;
    buildResultInformation.SetDefault();

    pLayout->BuildWithName(&buildResultInformation, pDevice, pResourceAccessor, NULL, NULL, option, pFileName);

    totalRequiredBufferInfomation.requiredUi2dConstantBufferSize += buildResultInformation.requiredUi2dConstantBufferSize;
    totalRequiredBufferInfomation.requiredFontConstantBufferSize += buildResultInformation.requiredFontConstantBufferSize;
}

//------------------------------------------------------------------------------
void InitializeGraphicsResource()
{
    const uint32_t CharMax = 512;
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // グラフィックスリソースの設定
    g_pGraphicsResource = AllocAndConstruct<nn::ui2d::GraphicsResource>();
    NN_ASSERT_NOT_NULL(g_pGraphicsResource);

    // メモリープールの指定を省略することもできます。
    // その場合は、内部で自動的にメモリープールが生成・初期化されます。
    if (NN_STATIC_CONDITION(g_IsUseExternalMemoryPoolForGraphicsResource))
    {
        const size_t SizeOfGraphicsResourceMemPool = nn::ui2d::GraphicsResource::CalculateMemoryPoolSize(pDevice, CharMax);

        g_OffsetForMemoryPoolOfGraphicsResource = g_GfxFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            SizeOfGraphicsResourceMemPool,
            nn::ui2d::GraphicsResource::CalculateMemoryPoolAlignment(pDevice));

        g_pGraphicsResource->Setup(pDevice, CharMax,
            g_GfxFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer),
            g_OffsetForMemoryPoolOfGraphicsResource,
            SizeOfGraphicsResourceMemPool);
    }
    else
    {
        g_OffsetForMemoryPoolOfGraphicsResource = NULL;
        g_pGraphicsResource->Setup(pDevice, CharMax);
    }

    g_pGraphicsResource->RegisterCommonSamplerSlot(RegisterSlotForSampler, &g_GfxFramework);
}

//------------------------------------------------------------------------------
void FinalizeGraphicsResource()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pGraphicsResource->UnregisterCommonSamplerSlot(UnregisterSlotForSampler, &g_GfxFramework);
    g_pGraphicsResource->Finalize(pDevice);

    if (g_OffsetForMemoryPoolOfGraphicsResource != NULL)
    {
        g_GfxFramework.FreePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_OffsetForMemoryPoolOfGraphicsResource);
        g_OffsetForMemoryPoolOfGraphicsResource = NULL;
    }

    DestructAndFree<nn::ui2d::GraphicsResource>(g_pGraphicsResource);
    g_pGraphicsResource = NULL;
}

//------------------------------------------------------------------------------
// nn::fs用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* FsAllocateFunction(size_t size)
{
    return Ui2dAllocateFunction(size, 4, NULL);
}

//-----------------------------------------------------------------------------
static void FsDeallocateFunction(void* ptr, size_t size)
{
    NN_UNUSED(size);
    Ui2dDeallocateFunction(ptr, NULL);
}

//-----------------------------------------------------------------------------
// ファイルシステムの初期化
//-----------------------------------------------------------------------------
static void InitializeFileSystem()
{
    nn::fs::SetAllocator( FsAllocateFunction, FsDeallocateFunction );

    size_t cacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    result = nn::fs::MountRom("Contents", g_MountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_UNUSED(result);
}

//------------------------------------------------------------------------------
//  計算処理コールバック
//------------------------------------------------------------------------------
void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    int* pFrame = reinterpret_cast<int*>(pUserData);

    // 各サンプルごとの更新処理
    {
        switch (g_SampleMode)
        {
        case SampleMode_Simple:
            break;
        case SampleMode_MultiArcResource:
            CalculateMultiArcResource(*pFrame);
            break;
        case SampleMode_TextureChange:
            CalculateTextureChange(*pFrame);
            break;
        case SampleMode_UserShader:
            CalculateUserShader();
            break;
        case SampleMode_FontChange:
            break;
        case SampleMode_ComplexFont:
            break;
        case SampleMode_CannonButton:
            CalculateCannonButton();
            break;
        case SampleMode_CapturePane:
            CalculateCapturePane();
            break;
        case SampleMode_AlignmentPane:
            CalculateAlignmentPane(*pFrame);
            break;
        case SampleMode_CommandMenu:
            CalculateMenu();
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // レイアウトの更新処理
    {
        g_pLayout->AnimateAndUpdateAnimFrame();

        // Calculate
        {
            g_pDrawInfo->Map(g_Ui2dConstantBufferIndex);
            {
                g_pLayout->Calculate(*g_pDrawInfo);
                if (NN_STATIC_CONDITION(g_SampleMode == SampleMode_MultiArcResource))
                {
                    CalculateMultiArcResourceLayout();
                }
            }
            g_pDrawInfo->Unmap();
            g_pDrawInfo->SetGpuAccessBufferIndex(g_Ui2dConstantBufferIndex);

            ++g_Ui2dConstantBufferIndex;
            g_Ui2dConstantBufferIndex %= nn::ui2d::ConstantBufferCount;
        }
    }
}

//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    NN_UNUSED(pUserData);

    // 描画処理
    g_GfxFramework.BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* pCmdBuffer = g_GfxFramework.GetRootCommandBuffer(bufferIndex);
        nn::gfx::ColorTargetView* pTarget = g_GfxFramework.GetColorTargetView();
        nn::gfx::DepthStencilView* pDepthStencilView = g_GfxFramework.GetDepthStencilView();

        // バッファのクリア
        pCmdBuffer->ClearColor(pTarget, 0.0f, 0.0f, 0.5f, 1.0f, NULL);
        pCmdBuffer->ClearDepthStencil(
            pDepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

        // レンダーステートの設定
        pCmdBuffer->SetRenderTargets(1, &pTarget, pDepthStencilView);
        pCmdBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());
        pCmdBuffer->SetBlendState(g_GfxFramework.GetBlendState(
            nns::gfx::GraphicsFramework::BlendStateType_Alpha));
        pCmdBuffer->SetDepthStencilState(g_GfxFramework.GetDepthStencilState(
            nns::gfx::GraphicsFramework::DepthStencilStateType_Disabled));
        pCmdBuffer->SetRasterizerState(g_GfxFramework.GetRasterizerState(
            nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));

        if (g_SampleMode == SampleMode_CapturePane)
        {
            DrawCapturePaneSampleBeforeCaptureTexture(*pCmdBuffer);
        }

        pCmdBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Texture);

        // キャプチャテクスチャの更新処理
        {
            g_pLayout->DrawCaptureTexture(g_GfxFramework.GetDevice(), *g_pDrawInfo, *pCmdBuffer);
        }

        {
            // レンダーステートの設定
            pCmdBuffer->SetRenderTargets(1, &pTarget, pDepthStencilView);
            pCmdBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());
            pCmdBuffer->SetBlendState(g_GfxFramework.GetBlendState(
                nns::gfx::GraphicsFramework::BlendStateType_Alpha));
            pCmdBuffer->SetDepthStencilState(g_GfxFramework.GetDepthStencilState(
                nns::gfx::GraphicsFramework::DepthStencilStateType_Disabled));
            pCmdBuffer->SetRasterizerState(g_GfxFramework.GetRasterizerState(
                nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));
        }

        // Draw
        {
            g_pLayout->Draw(*g_pDrawInfo, *pCmdBuffer);
            if (NN_STATIC_CONDITION(g_SampleMode == SampleMode_MultiArcResource))
            {
                DrawMultiArcResourceLayout(pCmdBuffer);
            }
        }
    }
    g_GfxFramework.EndFrame(bufferIndex);
}

//------------------------------------------------------------------------------
// アプリケーションの初期化処理
//------------------------------------------------------------------------------
static void InitializeApp()
{
    // ui2d ランタイムの初期化処理
    switch (g_SampleMode)
    {
    case SampleMode_Simple:
        InitializeSimple();
        break;
    case SampleMode_MultiArcResource:
        InitializeMultiArcResource();
        break;
    case SampleMode_TextureChange:
        InitializeSimple();
        InitializeTextureChange();
        break;
    case SampleMode_UserShader:
        InitializeUserShader();
        break;
    case SampleMode_FontChange:
        InitializeSimple();
        InitializeFontChange();
        break;
    case SampleMode_ComplexFont:
        InitializeComplexFont();
        break;
    case SampleMode_CannonButton:
        InitializeCannonButton();
        break;
    case SampleMode_CapturePane:
        InitializeCapturePane();
        break;
    case SampleMode_AlignmentPane:
        InitializeAlignmentPane();
        break;
    case SampleMode_CommandMenu:
        InitializeMenu(commandMenuNames, NN_ARRAY_SIZE(commandMenuNames));
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    g_LoopCount = 0;
}

//------------------------------------------------------------------------------
// アプリケーションの終了処理
//------------------------------------------------------------------------------
static void FinalizeApp()
{
    switch (g_SampleMode)
    {
    case SampleMode_Simple:
        FinalizeSimple();
        break;
    case SampleMode_MultiArcResource:
        FinalizeMultiArcResource();
        break;
    case SampleMode_TextureChange:
        FinalizeSimple();
        FinalizeTextureChange();
        break;
    case SampleMode_UserShader:
        FinalizeUserShader();
        break;
    case SampleMode_FontChange:
        FinalizeSimple();
        FinalizeFontChange();
        break;
    case SampleMode_ComplexFont:
        FinalizeComplexFont();
        break;
    case SampleMode_CannonButton:
        FinalizeCannonButton();
        break;
    case SampleMode_CapturePane:
        FinalizeCapturePane();
        break;
    case SampleMode_AlignmentPane:
        FinalizeAlignmentPane();
        break;
    case SampleMode_CommandMenu:
        FinalizeMenu();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

}

//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // 引数から起動
    if (nn::os::GetHostArgc() > 1)
    {
        int idxDemo = atoi(nn::os::GetHostArgv()[1]);
        g_NextSampleMode = g_SampleMode = (SampleMode)std::max(std::min(idxDemo, (int)SampleMode_CannonButton), (int)SampleMode_Simple);
    }

    // メニュー以外の直接起動は１度のみの再生モードで実行
    if (g_SampleMode != SampleMode_CommandMenu)
    {
        g_OneTimeSample = true;
    }


    // アプリケーションで使用するヒープ領域を初期化
    g_pMemoryHeap.Reset(new uint8_t[ApplicationHeapSize]);
    g_ApplicationHeapHandle.Initialize(g_pMemoryHeap.Get(), ApplicationHeapSize);

    // ファイルシステムの初期化(gfx の初期化より前に実行します)
    InitializeFileSystem();

    // グラフィックスフレームワークの初期化
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(1024 * 1024 * 32);
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetBufferCount(2);
    fwInfo.SetSwapChainBufferCount(2);

    // キャプチャテクスチャ機能を使用するための RenderTarget 用のメモリをデフォルトから 64MB 増やす
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 80 * 1024 * 1024);

    g_GfxFramework.Initialize(fwInfo);

    // ウィンドウハンドルを取得する
    nn::vi::NativeWindowHandle hwnd = NULL;
    nn::Result result = nn::vi::GetNativeWindow(&hwnd, g_GfxFramework.GetLayer());
    NN_ASSERT(result.IsSuccess());

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    SetForegroundWindow(static_cast<HWND>(hwnd));
#endif

    const int displayWidth = g_GfxFramework.GetDisplayWidth();
    const int displayHeight = g_GfxFramework.GetDisplayHeight();
    InitializeInputController(displayWidth, displayHeight, reinterpret_cast<void*>(hwnd));

    //------------------------------------------
    // メインループ
    NN_LOG("Start demo.\n");
    for(;;)
    {
        //------------------------------------------
        // 初期化
        {
            InitializeApp();
        }

        int frame = 0;
        g_GfxFramework.SetCalculateCallback(CalculateCallback, &frame);
        g_GfxFramework.SetMakeCommandCallback(MakeCommandCallback, &frame);
        g_GfxFramework.ResetFrame();

        for(;;)
        {
            UpdateInputController();

            g_GfxFramework.ProcessFrame();

            frame++;

            if (g_SampleMode != SampleMode_CommandMenu)
            {
                g_LoopCount++;

                if (g_LoopCount >= 60 * 10 || GetPad().IsTriggered(Pad::BUTTON_B) || GetMouseState().buttons.Test(nn::hid::MouseButton::Right::Index))
                {
                    if (g_OneTimeSample == true)
                    {
                        g_NextSampleMode = SampleMode_Exit;
                    }
                    else
                    {
                        g_NextSampleMode = SampleMode_CommandMenu;
                    }
                }
            }

            // モードが変わった場合
            if (g_NextSampleMode != g_SampleMode)
            {
                g_GfxFramework.QueueFinish();

                //------------------------------------------
                // 終了処理
                FinalizeApp();

                // グラフィックスフレームワーク 終了処理
                g_GfxFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_OffsetToUi2dDrawMemoryPool);

                g_SampleMode = g_NextSampleMode;
                frame = 0;
                break;
            }
        }

        if (g_SampleMode == SampleMode_Exit)
        {
            break;
        }
    }
    // ファイルシステム終了処理。
    nn::fs::Unmount("Contents");
    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;

    g_GfxFramework.Finalize();

    // ヒープメモリーの終了処理
    g_ApplicationHeapHandle.Finalize();
    delete[] reinterpret_cast<uint8_t*>(g_pMemoryHeap.Get());

    NN_ASSERT(g_AllocationCount == 0);
}
