﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/util/util_BytePtr.h>

#include <nn/audio.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/os.h>

#include "Audio.h"

#if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic ignored "-Wunused-const-variable"
#pragma clang diagnostic ignored "-Wunused-function"
#endif

//-----------------------------------------------------------------------------

namespace {

    const int RenderRate = 48000;
    const int RenderCount = (RenderRate / 200);
    const int ChannelCount = 2;
    const int8_t BusIndexes[2] = { 0, 1 };

    const int VoiceCount = 24;
    const int SeCount = 4;

    const int CircularBufferSinkFrameCount = 10;
    const int PerformanceBufferCount = 3;

    const char* g_SeFileNames[SeCount] =
    {
        "asset:/AudioCommon/SampleSe0.adpcm",
        "asset:/AudioCommon/SampleSe1.adpcm",
        "asset:/AudioCommon/SampleSe2.adpcm",
        "asset:/AudioCommon/SampleSe3.adpcm",
    };

    const size_t PoolMemorySize = 26 * 1024 * 1024;

    size_t g_SeDataSize[SeCount] = { 0 };

    nn::mem::StandardAllocator g_MemoryPoolAllocator;
    nn::audio::MemoryPoolType g_MemoryPool;
    void* g_WorkBufferForMemoryPool = nullptr;

    void* g_WorkBufferForAudioRenderer = nullptr;
    void* g_WorkBufferForConfig = nullptr;
    void* g_WorkBufferForCircularBufferSink = nullptr;
    void* g_WorkBuffersForPerformanceMetrics[PerformanceBufferCount] = { 0 };

    void* g_SeDataBuffer[SeCount] = { 0 };
    void* g_ReverbBuffer = nullptr;

    nn::audio::AudioRendererHandle g_AudioRendererHandle;
    nn::audio::AudioRendererConfig g_AudioRendererConfig;
    nn::os::SystemEvent g_AudioRendererEvent;

    nn::audio::FinalMixType g_FinalMix;
    nn::audio::DeviceSinkType g_DeviceSink;
    nn::audio::CircularBufferSinkType g_CircularBufferSink;

    nn::audio::VoiceType g_Voice[VoiceCount];
    nn::audio::WaveBuffer g_WaveBuffers[SeCount];

    nn::audio::AdpcmHeaderInfo* g_AdpcmHeader[SeCount];

    nn::audio::ReverbType g_Reverb;

    size_t g_PerformanceBufferSize = 0;
    int g_AudioPerformanceBufferIndex = 0;

    size_t g_CircularBufferSinkReadBufferSize = 0;
    void* g_CircularBufferSinkReadBuffer = nullptr;

    nn::Result GetFileSize(size_t* outValue, nn::fs::FileHandle handle)
    {
        int64_t size;
        nn::Result result = nn::fs::GetFileSize(&size, handle);
        if (result.IsSuccess())
        {
            *outValue = static_cast<size_t>(size);
        }
        return result;
    }

    std::size_t ReadAdpcmFile(
        nn::mem::StandardAllocator& allocator,
        nn::mem::StandardAllocator& memoryPoolAllocator,
        nn::audio::AdpcmHeaderInfo* header,
        void** adpcmData,
        const char* filename)
    {
        nn::fs::FileHandle handle;
        nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        size_t size = 0;
        result = GetFileSize(&size, handle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        void* buffer = allocator.Allocate(size);
        NN_ABORT_UNLESS_NOT_NULL(buffer);

        result = nn::fs::ReadFile(handle, 0, buffer, size);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::fs::CloseFile(handle);

        nn::audio::ParseAdpcmHeader(header, buffer, nn::audio::AdpcmHeaderSize);

        size_t adpcmDataSize = size - nn::audio::AdpcmHeaderSize;
        *adpcmData = memoryPoolAllocator.Allocate(adpcmDataSize, nn::audio::BufferAlignSize);
        NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

        std::memcpy(*adpcmData, nn::util::ConstBytePtr(buffer, nn::audio::AdpcmHeaderSize).Get(), adpcmDataSize);

        allocator.Free(buffer);

        return adpcmDataSize;
    }

    nn::audio::VoiceType* FindInvalidVoice()
    {
        for (int i = 0; i < VoiceCount; i++)
        {
            if (!nn::audio::IsVoiceValid(&g_Voice[i]))
            {
                return &g_Voice[i];
            }
        }

        return nullptr;
    }
}

void InitializeAudio(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    // メモリプール用アロケータの初期化
    g_WorkBufferForMemoryPool = allocator.Allocate(PoolMemorySize, nn::audio::MemoryPoolType::SizeGranularity);
    g_MemoryPoolAllocator.Initialize(g_WorkBufferForMemoryPool, PoolMemorySize);

    // AudioRenderer の初期化
    const nn::audio::AudioRendererParameter audioRendererParameter = GetAudioRendererParameter();
    size_t audioRendererBufferSize = nn::audio::GetAudioRendererWorkBufferSize(audioRendererParameter);
    g_WorkBufferForAudioRenderer = allocator.Allocate(audioRendererBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(g_WorkBufferForAudioRenderer);

    nn::Result result = nn::audio::OpenAudioRenderer(
        &g_AudioRendererHandle,
        &g_AudioRendererEvent,
        audioRendererParameter,
        g_WorkBufferForAudioRenderer,
        audioRendererBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // AudioRendererConfig の初期化
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(audioRendererParameter);
    g_WorkBufferForConfig = allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(g_WorkBufferForConfig);

    nn::audio::InitializeAudioRendererConfig(&g_AudioRendererConfig, audioRendererParameter, g_WorkBufferForConfig, configBufferSize);

    // メモリープールの作成
    bool rt = nn::audio::AcquireMemoryPool(&g_AudioRendererConfig, &g_MemoryPool, g_WorkBufferForMemoryPool, PoolMemorySize);
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&g_MemoryPool);
    NN_ABORT_UNLESS(rt);

    // FinalMix の追加
    nn::audio::AcquireFinalMix(&g_AudioRendererConfig, &g_FinalMix, ChannelCount);

    // DeviceSink の追加
    result = nn::audio::AddDeviceSink(&g_AudioRendererConfig, &g_DeviceSink, &g_FinalMix, BusIndexes, ChannelCount, "MainAudioOut");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // CircularBufferSink の追加
    size_t bufferSizeForCircularBufferSink = nn::audio::GetRequiredBufferSizeForCircularBufferSink(
        &audioRendererParameter,
        ChannelCount,
        CircularBufferSinkFrameCount,
        nn::audio::SampleFormat_PcmInt16);
    g_WorkBufferForCircularBufferSink = g_MemoryPoolAllocator.Allocate(bufferSizeForCircularBufferSink, nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(g_WorkBufferForCircularBufferSink);

    result = nn::audio::AddCircularBufferSink(
        &g_AudioRendererConfig,
        &g_CircularBufferSink,
        &g_FinalMix,
        BusIndexes,
        ChannelCount,
        g_WorkBufferForCircularBufferSink,
        bufferSizeForCircularBufferSink,
        nn::audio::SampleFormat_PcmInt16);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Reverb の追加
    size_t bufferSizeForReverb = nn::audio::GetRequiredBufferSizeForReverb(audioRendererParameter.sampleRate, ChannelCount);
    g_ReverbBuffer = g_MemoryPoolAllocator.Allocate(bufferSizeForReverb, nn::audio::BufferAlignSize);

    result = nn::audio::AddReverb(&g_AudioRendererConfig, &g_Reverb, g_ReverbBuffer, bufferSizeForReverb, &g_FinalMix, ChannelCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::audio::SetReverbInputOutput(&g_Reverb, BusIndexes, BusIndexes, ChannelCount);

    nn::audio::SetReverbEarlyMode(&g_Reverb, nn::audio::ReverbType::EarlyMode_Hall);
    nn::audio::SetReverbEarlyGain(&g_Reverb, 0.8f);
    nn::audio::SetReverbPredelayTime(&g_Reverb, nn::TimeSpan::FromMilliSeconds(10));
    nn::audio::SetReverbLateMode(&g_Reverb, nn::audio::ReverbType::LateMode_Hall);
    nn::audio::SetReverbLateGain(&g_Reverb, 0.5f);
    nn::audio::SetReverbDecayTime(&g_Reverb, nn::TimeSpan::FromSeconds(1));
    nn::audio::SetReverbHighFrequencyDecayRatio(&g_Reverb, 0.5f);
    nn::audio::SetReverbColoration(&g_Reverb, 0.0f);
    nn::audio::SetReverbReverbGain(&g_Reverb, 0.8f);
    nn::audio::SetReverbOutGain(&g_Reverb, 1.0f);

    // ここまでの設定を AudioRenderer に反映して、開始
    result = nn::audio::RequestUpdateAudioRenderer(g_AudioRendererHandle, &g_AudioRendererConfig);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::audio::StartAudioRenderer(g_AudioRendererHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SE の準備
    for (int i = 0; i < SeCount; i++)
    {
        g_AdpcmHeader[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(
            g_MemoryPoolAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo),
                NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        g_SeDataSize[i] = ReadAdpcmFile(allocator, g_MemoryPoolAllocator, g_AdpcmHeader[i], &g_SeDataBuffer[i], g_SeFileNames[i]);
    }
}

void FinalizeAudio(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    // レンダリングを終了
    nn::audio::StopAudioRenderer(g_AudioRendererHandle);
    nn::audio::CloseAudioRenderer(g_AudioRendererHandle);
    nn::os::DestroySystemEvent(g_AudioRendererEvent.GetBase());

    // エフェクトバッファの解放
    g_ReverbBuffer = nullptr;

    // 波形バッファの解放
    for (int i = 0; i < SeCount; ++i)
    {
        g_AdpcmHeader[i] = nullptr;
        g_SeDataBuffer[i] = nullptr;
    }

    // メモリプールアロケータの解放
    g_MemoryPoolAllocator.Finalize();
    if (g_WorkBufferForMemoryPool)
    {
        allocator.Free(g_WorkBufferForMemoryPool);
        g_WorkBufferForMemoryPool = nullptr;
    }

    // Renderer, Config バッファの解放
    if (g_WorkBufferForConfig)
    {
        allocator.Free(g_WorkBufferForConfig);
        g_WorkBufferForConfig = nullptr;
    }
    if (g_WorkBufferForAudioRenderer)
    {
        allocator.Free(g_WorkBufferForAudioRenderer);
        g_WorkBufferForAudioRenderer = nullptr;
    }
}

nn::audio::AudioRendererParameter GetAudioRendererParameter() NN_NOEXCEPT
{
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);

    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 2; // FinalMix(2)
    parameter.voiceCount = VoiceCount;
    parameter.subMixCount = 2;
    parameter.sinkCount = 2; // DeviceSink + CircularBufferSink
    parameter.effectCount = 3;
    parameter.performanceFrameCount = 5;

    return parameter;
}

void InitializeAudioPerformanceMetrics(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    // nn::audio でパフォーマンス情報収集に必要なバッファを確保
    g_PerformanceBufferSize = GetRequiredBufferSizeForAudioPerformanceFrames();

    for (int i = 0; i < PerformanceBufferCount; i++)
    {
        g_WorkBuffersForPerformanceMetrics[i] = allocator.Allocate(g_PerformanceBufferSize);
        NN_ABORT_UNLESS_NOT_NULL(g_WorkBuffersForPerformanceMetrics[i]);
    }
}

void FinalizeAudioPerformanceMetrics(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    for (int i = 0; i < PerformanceBufferCount; i++)
    {
        allocator.Free(g_WorkBuffersForPerformanceMetrics[i]);
        g_WorkBuffersForPerformanceMetrics[i] = nullptr;
    }
}

void InitializeCircularBufferSink(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    nn::audio::AudioRendererParameter parameter = GetAudioRendererParameter();
    g_CircularBufferSinkReadBufferSize = CircularBufferSinkFrameCount * parameter.sampleCount * ChannelCount * sizeof(int16_t);
    g_CircularBufferSinkReadBuffer = allocator.Allocate(g_CircularBufferSinkReadBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(g_CircularBufferSinkReadBuffer);
}

void FinalizeCircularBufferSink(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    allocator.Free(g_CircularBufferSinkReadBuffer);
    g_CircularBufferSinkReadBuffer = nullptr;
}

void WaitAudioRendererEvent() NN_NOEXCEPT
{
    g_AudioRendererEvent.Wait();
}

void RequestUpdateAudioRenderer() NN_NOEXCEPT
{
    for (int i = 0; i < VoiceCount; i++)
    {
        if (!nn::audio::IsVoiceValid(&g_Voice[i]))
        {
            continue;
        }

        // 再生が完了したら、リリースする
        if (nn::audio::GetReleasedWaveBuffer(&g_Voice[i]) != nullptr)
        {
            nn::audio::ReleaseVoiceSlot(&g_AudioRendererConfig, &g_Voice[i]);
        }
    }

    nn::Result result = nn::audio::RequestUpdateAudioRenderer(g_AudioRendererHandle, &g_AudioRendererConfig);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

bool PlaySe(int index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(index, 0);
    NN_SDK_REQUIRES_LESS(index, SeCount);

    nn::audio::VoiceType* voice = FindInvalidVoice();

    if (voice == nullptr)
    {
        return false;
    }

    nn::audio::AcquireVoiceSlot(
        &g_AudioRendererConfig,
        voice,
        g_AdpcmHeader[index]->sampleRate,
        1,
        nn::audio::SampleFormat_Adpcm,
        nn::audio::VoiceType::PriorityHighest,
        &g_AdpcmHeader[index]->parameter,
        sizeof(nn::audio::AdpcmParameter));

    nn::audio::SetVoiceDestination(&g_AudioRendererConfig, voice, &g_FinalMix);

    g_WaveBuffers[index].buffer = g_SeDataBuffer[index];
    g_WaveBuffers[index].size = g_SeDataSize[index];
    g_WaveBuffers[index].startSampleOffset = 0;
    g_WaveBuffers[index].endSampleOffset = g_AdpcmHeader[index]->sampleCount;
    g_WaveBuffers[index].loop = false;
    g_WaveBuffers[index].isEndOfStream = false;
    g_WaveBuffers[index].pContext = &g_AdpcmHeader[index]->loopContext;
    g_WaveBuffers[index].contextSize = sizeof(nn::audio::AdpcmContext);

    nn::audio::AppendWaveBuffer(voice, &g_WaveBuffers[index]);
    nn::audio::SetVoicePlayState(voice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(voice, &g_FinalMix, 0.707f / 2, 0, BusIndexes[0]);
    nn::audio::SetVoiceMixVolume(voice, &g_FinalMix, 0.707f / 2, 0, BusIndexes[1]);

    return true;
}

bool IsEnableReverb() NN_NOEXCEPT
{
    return nn::audio::IsReverbEnabled(&g_Reverb);
}

void EnableReverb(bool enable) NN_NOEXCEPT
{
    nn::audio::SetReverbEnabled(&g_Reverb, enable);
}

size_t GetRequiredBufferSizeForAudioPerformanceFrames() NN_NOEXCEPT
{
    return nn::audio::GetRequiredBufferSizeForPerformanceFrames(GetAudioRendererParameter());
}

void* SwitchAudioPerformanceFrameBuffer() NN_NOEXCEPT
{
    void* lastBuffer = nn::audio::SetPerformanceFrameBuffer(
        &g_AudioRendererConfig,
        g_WorkBuffersForPerformanceMetrics[g_AudioPerformanceBufferIndex],
        g_PerformanceBufferSize);

    g_AudioPerformanceBufferIndex++;
    if (g_AudioPerformanceBufferIndex + 1 >= PerformanceBufferCount)
    {
        g_AudioPerformanceBufferIndex = 0;
    }

    return lastBuffer;
}

size_t ReadCircularBufferSink(const void** outBuffer) NN_NOEXCEPT
{
    *outBuffer = g_CircularBufferSinkReadBuffer;
    return nn::audio::ReadCircularBufferSink(
        &g_CircularBufferSink,
        g_CircularBufferSinkReadBuffer,
        g_CircularBufferSinkReadBufferSize);
}
