﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{SpyAtk.cpp,PageSampleSpyAtk}
 *
 * @brief
 *  nn::spy, nn::spy::atk ライブラリのサンプルプログラム
 */

/**
 * @page PageSampleSpyAtk SpyAtk
 * @tableofcontents
 *
 * @brief
 *  nn::spy, nn::spy::atk ライブラリ サンプルプログラムの解説です。
 *
 * @section PageSampleSpyAtk_SectionBrief 概要
 *  nn::spy, nn::spy::atk ライブラリを使って、 nn::atk の状態をモニタするサンプルです。
 *
 * @section PageSampleSpyAtk_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/SpyAtk Samples/Sources/Applications/SpyAtk @endlink 以下にあります。
 *
 * @section PageSampleSpyAtk_SectionNecessaryEnvironment 必要な環境
 *  オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleSpyAtk_SectionHowToOperate 操作方法
 *  キーボードや DebugPad による入力を用いてサウンドを再生し、サウンドの再生状態や処理負荷、出力波形を Spy ツールで確認することができます。
 *  サンプルを起動するとコンソールに操作方法が表示されます。
 *  Spy ツールのリボンから接続先を設定して接続トグルボタンを ON にすると、通信が開始されます。
 *
 * @section PageSampleSpyAtk_SectionPrecaution 注意事項
 *  Release ビルドでは、Spy.exe と通信できません。
 *  開発用機能である Spy は、誤って製品に含まれないようにするために、Release ビルドでは無効化されています。
 *  Release ビルドで Spy を利用したい場合は、spy_Config.h を編集して NN_ENABLE_SPY マクロを定義する必要があります。
 *
 * @section PageSampleSpyAtk_SectionHowToExecute 実行手順
 *  1. サンプルプログラムを Debug or Develop ビルドし、実行します。
 *  2. NintendoTargetManager が起動していない場合は、起動します。
 *  3. PC 上で Spy ツールを起動して、サンプルプログラムと接続します。
 *
 * @section PageSampleSpyAtk_SectionDetail 解説
 *
 * @subsection PageSampleSpyAtk_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  SpyAtk.cpp
 *  @includelineno SpyAtk.cpp
 *
 * @subsection PageSampleSpyAtk_SectionSampleDetail サンプルプログラムの解説
 *  このサンプルプログラムの流れは、以下の通りです。
 *
 * - htcs などの初期化
 * - InitializeAtk()
 *   - atk の初期化
 *   - パフォーマンス情報の取得のためプロファイラを有効化
 * - InitializeSpy()
 *   - nn::spy::SpyController の初期化
 * - InitializeSpyAtk()
 *   - nn::spy::atk::AtkSpyModule の初期化
 *   - nn::spy::atk::AtkSpyModule を nn::spy::SpyController に登録
 * - StartSpy()
 *   - Spy ツールとの通信を開始
 * - メインループ
 *   - (必要に応じて) nn::spy::SpyController::SetCurrentApplicationFrame() でアプリケーションフレームを送信
 *   - nn::spy::atk::AtkSpyModule::PushSoundState() で サウンドの再生状態の情報を送信
 *   - nn::spy::atk::AtkSpyModule::PushPerformanceMetrics() で パフォーマンス情報を送信
 *   - nn::spy::atk::AtkSpyModule::PushWaveform() で 出力波形を送信
 *   - nn::spy::atk::AtkSpyModule::PushAtkProfiles() で nn::atk パフォーマンス情報を送信
 *   - nn::spy::atk::AtkSpyModule::PushSequenceVariable() で シーケンス変数情報を送信
 *   - nn::spy::atk::AtkSpyModule::PushStreamSoundInfo() でストリームサウンド情報を送信
 *   - nn::spy::DebugModule::PushDataBufferUsage() で Spy のデバッグ情報(データバッファ使用量)を送信
 * - StopSpy()
 *   - Spy ツールとの通信を終了
 * - FinalizeSpyAtk()
 *   - nn::spy::atk::AtkSpyModule を nn::spy::SpyController から登録解除
 *   - nn::spy::atk::AtkSpyModule の終了
 * - FinalizeSpy()
 *   - nn::spy::SpyController の終了
 * - FinalizeAtk()
 *   - atk の終了
 * - htcs などの終了
 *
 */

#include <nn/nn_Assert.h>
#include <nns/nns_Log.h>
#include <nn/htcs.h>
#include <nn/spy.h>
#include <nn/spy/atk/spy_AtkSpyModule.h>

#include <nns/atk/atk_SampleCommon.h>

#include "Atk.h"

//-----------------------------------------------------------------------------

namespace {

    const size_t SpyControllerDataBufferLength = 1024 * 1024;

    nn::spy::SpyController g_SpyController;
    nn::spy::atk::AtkSpyModule g_AtkSpyModule;

    void* g_WorkBufferForSpyController = nullptr;
    void* g_WorkBufferForAtkSpyModule = nullptr;

    void* g_CircularBufferSinkBuffer = nullptr;
    size_t g_CircularBufferSinkBufferSize;

    int g_AppFrameCount = 0;

    void InitializeHtcs()
    {
        nn::htcs::Initialize(nns::atk::Allocate, nns::atk::Free);
    }

    void FinalizeHtcs()
    {
        nn::htcs::Finalize();
    }

    void InitializeSpy()
    {
        // nn::spy::SpyController を初期化します。
        nn::spy::SpyController::InitializeArg initializeArg;
        initializeArg.dataBufferLength = SpyControllerDataBufferLength;

        size_t bufferSize = nn::spy::SpyController::GetRequiredMemorySize(initializeArg);
        g_WorkBufferForSpyController = (bufferSize == 0) ? nullptr : nns::atk::Allocate(bufferSize);
        NN_ABORT_UNLESS(bufferSize == 0 || g_WorkBufferForSpyController != nullptr);

        g_SpyController.Initialize(initializeArg, g_WorkBufferForSpyController, bufferSize);

        g_AppFrameCount = 0;
    }

    void FinalizeSpy()
    {
        g_SpyController.Finalize();

        nns::atk::Free(g_WorkBufferForSpyController);
        g_WorkBufferForSpyController = nullptr;
    }

    void InitializeSpyAtk()
    {
        // バッファを確保して、AtkSpyModule を初期化
        nn::spy::atk::AtkSpyModule::InitializeArg initializeArg;
        initializeArg.isPerformanceMetricsEnabled = true;
        initializeArg.isWaveformEnabled = true;
        initializeArg.isAtkProfilesEnabled = true;
        size_t bufferSize = nn::spy::atk::AtkSpyModule::GetRequiredMemorySize(initializeArg);
        g_WorkBufferForAtkSpyModule = (bufferSize == 0) ? nullptr : nns::atk::Allocate(bufferSize);
        g_AtkSpyModule.Initialize(initializeArg, g_WorkBufferForAtkSpyModule, bufferSize);

        // 出力波形の読み出し用のバッファを確保
        g_CircularBufferSinkBufferSize = GetCircularBufferSinkBufferSize();
        g_CircularBufferSinkBuffer = nns::atk::Allocate(g_CircularBufferSinkBufferSize);

        // AtkSpyModule の登録
        g_SpyController.RegisterModule(g_AtkSpyModule);
    }

    void FinalizeSpyAtk()
    {
        // AtkSpyModule の登録解除
        g_SpyController.UnregisterModule(g_AtkSpyModule);

        // AtkSpyModule の終了処理
        g_AtkSpyModule.Finalize();

        // AtkSpyModule のバッファ解放
        nns::atk::Free(g_WorkBufferForAtkSpyModule);
        g_WorkBufferForAtkSpyModule = nullptr;

        // 出力波形の読み出し用のバッファを解放
        nns::atk::Free(g_CircularBufferSinkBuffer);
        g_CircularBufferSinkBuffer = nullptr;
    }

    // Spy ツールと通信を開始します。
    void StartSpy()
    {
        nn::spy::SpyController::OpenArg openArg;
        bool openResult = g_SpyController.Open(openArg);
        NN_ABORT_UNLESS(openResult);
    }

    // Spy ツールとの通信を終了します。
    void StopSpy()
    {
        g_SpyController.Close();
    }

    // AtkSpyModule を使って、サウンドの再生状態の情報を Spy ツールに送信します。
    void PushSoundState()
    {
        g_AtkSpyModule.PushSoundState(GetAtkSoundArchivePlayer());
    }

    // AtkSpyModule を使って、パフォーマンス情報を Spy ツールに送信します。
    void PushPerformanceMetrics()
    {
        g_AtkSpyModule.PushPerformanceMetrics();
    }

    // AtkSpyModule を使って、出力波形を Spy ツールに送信します。
    void PushWaveform()
    {
        if (g_AtkSpyModule.IsWaveformEnabled())
        {
            // nn::atk は nn::audio::RequestUpdateAudioRenderer() の
            // 完了時刻を提供しないので、波形データ取得の直前の時刻で代用します。
            nn::os::Tick tick = nn::os::GetSystemTick();

            // 波形データを取得します。
            size_t readBytes = ReadCircularBufferSink(
                g_CircularBufferSinkBuffer,
                g_CircularBufferSinkBufferSize);

            // 波形データを送信します。
            g_AtkSpyModule.PushWaveform(
                g_CircularBufferSinkBuffer,
                readBytes,
                tick);
        }
    }

    // AtkSpyModule を使って、nn::atk パフォーマンス情報を Spy ツールに送信します。
    void PushAtkProfiles()
    {
        g_AtkSpyModule.PushAtkProfiles();
    }

    // AtkSpyModule を使って、シーケンス変数情報を Spy ツールに送信します。
    void PushSequenceVariable()
    {
        g_AtkSpyModule.PushSequenceVariable(GetAtkSoundArchivePlayer());
    }

    // AtkSpyModule を使って、ストリームサウンド情報を Spy ツールに送信します。
    void PushStreamSoundInfo()
    {
        g_AtkSpyModule.PushStreamSoundInfo(GetAtkSoundArchivePlayer());
    }

    // DebugModule を使って、Spy のデバッグ情報を Spy ツールに送信します。
    void PushSpyDebugInfo()
    {
        // データバッファ使用量を送信します。
        g_SpyController.GetDebugModule().PushDataBufferUsage();
    }

    void PrintUsage()
    {
        NNS_LOG("--------------------------------------------------\n");
        NNS_LOG(" SpyAtk:\n");
        NNS_LOG("\n");
#if !defined(NN_BUILD_CONFIG_SPY_ENABLED)
        NNS_LOG(" Warning!\n");
        NNS_LOG("   nn::spy is disabled.\n");
        NNS_LOG("   nn::spy is only available on Debug or Develop build.\n");
        NNS_LOG("\n");
#endif
        NNS_LOG(" [A]                StartSound SEQ  (SEQ_MARIOKART)\n");
        NNS_LOG(" [X]                StartSound WSD  (SE_YOSHI)\n");
        NNS_LOG(" [Y]                StartSound STRM (STRM_MARIOKART)\n");
        NNS_LOG(" [B]                Stop Sound\n");
        NNS_LOG(" [RIGHT]            HoldSound SEQ  (SEQ_MARIOKART)\n");
        NNS_LOG(" [UP]               HoldSound WSD  (SE_YOSHI)\n");
        NNS_LOG(" [LEFT]             HoldSound STRM (STRM_MARIOKART)\n");
        NNS_LOG(" [L]                Print Usage\n");
        NNS_LOG(" [+/Start][Space]   Exit Application\n");
        NNS_LOG("--------------------------------------------------\n");
    }

    void PrintMessageLine(const char* message)
    {
        NNS_LOG(message);
        NNS_LOG("\n");
        nn::spy::LogModule::Write(g_SpyController, message);
    }

    void PrintMessage(const char* message)
    {
        NNS_LOG(message);
        nn::spy::LogModule::Write(g_SpyController, message);
    }

    void PrintResult(bool result)
    {
        const char* strResult = result ? " (success)" : " (failure)";
        NNS_LOG(strResult);
        NNS_LOG("\n");
        nn::spy::LogModule::Write(g_SpyController, strResult);
    }

    void Mainloop()
    {
        PrintUsage();

        while (NN_STATIC_CONDITION(true))
        {
            // アプリケーションフレームの開始時間を記録します。
            g_SpyController.SetCurrentApplicationFrame(g_AppFrameCount++);

            nns::atk::UpdateHidDevices();

            // サウンド再生開始
            if (nns::atk::IsTrigger<nn::hid::DebugPadButton::A>())
            {
                PrintMessage("StartSound(SEQ_MARIOKART)  ...");
                bool result = StartAtkSequenceSound();
                PrintResult(result);
            }
            if (nns::atk::IsTrigger<nn::hid::DebugPadButton::X>())
            {
                PrintMessage("StartSound(SE_YOSHI)       ...");
                bool result = StartAtkWaveSound();
                PrintResult(result);
            }
            if (nns::atk::IsTrigger<nn::hid::DebugPadButton::Y>())
            {
                PrintMessage("StartSound(STRM_MARIOKART) ...");
                bool result = StartAtkStreamSound();
                PrintResult(result);
            }

            // サウンド再生開始、または再生を続行
            if (nns::atk::IsHold<nn::hid::DebugPadButton::Right>())
            {
                PrintMessage("HoldSound(SEQ_MARIOKART)   ...");
                bool result = HoldAtkSequenceSound();
                PrintResult(result);
            }
            else if (nns::atk::IsHold<nn::hid::DebugPadButton::Up>())
            {
                PrintMessage("HoldSound(SE_YOSHI)        ...");
                bool result = HoldAtkWaveSound();
                PrintResult(result);
            }
            else if (nns::atk::IsHold<nn::hid::DebugPadButton::Left>())
            {
                PrintMessage("HoldSound(STRM_MARIOKART)  ...");
                bool result = HoldAtkStreamSound();
                PrintResult(result);
            }

            // サウンド停止
            if (nns::atk::IsTrigger<nn::hid::DebugPadButton::B>())
            {
                PrintMessageLine("StopSound");
                StopAtkSound();
            }

            // Usage 表示
            if (nns::atk::IsTrigger<nn::hid::DebugPadButton::L>())
            {
                PrintUsage();
            }

            // 終了
            if (nns::atk::IsTrigger<nn::hid::DebugPadButton::Start>())
            {
                PrintMessageLine("Exit");
                return;
            }

            // サウンドアーカイブプレイヤーの更新処理を行います。
            UpdateAtkSoundArchivePlayer();

            // サウンドの再生状態を送信します。
            PushSoundState();

            // パフォーマンス情報を送信します。
            PushPerformanceMetrics();

            // 出力波形を送信します。
            PushWaveform();

            // nn::atk パフォーマンス情報を送信します。
            PushAtkProfiles();

            // シーケンス変数情報を送信します。
            PushSequenceVariable();

            // ストリームサウンド情報を送信します
            PushStreamSoundInfo();

            // Spy のデバッグ情報を送信します。
            PushSpyDebugInfo();

            // Vsync の代わり
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
    }

} // namespace {anonymous}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();

    InitializeHtcs();

    // Atk を初期化します。
    // プロファイラを有効にしています。
    InitializeAtk();
    LoadAtkData();

    // Spy の初期化して開始します。
    InitializeSpy();
    InitializeSpyAtk();
    StartSpy();

    // メインループです。
    Mainloop();

    // Spy を停止して、終了します。
    StopSpy();
    FinalizeSpyAtk();
    FinalizeSpy();

    // Atk を終了します。
    FinalizeAtk();

    FinalizeHtcs();

    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

