﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include "MulticastSend.h"

namespace nns {
namespace socket {

//***************************************************************
//*
//*                    G L O B A L s
//*
//***************************************************************

const size_t     MulticastSend::SendBufferSize = 1500 + 1;

//***************************************************************
//*
//* MulticastSend
//*
//***************************************************************

MulticastSend::MulticastSend() NN_NOEXCEPT : m_SentTotalBytes(0),
                                             m_UdpSendSocket(-1),
                                             m_pSendData(nullptr),
                                             m_IsInitialized(false)
{
    memset(&m_MulitcastGroupAddr, 0, sizeof(m_MulitcastGroupAddr));
    memset(&m_SockAddr, 0, sizeof(m_SockAddr));
}

MulticastSend::~MulticastSend() NN_NOEXCEPT
{
    if (m_IsInitialized == true)
    {
        Finalize();
    }

    memset(&m_MulitcastGroupAddr, 0, sizeof(m_MulitcastGroupAddr));
    memset(&m_SockAddr, 0, sizeof(m_SockAddr));
}

uint64_t MulticastSend::GetTotalBytesSent() NN_NOEXCEPT
{
    return(m_SentTotalBytes);
}

int MulticastSend::Initialize(const char* pMulitcastGroupAddr, const uint16_t multicastUDPPort, const char* pNetworkInterface) NN_NOEXCEPT
{
    nn::socket::InAddr  ifaceAddr;
    size_t          ipAddrSize = 0;
    int             udpSocket = -1;
    int             rc = 0;
    int             ret = 0;
    u_char          optionTtl = 1;
    u_char          optionLoop = 0;

    do
    {
        // if already initialized
        if (m_IsInitialized == true)
        {
            NN_LOG("Object is already initialized!\n");
            break;
        }

        // Did we get a Multicast Group Address?
        if (pMulitcastGroupAddr == nullptr)
        {
            NN_LOG("Multicast Group Address IP Address (char) is required\n");
            ret = -1;
            break;
        }

        ipAddrSize = strnlen(pMulitcastGroupAddr, (size_t) (sizeof(m_MulitcastGroupAddr) - 1));
        if (ipAddrSize < 1 || ipAddrSize >= sizeof(m_MulitcastGroupAddr) - 1)
        {
            NN_LOG("Multicast Group Address IP Address (size_t) has illegal size [%ld]\n", ipAddrSize);
            ret = -1;
            break;
        }

        // Save Multicast Group Address
        memset(m_MulitcastGroupAddr, 0, sizeof(m_MulitcastGroupAddr));
        memcpy(m_MulitcastGroupAddr, pMulitcastGroupAddr, ipAddrSize);

        // Save the Multicast Group Port
        m_MulticastUDPPort = multicastUDPPort;

        // Create a UDP Socket
        udpSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Ip);
        if(udpSocket < 0)
        {
            NN_LOG("nn::socket::Socket(): Failed - (errno: %d)\n", nn::socket::GetLastError() );
            ret = -1;
            break;
        }

        // Initialize Socket Addresses
        memset(&m_SockAddr, 0, sizeof(m_SockAddr));

        // Establish UDP Source Address
        m_SockAddr.sin_family      = nn::socket::Family::Af_Inet;
        m_SockAddr.sin_port        = nn::socket::InetHtons(m_MulticastUDPPort);

        // Set Multicast Group IP Address
        rc = nn::socket::InetPton(nn::socket::Family::Af_Inet, m_MulitcastGroupAddr, &m_SockAddr.sin_addr.S_addr);
        if (rc < 0)
        {
            NN_LOG( "nn::socket::InetPton: Failed to set Multicast Group Address: [%s], Errno: [%d]\n",
                                    m_MulitcastGroupAddr, nn::socket::GetLastError() );
            ret = -1;
            break;
        }

        // TTL (Time To Live): Restrict sent UDP datagrams to the IP Multicast group to (this) LAN
        optionTtl = 1;
        rc = nn::socket::SetSockOpt(udpSocket, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Multicast_Ttl, &optionTtl, sizeof(optionTtl));
        if (rc < 0)
        {
            NN_LOG( "nn::socket::SetSockOpt: (nn::socket::Option::Ip_Multicast_Ttl) failed - (errno: %d)\n", nn::socket::GetLastError());
            ret = -1;
            break;
        }

        // (LOOPBACK): Do not echo back the UDP datagram that what we sent to the IP multicast Group
        optionLoop = 0;
        rc = nn::socket::SetSockOpt(udpSocket, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Multicast_Loop, &optionLoop, sizeof(optionLoop));
        if (rc < 0)
        {
            NN_LOG( "nn::socket::SetSockOpt: (nn::socket::Option::Ip_Multicast_Loop) failed - (errno: %d)\n", nn::socket::GetLastError());
            ret = -1;
            break;
        }

        // Set the Multicast Group outbound Interface
        if (pNetworkInterface != nullptr)
        {
            // Translate the Multicast IP Network Interface address to a network address
            rc = nn::socket::InetPton(nn::socket::Family::Af_Inet, pNetworkInterface, &ifaceAddr);
            if (rc != 1)
            {
                NN_LOG( "nn::socket::InetPton(): Failed to set Multicast IP Network Interface address: [%s] (errno: %d)\n",
                                                  pNetworkInterface, nn::socket::GetLastError() );
                ret = -1;
                break;
            }

            // Set the outbound Network Interface for Multicast traffic
            rc = nn::socket::SetSockOpt(udpSocket, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Multicast_If, &ifaceAddr, sizeof(ifaceAddr));
            if (rc < 0)
            {
                NN_LOG( "nn::socket::SetSockOpt: (nn::socket::Option::Ip_Multicast_If) failed - (errno: %d)\n", nn::socket::GetLastError() );
                ret = -1;
                break;
            }
        }

        // Allocate Send Buffer
        m_pSendData = new char[SendBufferSize];
        if (m_pSendData == nullptr)
        {
            NN_LOG( "new failed - allocating [%d] bytes of memory!\n", SendBufferSize);
            ret = -1;
            break;
        }

        // Initialize the Send Buffer memory
        memset(m_pSendData, 0, SendBufferSize);

        // Save created socket
        m_UdpSendSocket = udpSocket;

        // Object has been initialized
        m_IsInitialized = true;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (ret != 0)
    {
        if (udpSocket > -1)
        {
            nn::socket::Close(udpSocket);
            udpSocket = -1;
            m_UdpSendSocket = -1;
        }

        if (m_pSendData != nullptr)
        {
            delete [] m_pSendData;
            m_pSendData = nullptr;
        }

        // NOT initialized
        m_IsInitialized = false;
    }

    return(ret);

}     // NOLINT(impl/function_size)

int MulticastSend::Finalize() NN_NOEXCEPT
{
    int   ret = 0;

    if (m_IsInitialized == true)
    {
        if (m_UdpSendSocket > -1)
        {
            nn::socket::Close(m_UdpSendSocket);
            m_UdpSendSocket = -1;
        }

        if (m_pSendData != nullptr)
        {
            delete [] m_pSendData;
            m_pSendData = nullptr;
        }
        m_IsInitialized = false;
    }

    return(ret);
}

int MulticastSend::SendMessages(const int totalTimeInSeconds) NN_NOEXCEPT
{
    nn::socket::SockLenT            sendAddrLen = 0;
    ssize_t              rc1 = -1;
    size_t               outputSize = 0;
    int                  ret = 0;
    int                  len = 0;
    int                  timeLeft = -1;

    do
    {
        if (m_IsInitialized == false)
        {
            NN_LOG("Object is not initialized!\n");
            ret = -1;
            break;
        }

        NN_LOG( "[BEGIN] - Sending UDP packets to Multicast Group [%s:%d] for the next [%d] seconds\n",
                m_MulitcastGroupAddr, m_MulticastUDPPort, totalTimeInSeconds);

        for(timeLeft = totalTimeInSeconds; timeLeft > 0; timeLeft--)
        {
            // Initialize Payload Message
            memset(m_pSendData, 0, SendBufferSize);
            len = snprintf(m_pSendData, SendBufferSize, "Hello #%03d!", totalTimeInSeconds - timeLeft);
            if (len >= SendBufferSize - 1)
            {
                NN_LOG( "snprintf failed to save [Hello!] to buffer of length [%d]\n", SendBufferSize);
                ret = -1;
                break;
            }
            outputSize = strlen(m_pSendData);

            // Read from Multicast Group
            sendAddrLen = sizeof(m_SockAddr);
            rc1 = nn::socket::SendTo(m_UdpSendSocket, m_pSendData, outputSize, 0,
                                     reinterpret_cast<sockaddr *>(&m_SockAddr), sendAddrLen);
            if (rc1 < 0)
            {
                NN_LOG( "SendTo: Failed to Send Multicast Data - Errno: [%d]\n", nn::socket::GetLastError());
                ret = -1;
                break;
            }

            // Save total bytes
            m_SentTotalBytes = m_SentTotalBytes + rc1;

            // Display data read from multicast group
            NN_LOG( "Multicast Group [%s:%d] - SendTo: Len [%lu], Data [%.*s]\n",
                    m_MulitcastGroupAddr, m_MulticastUDPPort,
                    rc1, rc1, m_pSendData);

            // Pause between sends
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        NN_LOG( "[END] - Finished Sending UDP packets to Multicast Group [%s:%d]\n",
                m_MulitcastGroupAddr, m_MulticastUDPPort);

    } while (NN_STATIC_CONDITION(false));

    return(ret);
}

}}   // Namespace nns / socket
