﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{PctlStereoVision.cpp,PageSamplePctlStereoVision}
 *
 * @brief
 * ペアレンタルコントロール機能を用いた立体視機能の制限確認と設定変更を行うサンプルプログラム
 */

/**
 * @page PageSamplePctlStereoVision 立体視機能の利用制限
 * @tableofcontents
 *
 * @brief
 * ペアレンタルコントロール機能を用いた立体視機能の制限確認と設定変更を行うサンプルプログラムです。
 *
 * @section PageSamplePctlStereoVision_SectionBrief 概要
 * このサンプルでは、 nn::pctl::ConfirmStereoVisionPermission を使用した立体視機能の制限確認と、
 * nn::pctl::SetStereoVisionRestriction などを利用した機能制限の変更を行うシンプルなサンプルです。
 *
 * @section PageSamplePctlStereoVision_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/PctlStereoVision Samples/Sources/Applications/PctlStereoVision @endlink 以下にあります。
 *
 * @section PageSamplePctlStereoVision_SectionNecessaryEnvironment 必要な環境
 * 本プログラムは NX プラットフォームでのみ実行が可能です。
 *
 * @section PageSamplePctlStereoVision_SectionHowToOperate 操作方法
 * 最初の画面(メインシーン)からは2つのシーンに遷移可能です。@n
 * ここでAボタンを入力すると、立体視機能の制限があるかどうかを確認し、制限チェックの結果
 * 機能の利用が可能であればサブシーンに遷移します。
 * 利用が制限されている場合は、エラーが表示され遷移が行われません。@n
 * メインシーンでYボタンを入力すると、機能制限の設定変更を行うシーンに遷移し、
 * その中でAボタンを入力すると制限の状態を切り替えることができます。
 * 設定シーンに入る際、および設定変更を行う際に設定変更が
 * 許可されなかった場合はエラーが表示されます。
 *
 * 各サブシーンでBを入力するとメインシーンに戻り、メインシーンで
 * Bを入力するとアプリケーションを終了します。
 *
 * @section PageSamplePctlStereoVision_SectionPrecaution 注意事項
 * 実際に制限されているかどうかを確認する場合は、アプリケーション起動前に
 * HOMEメニューや DevMenuCommand 等でペアレンタルコントロール設定を行う必要があります。
 *
 * @section PageSamplePctlStereoVision_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSamplePctlStereoVision_SectionDetail 解説
 * 本サンプルプログラムは nn::pctl::ConfirmStereoVisionPermission を利用した
 * 立体視機能に関する制限の確認と、 nn::pctl::SetStereoVisionRestriction などを
 * 利用した立体視機能制限の変更を行うデモです。
 *
 * このサンプルプログラムではメインシーンとサブシーン、設定シーンの3つのシーンを持ちます。
 * このうちサブシーンは立体視機能の利用を想定したシーンであり、
 * シーンへの遷移前に制限確認を実施しています。
 * 事前に設定されたペアレンタルコントロール設定と立体視機能の制限設定に基づき、
 * 制限が発生する場合はサブシーンへの遷移を行いません。
 * また、設定シーンでは立体視機能制限の設定変更を行います。
 *
 * 具体的には各関数で以下の処理を行っています。
 *
 * - nnMain: 初期化・終了処理とメインループ開始処理
 * - MainLoop: メインループ処理
 *   - ユーザー入力によってシーン切り替えも行います。
 * - ChangeSceneToMain: メインシーンへの遷移処理
 * - ChangeSceneToSub: サブシーンへの遷移処理
 *   - サブシーンへ遷移するための制限確認を行います。
 * - ChangeSceneToSettings: 設定シーンへの遷移処理
 *   - 設定変更が可能かどうかの確認を行います。
 * - ChangeRestrictionValue: 設定変更の処理
 *   - 設定変更前の暗証番号入力などの確認を行います。
 * - PrintMain: メインシーンの描画処理
 * - PrintSub: サブシーンの描画処理
 * - PrintSettings: 設定シーンの描画処理
 */

// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>

// ユーザー入力取得用
#include <nns/hid.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/util/util_BitFlagSet.h>

// 画面描画用
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

// ペアレンタルコントロールサービス利用
#include <nn/pctl/pctl_ApiStereoVision.h>

// エラー表示用
#include <nn/err.h>

namespace
{
    // シーン定数
    enum SceneId
    {
        // メインシーン(起動時のシーン)
        SceneId_Main = 0,
        // サブシーン(制限をかけるべきシーン)
        SceneId_SubMain,
        // 設定シーン
        SceneId_Settings
    };

    // 状態によって出力関数を切り替えるための変数
    SceneId s_CurrentScene = SceneId_Main;

    // 現在の設定値を示す値
    bool s_CurrentRestrictionValue = false;

    // 設定変更完了メッセージを表示する残りフレーム数
    int s_ShowFramesForChangeCompleteMessage = 0;
} // namespace

//------------------------------------------------------------------------------
//  出力関数 - メインシーン
//------------------------------------------------------------------------------
void PrintMain(nn::gfx::util::DebugFontTextWriter& writer, int frame)
{
    nn::util::Color4u8Type color0 = { { 255, 255, 255, 255 } };
    nn::util::Color4u8Type color1 = { { 0, 255, 0, 255 } };

    writer.SetScale(1.0f, 1.0f);

    writer.SetTextColor(color0);
    writer.SetCursor(10, 10);
    writer.Print("PctlStereoVision - Main scene");

    if ((frame % 60) < 30)
    {
        writer.Print("■");
    }

    writer.SetTextColor(color1);
    writer.SetCursor(100, 100);
    writer.Print("Press A to enter sub scene for using stereo vision.\n");
    writer.Print("Press Y to enter sub scene for settings for stereo vision.\n");
    writer.Print("Press B to exit application.\n");
}

//------------------------------------------------------------------------------
//  出力関数 - サブシーン
//------------------------------------------------------------------------------
void PrintSubMain(nn::gfx::util::DebugFontTextWriter& writer, int frame)
{
    nn::util::Color4u8Type color0 = { { 255, 128, 128, 255 } };
    nn::util::Color4u8Type color1 = { { 0, 255, 0, 255 } };

    writer.SetScale(1.0f, 1.0f);

    writer.SetTextColor(color0);
    writer.SetCursor(10, 10);
    writer.Print("PctlStereoVision - Sub scene for stereo vision");

    if ((frame % 60) < 30)
    {
        writer.Print("■");
    }

    writer.SetTextColor(color1);
    writer.SetCursor(100, 100);
    writer.Print("Press B to back to main scene.\n");
}

//------------------------------------------------------------------------------
//  出力関数 - 設定シーン
//------------------------------------------------------------------------------
void PrintSettings(nn::gfx::util::DebugFontTextWriter& writer, int frame)
{
    nn::util::Color4u8Type color0 = { { 128, 255, 255, 255 } };
    nn::util::Color4u8Type color1 = { { 0, 255, 0, 255 } };
    nn::util::Color4u8Type color2 = { { 128, 128, 255, 255 } };

    writer.SetScale(1.0f, 1.0f);

    writer.SetTextColor(color0);
    writer.SetCursor(10, 10);
    writer.Print("PctlStereoVision - Settings scene for stereo vision\n");
    writer.Print("  Current value: %s", s_CurrentRestrictionValue ? "Restricted" : "Not restricted");

    if ((frame % 60) < 30)
    {
        writer.Print("■");
    }

    writer.SetTextColor(color1);
    writer.SetCursor(100, 100);
    writer.Print("Press A to change value to '%s'.\n", s_CurrentRestrictionValue ? "Not restricted" : "Restricted");
    writer.Print("Press B to back to main scene.\n");

    if (s_ShowFramesForChangeCompleteMessage > 0)
    {
        writer.SetTextColor(color2);
        writer.SetCursor(150, 200);
        writer.Print("Change settings completed.");
    }
}

//------------------------------------------------------------------------------
//  メインシーンへの切り替え
//------------------------------------------------------------------------------
void ChangeSceneToMain()
{
    s_CurrentScene = SceneId_Main;

    // メインシーンに戻った時点で立体視機能の利用許可状態をリセットすることにより、
    // アプリケーション中断からの復帰時に立体視機能の制限確認を行わないようにする
    nn::pctl::ResetConfirmedStereoVisionPermission();
}

//------------------------------------------------------------------------------
//  サブシーンへの切り替え
//------------------------------------------------------------------------------
void ChangeSceneToSub()
{
    // 制限を行うシーンに入る前に必ずシーンに入れるかどうかを確認する
    // ※ アプリケーションがバックグラウンドにいる間に状態が変化することもあるので
    //    値をキャッシュせずに毎回呼び出しを行う
    nn::Result result = nn::pctl::ConfirmStereoVisionPermission();
    if (result.IsSuccess())
    {
        s_CurrentScene = SceneId_SubMain;
    }
    else
    {
        // 制限されている旨を表示する
        nn::err::ShowError(result);
    }
}

//------------------------------------------------------------------------------
//  設定シーンへの切り替え
//------------------------------------------------------------------------------
void ChangeSceneToSettings()
{
    // 設定を行うシーンに入る前に必ずシーンに入れるかどうかを確認する
    nn::Result result = nn::pctl::ConfirmStereoVisionRestrictionConfigurable();
    if (result.IsSuccess())
    {
        s_CurrentScene = SceneId_Settings;
        // 設定値は毎フレーム取得しないように一旦ここで取得する
        s_CurrentRestrictionValue = nn::pctl::GetStereoVisionRestriction();
    }
    else
    {
        // エラーが発生した旨を表示する
        nn::err::ShowError(result);
    }
}

//------------------------------------------------------------------------------
//  設定値の変更
//------------------------------------------------------------------------------
void ChangeRestrictionValue()
{
    // 設定変更を行う直前に必ず確認する
    // (暗証番号の確認が行われます。)
    nn::Result result = nn::pctl::RequestStereoVisionRestrictionConfigurationPermission();
    if (result.IsSuccess())
    {
        // ここでは設定をトグルさせて変更する
        nn::pctl::SetStereoVisionRestriction(!s_CurrentRestrictionValue);

        // 反映後の値を取得する
        s_CurrentRestrictionValue = nn::pctl::GetStereoVisionRestriction();

        // 設定変更完了メッセージを表示するための値を設定
        s_ShowFramesForChangeCompleteMessage = 120;
    }
    else if (nn::pctl::ResultConfigurationCanceled::Includes(result))
    {
        // キャンセルの場合は特に何も表示しない
    }
    else
    {
        // それ以外の場合はエラーが発生した旨を表示する
        nn::err::ShowError(result);
    }
}

//------------------------------------------------------------------------------
//  計算処理コールバック
//------------------------------------------------------------------------------
struct CalculateCallbackUserData
{
    nns::hid::ControllerManager* pControllerManager;
    int* pFrame;
    bool* pIsExited;
    nn::gfx::util::DebugFontTextWriter* pWriter;
};

//------------------------------------------------------------------------------
//  ボタン押下判定
//------------------------------------------------------------------------------
bool HasAnyControllerButtonsDown(nns::hid::ControllerManager& controllerManager, nns::hid::ButtonSet buttons)
{
    nns::hid::Controller* pController = controllerManager.GetController(nns::hid::ControllerId_DebugPad, 0);
    if (pController && pController->HasAnyButtonsDown(buttons))
    {
        return true;
    }
    pController = controllerManager.GetController(nns::hid::ControllerId_GamePad, 0);
    return (pController && pController->HasAnyButtonsDown(buttons));
}

void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    CalculateCallbackUserData* pCallbackUserData = reinterpret_cast<CalculateCallbackUserData*>(pUserData);

    // フォント表示
    switch (s_CurrentScene)
    {
    case SceneId_Main:
        PrintMain(*pCallbackUserData->pWriter, *pCallbackUserData->pFrame);
        break;
    case SceneId_SubMain:
        PrintSubMain(*pCallbackUserData->pWriter, *pCallbackUserData->pFrame);
        break;
    case SceneId_Settings:
        PrintSettings(*pCallbackUserData->pWriter, *pCallbackUserData->pFrame);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 入力状態の更新
    pCallbackUserData->pControllerManager->Update();

    // シーン別の状態確認処理
    switch (s_CurrentScene)
    {
    case SceneId_Main:
        if (HasAnyControllerButtonsDown(*pCallbackUserData->pControllerManager, nns::hid::Button::B::Mask))
        {
            (*pCallbackUserData->pIsExited) = true;
        }
        else if (HasAnyControllerButtonsDown(*pCallbackUserData->pControllerManager, nns::hid::Button::A::Mask))
        {
            ChangeSceneToSub();
        }
        else if (HasAnyControllerButtonsDown(*pCallbackUserData->pControllerManager, nns::hid::Button::Y::Mask))
        {
            ChangeSceneToSettings();
        }
        break;
    case SceneId_SubMain:
        if (HasAnyControllerButtonsDown(*pCallbackUserData->pControllerManager, nns::hid::Button::B::Mask))
        {
            ChangeSceneToMain();
        }
        break;
    case SceneId_Settings:
        if (HasAnyControllerButtonsDown(*pCallbackUserData->pControllerManager, nns::hid::Button::A::Mask))
        {
            ChangeRestrictionValue();
        }
        else if (HasAnyControllerButtonsDown(*pCallbackUserData->pControllerManager, nns::hid::Button::B::Mask))
        {
            ChangeSceneToMain();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (s_ShowFramesForChangeCompleteMessage > 0)
    {
        --s_ShowFramesForChangeCompleteMessage;
    }

    ++(*pCallbackUserData->pFrame);
    if (*pCallbackUserData->pFrame == 120)
        (*pCallbackUserData->pFrame) = 0;
}

//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    nn::gfx::util::DebugFontTextWriter* pWriter = reinterpret_cast<nn::gfx::util::DebugFontTextWriter*>(pUserData);

    // コマンド生成
    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        // レンダーターゲット、ビューポートシザー設定
        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target, 0.1f, 0.1f, 0.1f, 1.0f, nullptr);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        // デバッグフォント用のコマンド生成
        pWriter->Draw(rootCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex);
}

//------------------------------------------------------------------------------
//  メインループ処理
//------------------------------------------------------------------------------
void MainLoop(nns::hid::ControllerManager& controllerManager, nns::gfx::GraphicsFramework& gfw, nn::gfx::util::DebugFontTextWriter& writer)
{
    int frame = 0;
    bool isExited = false;

    // フレームワーク設定
    CalculateCallbackUserData calculateUserData;
    calculateUserData.pControllerManager = &controllerManager;
    calculateUserData.pFrame = &frame;
    calculateUserData.pIsExited = &isExited;
    calculateUserData.pWriter = &writer;

    gfw.SetCalculateCallback(CalculateCallback, &calculateUserData);
    gfw.SetMakeCommandCallback(MakeCommandCallback, &writer);

    // 毎フレームのレンダリング
    while (!isExited)
    {
        gfw.ProcessFrame();
    }
    gfw.QueueFinish();
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // FS を使用する場合はフレームワークよりも前に初期化します

    nns::hid::ControllerManager controllerManager;
    nns::hid::util::SetControllerManagerWithDefault(&controllerManager);

    // フレームワーク初期化
    const int BufferCount = 2;
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetBufferCount(BufferCount);
    fwInfo.SetSwapChainBufferCount(BufferCount);
    nns::gfx::GraphicsFramework gfw;
    gfw.Initialize(fwInfo);

    // デバッグフォント初期化
    nns::gfx::GraphicsFramework::DebugFontTextWriter writer;
    gfw.InitializeDebugFontTextWriter(&writer, 1024, BufferCount);

    // 初期シーンとしてメインシーンに切り替え
    ChangeSceneToMain();

    // メインループ
    MainLoop(controllerManager, gfw, writer.object);

    // デバッグフォント終了
    gfw.FinalizeDebugFontTextWriter(&writer);

    // フレームワーク終了
    gfw.Finalize();
}
