﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsVirtualAddressMemory.cpp,PageSampleOsVirtualAddressMemory}
 *
 * @brief
 *  仮想アドレスメモリ機能のサンプルプログラム
 */

/**
 * @page PageSampleOsVirtualAddressMemory VirtualAddressMemory
 * @tableofcontents
 *
 * @brief
 *  仮想アドレスメモリ機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsVirtualAddressMemory_SectionBrief 概要
 *  ここでは、仮想アドレスメモリ機能を使ったサンプルプログラムの説明をします。
 *
 *  仮想アドレスメモリ機能の使い方については、仮想アドレスメモリ機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsVirtualAddressMemory_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsVirtualAddressMemory Samples/Sources/Applications/OsVirtualAddressMemory @endlink 以下にあります。
 *
 * @section PageSampleOsVirtualAddressMemory_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsVirtualAddressMemory_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsVirtualAddressMemory_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsVirtualAddressMemory_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsVirtualAddressMemory_SectionDetail 解説
 *
 * @subsection PageSampleOsVirtualAddressMemory_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsVirtualAddressMemory.cpp
 *  @includelineno OsVirtualAddressMemory.cpp
 *
 * @subsection PageSampleOsVirtualAddressMemory_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて VirtualAddressMemorySample() を呼び出し
 *  - VirtualAddressMemorySample() にて仮想アドレス空間を確保
 *  - 確保した仮想アドレス空間に対して物理メモリを割り当て
 *  - 割り当てた物理メモリの一部を解放
 *  - 物理メモリが割り当て済みの領域も含め、確保した仮想アドレス空間全体に物理メモリを割り当て
 *  - システムが仮想アドレスメモリのために使用しているメモリサイズを表示
 *  - アドレス空間を解放
 *
 *  VirtualAddressMemorySample() では、最初に 2MiB の仮想アドレス空間を確保します。
 *  確保した仮想アドレス空間に対して、以下の順で物理メモリの割り当て・解放を行います。
 *
 *  - 空間の中央半分 (1MiB 分) に物理メモリを割り当て
 *  - 割り当てた物理メモリの一部（中央 8KiB 分）を解放
 *  - 空間全体 (2MiB) に物理メモリを割り当て
 *
 *  物理メモリの割り当ては、確保した仮想アドレス空間内であれば空間の先頭以外の場所に対しても実施可能です。
 *  また、割り当てた物理メモリの一部のみを解放することも可能です。
 *
 *  GetVirtualAddressMemoryResourceUsage() を呼び出すことで、
 *  システムが仮想アドレスメモリのために使用しているメモリサイズを取得できます。
 *  詳細は OS ライブラリマニュアルの「SystemResourceSize の値の決定」を参照してください。
 *
 *  最後に仮想アドレス空間を解放します。
 *  このとき、空間内部に割り当て済みの物理メモリがあればあわせて解放されます。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *  なお、実行結果のうち、メモリのアドレスについては実行環境によって異なります。
 *
 *  @verbinclude  OsVirtualAddressMemory_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
}

//-----------------------------------------------------------------------------

void VirtualAddressMemorySample()
{
    const size_t RegionSize = 2 * 1024 * 1024;
    uintptr_t region;

    // アドレス空間を確保
    auto result = nn::os::AllocateAddressRegion(&region, RegionSize);
    if (result.IsFailure())
    {
        // アドレス空間の確保に失敗
        return;
    }

    NN_LOG("AllocateAddressRegion: start=0x%zx, size=0x%zx\n", region, RegionSize);

    // 確保した空間の真ん中半分に物理メモリを割り当て
    result = nn::os::AllocateMemoryPages(region + RegionSize / 4, RegionSize / 2);
    if (result.IsFailure())
    {
        // 物理メモリの割り当てに失敗
        return;
    }

    NN_LOG("AllocateMemoryPages:   start=0x%zx, size=0x%zx\n", region + RegionSize / 4, RegionSize / 2);

    // 割り当てた物理メモリの一部を解放
    result = nn::os::FreeMemoryPages(region + RegionSize / 2 - nn::os::MemoryPageSize, nn::os::MemoryPageSize * 2);
    if (result.IsFailure())
    {
        // 物理メモリの解放に失敗
        return;
    }

    NN_LOG("FreeMemoryPages:       start=0x%zx, size=0x%zx\n", region + RegionSize / 2 - nn::os::MemoryPageSize, nn::os::MemoryPageSize * 2);

    // 確保した空間全体に物理メモリを割り当て
    // 物理メモリが割り当て済みの領域が含まれていても問題ない
    result = nn::os::AllocateMemoryPages(region, RegionSize);
    if (result.IsFailure())
    {
        // 物理メモリの割り当てに失敗
        return;
    }

    NN_LOG("AllocateMemoryPages:   start=0x%zx, size=0x%zx\n", region, RegionSize);

    // システムが使用中のリソースサイズを表示
    auto usage = nn::os::GetVirtualAddressMemoryResourceUsage();
    NN_LOG("Assigned size: 0x%zx\n", usage.assignedSize);
    NN_LOG("Used size: 0x%zx\n", usage.usedSize);

    // 物理メモリ・アドレス空間ともに解放
    result = nn::os::FreeAddressRegion(region);
    if (result.IsFailure())
    {
        // 解放に失敗
        return;
    }

    NN_LOG("Free address region:   start=0x%zx\n", region);
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    VirtualAddressMemorySample();
}
