﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsMessageQueue.cpp,PageSampleOsMessageQueue}
 *
 * @brief
 *  メッセージキュー機能のサンプルプログラム
 */

/**
 * @page PageSampleOsMessageQueue MessageQueue
 * @tableofcontents
 *
 * @brief
 *  メッセージキュー機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsMessageQueue_SectionBrief 概要
 *  ここでは、メッセージキュー機能を使ったサンプルプログラムの説明をします。
 *
 *  メッセージキュー機能の使い方については、メッセージキュー機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsMessageQueue_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsMessageQueue Samples/Sources/Applications/OsMessageQueue @endlink 以下にあります。
 *
 * @section PageSampleOsMessageQueue_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsMessageQueue_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsMessageQueue_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsMessageQueue_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsMessageQueue_SectionDetail 解説
 *
 * @subsection PageSampleOsMessageQueue_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsMessageQueue.cpp
 *  @includelineno OsMessageQueue.cpp
 *
 * @subsection PageSampleOsMessageQueue_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのスレッドを生成、1 つのメッセージキューを初期化
 *  - ThreadFunction1() は文字列のポインタをメッセージキューに送信
 *  - ThreadFunction2() はメッセージキューから受信した文字列をログに出力
 *  - 上記の処理を数行分繰り返す
 *  - 2 つのスレッドが終了するまで nnMain は待機
 *  - nnMain() にて 2 つのスレッドを破棄
 *
 *  2 つのスレッドは、 ThreadFunction1() から ThreadFunction2() へ
 *  メッセージキューを介して文字列のポインタの送信を行ないます。@n
 *  一度に送信できるデータは 3 つに設定しており、バッファが満杯になれば
 *  ThreadFunction1() はブロッキングします。@n
 *  また、バッファに空きが出来ると待ちが解除されて次の送信を行ないます。
 *
 *  ThreadFunction2() はバッファが空っぽの間はブロッキングされており、
 *  1 つでもバッファにデータが送信されると、それを受信してログへ出力します。@n
 *  最後にエンドマーク（0x00）が送信されてくれば、スレッドを終了します。
 *
 *  このサンプルプログラムの実行結果を以下に示します。@n
 *  ThreadFunction1() で送信した文字列が ThreadFunction2() から出力されています。
 *
 *  @verbinclude  OsMessageQueue_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // メッセージキュー操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

const size_t    BufferSize = 3;                 // メッセージキューサイズ
uintptr_t       g_messageBuffer[ BufferSize ];  // メッセージキューバッファ

nn::os::MessageQueueType g_MessageQueue;            // メッセージキュー

}   // namespace

//-----------------------------------------------------------------------------

//
//  ウェイト＋文字列送信
//
inline void SendString(const char* message)
{
    nn::os::SendMessageQueue( &g_MessageQueue, reinterpret_cast<uintptr_t>( message ) );
}

//
//  1 つ目のスレッド（メッセージキュー送信する側）
//
void ThreadFunction1(void *arg)
{
    NN_UNUSED(arg);

    SendString("Hello!\n");
    SendString("This is sample program for MessageQueue OS-API.\n");
    SendString("The MessageQueue can send and receive a data of the type uintptr_t,\n");
    SendString("so it's suitable to send and receive an address value.\n");

    // エンドマーク
    nn::os::SendMessageQueue( &g_MessageQueue, 0x00 );
}


//
//  2 つ目のスレッド（メッセージキュー受信する側）
//
void ThreadFunction2(void *arg)
{
    NN_UNUSED(arg);

    uintptr_t   data;

    // エンドマークが来るまで文字出力
    for (;;)
    {
        nn::os::ReceiveMessageQueue( &data, &g_MessageQueue );
        if (data == 0x00)
        {
            return;
        }
        NN_LOG( reinterpret_cast<char*>( data ) );
    }
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::Result      result;

    // メッセージキューを初期化する
    nn::os::InitializeMessageQueue( &g_MessageQueue, g_messageBuffer, BufferSize );

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ThreadFunction1, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread1." );

    result = nn::os::CreateThread( &thread2, ThreadFunction2, NULL, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread2." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // メッセージキューをファイナライズする
    nn::os::FinalizeMessageQueue( &g_MessageQueue );
}

