﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/os/os_InterruptEvent.h>

#include "OsInterruptEvent-hw.tk2.h"

namespace nns { namespace os { namespace interruptEvent {

//-----------------------------------------------------------------------------
//  100msec 毎にタイマー割込みを発生させる
//-----------------------------------------------------------------------------

uintptr_t   TimerRegisterVirtualAddress;

#define NNS_REG_TMR1_PTV            (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x00))
#define NNS_REG_TMR1_PCR            (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x04))
#define NNS_REG_TIMERUS_USEC_CFG    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x14))

//-----------------------------------------------------------------------------

void ClearTimerInterruptStatus() NN_NOEXCEPT
{
    NNS_REG_TMR1_PCR |= 0x1 << 30;

    nn::dd::EnsureMemoryAccess();
}

//-----------------------------------------------------------------------------

void StartTimerInterrupt() NN_NOEXCEPT
{
    NN_LOG("Timer: padrs=0x%016llx\n", TimerRegisterPhysicalAddress);
    NN_LOG("Timer: size =0x%08zx\n",    TimerRegisterRegionSize);

    // タイマー I/O がマッピングされている仮想アドレスを取得
    uintptr_t vadrs = nn::dd::QueryIoMappingAddress(TimerRegisterPhysicalAddress, TimerRegisterRegionSize);
    if (!vadrs)
    {
        NN_ABORT("*** Timer Register is not mapped.");
    }
    TimerRegisterVirtualAddress = vadrs;

    NN_LOG("Timer: vadrs=0x%p\n",    TimerRegisterVirtualAddress);

    // clk_m 周波数を 12MHz に設定（p.55: Only 12 MHz is currently supported.)
    NNS_REG_TIMERUS_USEC_CFG = 0xb;

    ClearTimerInterruptStatus();

    // 100msec 周期に設定（p.256: If you program the value n, the counter trigger will actually be n+1.）
    NNS_REG_TMR1_PTV = 100 * 1000 - 1;

    // Periodic に設定
    NNS_REG_TMR1_PTV |= 0x1 << 30;

    // タイマーを開始
    NNS_REG_TMR1_PTV |= 0x1 << 31;
}

//-----------------------------------------------------------------------------

void StopTimerInterrupt() NN_NOEXCEPT
{
    // タイマーを停止
    NNS_REG_TMR1_PTV &= ~(0x1 << 31);

    ClearTimerInterruptStatus();
}

//-----------------------------------------------------------------------------

}}} // namespace nns::os::interruptEvent

