﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsConditionVariable.cpp,PageSampleOsConditionVariable}
 *
 * @brief
 *  条件変数機能のサンプルプログラム
 */

/**
 * @page PageSampleOsConditionVariable ConditionVariable
 * @tableofcontents
 *
 * @brief
 *  条件変数機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsConditionVariable_SectionBrief 概要
 *  ここでは、条件変数機能を使ったサンプルプログラムの説明をします。
 *
 *  条件変数機能の使い方については、条件変数機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsConditionVariable_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsConditionVariable Samples/Sources/Applications/OsConditionVariable @endlink 以下にあります。
 *
 * @section PageSampleOsConditionVariable_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsConditionVariable_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsConditionVariable_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsConditionVariable_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsConditionVariable_SectionDetail 解説
 *
 * @subsection PageSampleOsConditionVariable_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsConditionVariable.cpp
 *  @includelineno OsConditionVariable.cpp
 *
 * @subsection PageSampleOsConditionVariable_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのスレッドを生成
 *  - 2 つのスレッドは、 ThreadFunction1() と ThreadFunction2()
 *  - それぞれのスレッドでは、互いに異なる条件変数通知と条件変数待機を行なう
 *  - 2 つのスレッドが終了するまで nnMain() は待機
 *  - nnMain() にて 2 つのスレッドを破棄
 *
 *  2 つのスレッドは、char 型の g_CharCode 変数を共有リソースとして、
 *  ThreadFunction1() から ThreadFunction2() へ文字コードの送信を行ないます。
 *  一度に送信できる文字コードは char 型１文字のみです。
 *
 *  ThreadFunction1() は、 SendCharCode() で１文字送信を行ないます。
 *  ThreadFunction2() は、 ReceiveCharCode() で１文字受信を行ない、
 *  受信したものをログへ出力します。
 *
 *  SendCharCode() は、g_CharCode == 0 になるまで待機します。
 *  この時、g_ConditionEmpty 条件変数を WaitConditionVariable() で待機します。
 *
 *  一方、 ReceiveCharCode() は、g_CharCode != 0 になるまで待機します。
 *  この時、g_ConditionNotEmpty 条件変数を WaitConditionVariable() で待機します。
 *
 *  それぞれの条件変数に SignalConditionVariable() を発行するのは、
 *  g_CharCode を然るべき状態に変更したもう片方のスレッドです。@n
 *  つまり、受信待ちに NotEmpty 通知するのは送信側のスレッドであり、
 *  送信待ちに Empty 通知するのは受信側のスレッドということになります。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *  ThreadFunction1() で送信した文字列が ThreadFunction2() から出力されています。
 *
 *  @verbinclude  OsConditionVariable_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // 条件変数操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

//-----

char                      g_CharCode = 0;       // 共有リソース（文字コード）

nn::os::MutexType             g_Mutex;              // 操作対象のミューテックス
nn::os::ConditionVariableType g_ConditionEmpty;     // Empty になるのを待機する条件変数
nn::os::ConditionVariableType g_ConditionNotEmpty;  // Not Empty になるのを待機する条件変数

}   // namespace

//-----------------------------------------------------------------------------

//
//  1 文字送信
//
void SendCharCode(const char code)
{
    if (code == 0)
    {
        return;
    }

    nn::os::LockMutex( &g_Mutex );

    // 送信バッファが空（Empty）になるまで送信待ち
    while ( g_CharCode != 0 )
    {
        nn::os::WaitConditionVariable( &g_ConditionEmpty, &g_Mutex );
    }

    // 送信文字コードをバッファに積んで NotEmpty の条件変数を通知
    g_CharCode = code;
    nn::os::SignalConditionVariable( &g_ConditionNotEmpty );

    nn::os::UnlockMutex( &g_Mutex );
}

//
//  1 文字受信
//
char ReceiveCharCode()
{
    nn::os::LockMutex( &g_Mutex );

    // 送信バッファが空（Empty）ならば NotEmpty になるまで受信待ち
    while ( g_CharCode == 0 )
    {
        nn::os::WaitConditionVariable( &g_ConditionNotEmpty, &g_Mutex );
    }

    // バッファから文字コードを取得して Empty の条件変数を通知
    char  code = g_CharCode;
    g_CharCode = 0;
    nn::os::SignalConditionVariable( &g_ConditionEmpty );

    nn::os::UnlockMutex( &g_Mutex );

    return code;
}

//-----------------------------------------------------------------------------

//
//  文字列を１文字ずつ送信する
//
void SendString(const char* message)
{
    char* ptr = const_cast<char*>( message );
    char  code;

    while ( (code = *ptr) != '\0' )
    {
        SendCharCode( code );
        ++ptr;
    }
}

//
//  1 つ目のスレッド
//
void ThreadFunction1(void *arg)
{
    NN_UNUSED(arg);

    SendString("Hello!\n");
    SendString("This is a ConditionVariable sample code.\n");

    // エンドマーク
    SendString("\xff");
}


//
//  2 つ目のスレッド
//
void ThreadFunction2(void *arg)
{
    NN_UNUSED(arg);

    char buf[2] = { '\0', '\0' };
    char code;

    // エンドマークが来るまで文字出力
    while ( (code = ReceiveCharCode()) != '\xff' )
    {
        buf[0] = code;
        NN_LOG( buf );
    }
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::Result      result;

    // 条件変数とミューテックスを初期化する
    nn::os::InitializeMutex( &g_Mutex, false, 0 );
    nn::os::InitializeConditionVariable( &g_ConditionEmpty );
    nn::os::InitializeConditionVariable( &g_ConditionNotEmpty );

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ThreadFunction1, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread1." );

    result = nn::os::CreateThread( &thread2, ThreadFunction2, NULL, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread2." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // 条件変数とミューテックスをファイナライズする
    nn::os::FinalizeMutex( &g_Mutex );
    nn::os::FinalizeConditionVariable( &g_ConditionEmpty );
    nn::os::FinalizeConditionVariable( &g_ConditionNotEmpty );
}

