﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OeLaunchParameter.cpp,PageSampleOeLaunchParameter}
 *
 * @brief
 *  起動パラメータの取得サンプルプログラム
 */

/**
 * @page PageSampleOeLaunchParameter 起動パラメータ取得サンプル
 * @tableofcontents
 *
 * @brief
 *  アプリケーション起動パラメータの取得方法の解説です。
 *
 * @section PageSampleOeLaunchParameter_SectionBrief 概要
 *  ここでは、アプリケーション動作環境操作ライブラリを用いて、
 *  起動パラメータの取得方法について解説します。
 *
 *  アプリケーション動作環境操作ライブラリの使い方については、
 *  @ref nn::oe "アプリケーション動作環境操作ライブラリの関数リファレンス"
 *  も併せて参照して下さい。
 *
 * @section PageSampleOeLaunchParameter_SectionFileStructure ファイル構成
 *  本サンプルプログラムは、
 *  @link ../../../Samples/Sources/Applications/OeLaunchParameter
 *  Samples/Sources/Applications/OeLaunchParameter @endlink 以下にあります。
 *
 * @section PageSampleOeLaunchParameter_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleOeLaunchParameter_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOeLaunchParameter_SectionPrecaution 注意事項
 *  このプログラムは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOeLaunchParameter_SectionHowToExecute 実行手順
 *  - 予め SD カードを開発機に挿入しておきます。
 *  - 開発機のメニュー画面を DevMenu にしておきます（HOME メニューでは下記の機能を利用できません）。
 *  - 以下のように OeLaunchParameter_Parameter.nxargs というバイナリファイルを起動パラメータとして登録します。以下を実行すると、OeLaunchParameter_Parameter.nxargs ファイルが SD カードにインストールされます。
 *
 *    @code
 *    $ RunOnTarget.exe DevMenuCommand.nsp -- application install-nxargs "D:\NintendoSDK\Samples\Sources\Applications\OeLaunchParameter\OeLaunchParameter_Parameter.nxargs" --application-id 0x0100005003486000
 *    @endcode
 *
 *  - サンプルプログラムをビルドし、実行して下さい。
 *
 * @section PageSampleOeLaunchParameter_SectionDetail 解説
 *
 * @subsection PageSampleOeLaunchParameter_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OeLaunchParameter.cpp
 *  @includelineno OeLaunchParameter.cpp
 *
 * @subsection PageSampleOeLaunchParameter_SectionSampleDetail サンプルプログラムの解説
 *
 *  このサンプルプログラムでは以下のような流れになっています。
 *
 *  - nninitStartup() で new/delete 用のメモリプールを設定
 *  - nnMain() 関数にて以下の処理を実行
 *      - LaunchParameter クラスのインスタンスを配置
 *      - LaunchParameter::InitializeAndTryPop() で起動パラメータを取得
 *      - LaunchParameter::GetNewsParameterBufferPointer() で News から渡された起動パラメータを参照
 *
 *  実際に起動パラメータを取得しているのでは InitializeAndTryPop() 関数です。
 *  この関数では nn::oe::TryPopLaunchParameter() が false を返すまで繰り返し
 *  起動パラメータの取得を試みます。起動パラメータは将来的に、
 *  News だけでなく複数のパラメータを受け取るようになる可能性があるため、
 *  このように実装しておく必要があります。
 *
 *  また、プログラムにとって未知のフォーマットの起動パラメータを受け取った
 *  場合には、アボート等にはせずにその起動パラメータをスキップ（無視して破棄）
 *  するようにし、将来的にも下位互換が維持されるようにしておきます。
 *
 *  本サンプルプログラムでは、全ての起動パラメータの先頭 16byte に、
 *  LaunchParameterHeader 構造体で示されるヘッダが配置される想定です。
 *  このようなヘッダの有無や、詳細なフォーマットはアプリケーション開発者が
 *  独自に定義すべきものですが、将来の拡張を見越してこういった付加情報を
 *  配置しておくことを推奨します。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OeLaunchParameter_OutputExample.txt
 *
 * @subsection PageSampleOeLaunchParameter_Link 関連ドキュメント
 *
 *  起動パラメータについては以下のドキュメントも参考にして下さい。
 *
 *  - 「ゲームニュース配信ガイド」の「特定のアプリケーションを起動する」の章
 *
 *
 */

//-----------------------------------------------------------------------------

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/oe.h>

//-----------------------------------------------------------------------------
//
//  本来はアプリケーション独自のヘッダファイルなどで定義しておく
//

class LaunchParameter
{
private:
    // 起動パラメータ種別
    enum LaunchParameterKind
    {
        LaunchParameterKind_News                = 1,
        LaunchParameterKind_ApplicationSpecific = 2,
    };

    // 起動パラメータ先頭に配置するヘッダ（16byte 固定）
    struct LaunchParameterHeader
    {
        LaunchParameterKind parameterKind;  // 起動パラメータ種別
        uint32_t _reserved0;                // 予約（将来の拡張用）
        uint32_t _reserved1;                // 予約（将来の拡張用）
        uint32_t _reserved2;                // 予約（将来の拡張用）
    };

    const size_t NewsParameterBufferSize = 0x1000;
    char* m_NewsParameterBuffer = nullptr;

public:
    ~LaunchParameter()
    {
        delete[] m_NewsParameterBuffer;
    }

    // 初期化と起動パラメータ取得
    void InitializeAndTryPop();

    // ゲームニュース関連
    bool IsNewsParameterAvailable() const
    {
        return m_NewsParameterBuffer != nullptr;
    }

    char* GetNewsParameterBufferPointer() const
    {
        return m_NewsParameterBuffer;
    }
};

void LaunchParameter::InitializeAndTryPop()
{
    const size_t bufferSize = 0x1000;
    uint8_t* pBuffer = new uint8_t[bufferSize];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pBuffer;
    };

    for (;;)
    {
        // 起動パラメータをひとつ取得
        size_t paramSize;
        if ( !nn::oe::TryPopLaunchParameter(&paramSize, pBuffer, bufferSize) )
        {
            break;
        }

        // ヘッダサイズ未満のパラメータは未知のフォーマットであるためスキップ
        if (paramSize < sizeof(LaunchParameterHeader))
        {
            continue;
        }

        // ヘッダ部の解析
        auto   pHeader  = reinterpret_cast<LaunchParameterHeader*>(pBuffer);
        auto   pData    = pBuffer   + sizeof(LaunchParameterHeader);
        size_t dataSize = paramSize - sizeof(LaunchParameterHeader);

        switch (pHeader->parameterKind)
        {
            case LaunchParameterKind_News:
            {
                // ゲームニュースの起動パラメータが想定より大きければスキップ
                if (paramSize > NewsParameterBufferSize)
                {
                    continue;
                }
                this->m_NewsParameterBuffer = new char[dataSize];
                std::memcpy(this->m_NewsParameterBuffer, pData, dataSize);
                break;
            }
            case LaunchParameterKind_ApplicationSpecific:
            {
                // アプリケーション自身が設定した起動パラメータ
                // ただし、NX Addon 4.2 ではこのような起動パラメータを
                // 設定する機能はないためスキップ。
                break;
            }
            default:
            {
                // 未知のパラメータのためスキップする
                continue;
            }
        }
    }
}


//-----------------------------------------------------------------------------
//
//  メモリの初期化
//
extern "C" void nninitStartup()
{
    // new/delete 用のメモリをヒープから確保
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize(MemoryHeapSize) );

    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock(&address, MemoryHeapSize) );

    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}


//-----------------------------------------------------------------------------
//
//  メイン関数
//

extern "C" void nnMain()
{
    LaunchParameter param;

    param.InitializeAndTryPop();

    if (param.IsNewsParameterAvailable())
    {
        NN_LOG("----------------------------------------------------\n");
        NN_LOG("News Parameter = '%s'\n", param.GetNewsParameterBufferPointer());
        NN_LOG("----------------------------------------------------\n");
    }
    else
    {
        NN_LOG("News Parameter is not found.\n");
    }
}
