﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OeHdcpAuthenticationState.cpp,PageSampleOeHdcpAuthenticationState}
 *
 * @brief
 *  HDCP の認証状態取得のサンプルプログラム
 */

/**
 * @page PageSampleOeHdcpAuthenticationState HDCP の認証状態取得
 * @tableofcontents
 *
 * @brief
 *  アプリケーション動作環境操作ライブラリを用いた HDCP の認証状態取得の解説です。
 *
 * @section PageSampleOeHdcpAuthenticationState_SectionBrief 概要
 *  ここでは、アプリケーション動作環境操作ライブラリを用いて HDCP の認証状態を取得する方法を解説します。
 *
 *  アプリケーション動作環境操作ライブラリの使い方については、
 *  @ref nn::oe "アプリケーション動作環境操作ライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOeHdcpAuthenticationState_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OeHdcpAuthenticationState
 *  Samples/Sources/Applications/OeHdcpAuthenticationState @endlink 以下にあります。
 *
 * @section PageSampleOeHdcpAuthenticationState_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleOeHdcpAuthenticationState_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOeHdcpAuthenticationState_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOeHdcpAuthenticationState_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOeHdcpAuthenticationState_SectionDetail 解説
 *
 * @subsection PageSampleOeHdcpAuthenticationState_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OeHdcpAuthenticationState.cpp
 *  @includelineno OeHdcpAuthenticationState.cpp
 *
 * @subsection PageSampleOeHdcpAuthenticationState_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムは HDCP の認証状態を取得し、それをログに出力します。
 *  HDCP の認証状態の遷移については nn::oe::GetHdcpAuthenticationStateChangeEvent() の詳細を参照して下さい。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - アプリケーション動作環境操作ライブラリを初期化。
 *  - 現在の HDCP 認証状態を取得。
 *  - HDCP の認証状態が変更されたときのシステムイベントを取得。
 *  - HDCP の認証状態が変更されるのを待機。
 *  - 現在の HDCP 認証状態を取得。
 *
 *  このサンプルプログラムの実行結果例を以下に示します。
 *
 *  @verbinclude  OeHdcpAuthenticationState_OutputExample.txt
 *
 */

//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/oe.h>
#include <nn/oe/oe_HdcpApis.private.h>

//-----------------------------------------------------------------------------

//
//  HdcpAuthenticationState から文字列を取得する関数
//
const char* getHdcpAuthenticationStateString(nn::oe::HdcpAuthenticationState state)
{
    static const char* hdcpAuthenticationStateString[] =
    {
        "Unauthenticated",
        "Processing",
        "Authenticated"
    };
    return hdcpAuthenticationStateString[state];
}

//
//  メイン関数
//
extern "C" void nnMain()
{
    nn::oe::HdcpAuthenticationState state;

    // アプリケーション動作環境操作ライブラリを初期化。
    nn::oe::Initialize();

    // 現在の HDCP 認証状態を取得。
    state = nn::oe::GetHdcpAuthenticationState();
    NN_LOG("Hdcp Authentication State : %s\n", getHdcpAuthenticationStateString(state));

    // HDCP の認証状態が変更されたときのシステムイベントを取得。
    nn::os::SystemEvent hdcpAuthenticationStateChangeEvent;
    nn::oe::GetHdcpAuthenticationStateChangeEvent( &hdcpAuthenticationStateChangeEvent );

    for (;;)
    {
        // HDCP の認証状態が変更されるのを待機。
        hdcpAuthenticationStateChangeEvent.Wait();

        // 現在の HDCP 認証状態を取得。
        state = nn::oe::GetHdcpAuthenticationState();
        NN_LOG("Hdcp Authentication State : %s\n", getHdcpAuthenticationStateString(state));
    }
}
