﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OeDisableAutoSleep.cpp,PageSampleOeDisableAutoSleep}
 *
 * @brief
 *  アプリケーション中で一時的に自動スリープを無効化する機能のサンプルプログラム
 */

/**
 * @page PageSampleOeDisableAutoSleep 一時的な自動スリープの無効化
 * @tableofcontents
 *
 * @brief
 *  アプリケーション中で一時的に自動スリープを無効化する機能の解説です。
 *
 * @section PageSampleOeDisableAutoSleep_SectionBrief 概要
 *  アプリケーション中で一時的に自動スリープを無効化する方法を解説します。
 *
 * @section PageSampleOeDisableAutoSleep_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OeDisableAutoSleep
 *  Samples/Sources/Applications/OeDisableAutoSleep @endlink 以下にあります。
 *
 * @section PageSampleOeDisableAutoSleep_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleOeDisableAutoSleep_SectionHowToOperate 操作方法
 *  メインループ中は、無線接続したジョイコンまたはデバッグパッドにより以下の操作が可能です。
 *
 *  | 操作                 | 機能 |
 *  | -------------------- | ---- |
 *  | A                    | 90 秒間だけ自動スリープを無効化する |
 *  | X                    | BeginAutoSleepDisabledSection を呼び出す |
 *  | Y                    | EndAutoSleepDisabledSection を呼び出す |
 *  | マイナス (-)         | サンプルプログラムを終了する |
 *
 * @section PageSampleOeDisableAutoSleep_SectionPrecaution 注意事項
 *  特になし
 *
 * @section PageSampleOeDisableAutoSleep_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *  自動スリープ無効化の効果を実際に確認するには、
 *  あらかじめ本体設定で携帯モード中の自動スリープまでの時間を最短 (1 分) にしておき、
 *  本サンプルを携帯モードで起動する必要があります。
 *
 * @section PageSampleOeDisableAutoSleep_SectionDetail 解説
 *
 * @subsection PageSampleOeDisableAutoSleep_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OeDisableAutoSleep.cpp
 *  @includelineno OeDisableAutoSleep.cpp
 *
 * @subsection PageSampleOeDisableAutoSleep_SectionSampleDetail サンプルプログラムの解説
 *
 *  A ボタンを押すと、その瞬間から 90 秒間にわたり自動スリープを無効化します。
 *
 *    * 90 秒後に無効化を解除した時点ですでに無操作状態の期間が本来の自動スリープ開始時間に達していた場合、その場で自動スリープが発動します。
 *        * 本体を携帯モードにし、自動スリープまでの時間を最短 (1 分) にしておくとこの挙動を確認できます。
 *        * 本機能では画面焼け軽減は無効化されないことも確認できます。
 *    * 90 秒経過する前に A ボタンを押すと、その場で自動スリープの無効化を終了します。
 *
 *  X/Y ボタンで BeginAutoSleepDisabledSection / EndAutoSleepDisabledSection を重ねて呼び出します。
 *
 *    * Begin の呼び出しがネストされた場合、Begin を呼び出した総数と同じ回数 End を呼び出したときに初めて解除される挙動を確認できます。
 *
 */

//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nns/gfxLog.h>
#include <nns/nns_Log.h>
#include <nns/hid.h>

#include <nn/oe/oe_DisableAutoSleepApi.h>

//-----------------------------------------------------------------------------

namespace {

    nn::os::TimerEvent g_OneSecondTimer(nn::os::EventClearMode_AutoClear);
    int g_ElapsedSeconds = 0;

    void RunTimeCountingThread() NN_NOEXCEPT
    {
        static nn::os::ThreadType s_Thread;
        static NN_ALIGNAS(4096) char s_ThreadStack[4 * nn::os::MemoryPageSize];
        nn::os::CreateThread(
            &s_Thread,
            [](void* arg) {
                NN_UNUSED(arg);
                while ( NN_STATIC_CONDITION(true) )
                {
                    g_OneSecondTimer.Wait();
                    ++g_ElapsedSeconds;
                    NNS_LOG("%d sec\n", g_ElapsedSeconds);
                }
            },
            nullptr,
            s_ThreadStack,
            sizeof(s_ThreadStack),
            nn::os::DefaultThreadPriority
        );
        nn::os::SetThreadNamePointer(&s_Thread, "1sTimerThread");
        nn::os::StartThread(&s_Thread);
    }

    void PrintIsAutoSleepDisabled() NN_NOEXCEPT
    {
        NNS_LOG("Auto sleep is %s now.\n", nn::oe::IsAutoSleepDisabled() ? "disabled" : "not disabled");
    }
    void BeginAutoSleepDisabled() NN_NOEXCEPT
    {
        nn::oe::BeginAutoSleepDisabledSection();
        NNS_LOG("Called BeginAutoSleepDisabledSection().\n");
        PrintIsAutoSleepDisabled();
    }
    void EndAutoSleepDisabled() NN_NOEXCEPT
    {
        nn::oe::EndAutoSleepDisabledSection();
        NNS_LOG("Called EndAutoSleepDisabledSection().\n");
        PrintIsAutoSleepDisabled();
    }

    class TimedDisableMode
    {
    public:
        TimedDisableMode() NN_NOEXCEPT :
            m_TimedDisableModeStartTick(nn::os::Tick()),
            m_IsInTimedDisableMode(false)
        {
        }
        void Begin() NN_NOEXCEPT
        {
            NNS_LOG("Disabling auto sleep for 90 seconds from now.\n");
            m_IsInTimedDisableMode = true;
            m_TimedDisableModeStartTick = nn::os::GetSystemTick();

            g_ElapsedSeconds = 0;
            g_OneSecondTimer.StartPeriodic(nn::TimeSpan::FromSeconds(1), nn::TimeSpan::FromSeconds(1));

            BeginAutoSleepDisabled();
        }
        void End() NN_NOEXCEPT
        {
            m_IsInTimedDisableMode = false;
            m_TimedDisableModeStartTick = nn::os::Tick();

            g_OneSecondTimer.Stop();
            g_ElapsedSeconds = 0;

            EndAutoSleepDisabled();
        }
        void Update() NN_NOEXCEPT
        {
            if ( m_IsInTimedDisableMode )
            {
                if ( nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - m_TimedDisableModeStartTick) >= Duration )
                {
                    NNS_LOG("90 seconds passed.\n");
                    End();
                    NNS_LOG("Auto sleep may happen immediately now.\n");
                }
            }
        }
        bool IsInTimedDisableMode() const NN_NOEXCEPT
        {
            return m_IsInTimedDisableMode;
        }

    private:
        const nn::TimeSpan Duration = nn::TimeSpan::FromSeconds(90);

    private:
        nn::os::Tick m_TimedDisableModeStartTick;
        bool m_IsInTimedDisableMode;
    };

}


//
//  メイン関数
//
extern "C" void nnMain()
{
    // ログ描画用スレッドのコアを指定します。
    nns::gfxLog::SetThreadCoreNumber(0);

    NNS_LOG("=============================================\n");
    NNS_LOG("A                  : Disable auto sleep for 90 sec\n");
    NNS_LOG("X                  : Call BeginAutoSleepDisabled\n");
    NNS_LOG("Y                  : Call EndAutoSleepDisabledSection\n");
    NNS_LOG("Minus              : Quit program\n");
    NNS_LOG("=============================================\n");

    nns::hid::ControllerManager controllerManager;
    nns::hid::util::SetControllerManagerWithDefault(&controllerManager);
    auto& gamePad = *reinterpret_cast<nns::hid::GamePad*>(controllerManager.GetController(nns::hid::ControllerId_GamePad, 0));
    auto& debugPad = *reinterpret_cast<nns::hid::DebugPad*>(controllerManager.GetController(nns::hid::ControllerId_DebugPad, 0));

    RunTimeCountingThread();

    auto timedDisableMode = TimedDisableMode();

    PrintIsAutoSleepDisabled();

    while ( NN_STATIC_CONDITION(true) )
    {
        controllerManager.Update();
        timedDisableMode.Update();

        const auto SwitchTimedDisableModeInputMask = nns::hid::Button::A::Mask;
        const auto BeginAutoSleepDisabledSectionInputMask = nns::hid::Button::X::Mask;
        const auto EndAutoSleepDisabledSectionInputMask = nns::hid::Button::Y::Mask;
        const auto EndDemoInputMask = nns::hid::Button::Minus::Mask;

        if ( (gamePad.IsConnected() && gamePad.HasAnyButtonsDown(BeginAutoSleepDisabledSectionInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtonsDown(BeginAutoSleepDisabledSectionInputMask)) )
        {
            BeginAutoSleepDisabled();
        }
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtonsDown(EndAutoSleepDisabledSectionInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtonsDown(EndAutoSleepDisabledSectionInputMask)) )
        {
            EndAutoSleepDisabled();
        }
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtonsDown(SwitchTimedDisableModeInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtonsDown(SwitchTimedDisableModeInputMask)) )
        {
            if ( timedDisableMode.IsInTimedDisableMode() )
            {
                timedDisableMode.End();
            }
            else
            {
                timedDisableMode.Begin();
            }
        }
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtonsDown(EndDemoInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtonsDown(EndDemoInputMask)) )
        {
            // サンプルプログラムを終了する
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // 自動スリープ無効化 API の影響範囲はアプリケーション内に閉じているため、
    // 仮に BeginAutoSleepDisabledSection() を呼び出したまま終了しても問題ありません。

    // プログラムの終了
    NNS_LOG("\nEnd of Demo.\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
}
