﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NifmRequest.cpp,PageSampleNifmRequest}
 *
 * @brief
 *  ネットワーク接続の利用要求の直接提出のサンプルプログラム
 */

/**
 * @page PageSampleNifmRequest ネットワーク接続の利用要求の直接提出
 * @tableofcontents
 *
 * @brief
 *  ネットワーク接続の利用要求の直接提出のサンプルプログラムの解説です。
 *
 * @section PageSampleNifmRequest_SectionBrief 概要
 *  先ず NIFM の API を利用してネットワーク接続の利用要求を構築し、システムに提出します。
 *  その結果、ネットワーク接続の利用が可能であれば、10秒間状態を監視したあと要求を取り下げます。
 *
 * @section PageSampleNifmRequest_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/NifmRequest
 *  Samples/Sources/Applications/NifmRequest @endlink 以下にあります。
 *
 * @section PageSampleNifmRequest_SectionNecessaryEnvironment 必要な環境
 *  特にありません。
 *
 * @section PageSampleNifmRequest_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleNifmRequest_SectionPrecaution 注意事項
 *  ネットワーク接続や通信処理に失敗した場合におこなうべき処理の詳細は未定です。
 *  要求の状態変化のシグナル待ちや剥奪予告には未対応です。
 *
 * @section NifmRequest_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleNifmRequest_SectionDetail 解説
 *
 * @subsection PageSampleNifmRequest_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  NifmRequest.cpp
 *  @includelineno NifmRequest.cpp
 *
 * @subsection PageSampleNifmRequest_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - NIFM の初期化を行います。
 *  - 利用要求の構築をおこないます。
 *  - 構築した利用要求をシステムに提出し、ネットワーク接続の利用可否を確認します。
 *  - 10秒間状態を監視し、何も起きないか、途中で許可の剥奪が起きれば要求を取り下げて終了します。
 *
 * このサンプルプログラムの実行結果を以下に示します。
 *
 * @verbinclude NifmRequest_Output.txt
 */

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_ApiRequest.h>

#include <nn/os.h>


extern "C" void nnMain()
{
    // nifm の API は、利用前にライブラリの初期化が必要です
    nn::nifm::Initialize();

    {
        // ローカル通信用に通信デバイスの占有を要求する例
        nn::nifm::RequestParameters requestParameters = { nn::nifm::RequirementPreset_LocalGeneric };

        nn::nifm::Request request(requestParameters);

#if 0
        // 要件をカスタマイズする場合

        // 要件をカスタマイズするにはまず要求のハンドルを取得する
        nn::nifm::RequestHandle handle = request.GetHandle();

        // 優先度を変更する例
        nn::nifm::SetRequestPriority(handle, 32);
#endif

        // 通信処理の前に、ネットワーク接続の利用要求をシステムに提出し、
        // ネットワーク接続の利用可否を確認します
#if 1
        // ブロッキング API を利用する場合
        request.SubmitAndWait();
#else
        // ノンブロッキング API を利用し、ポーリングで完了を確認する場合
        request.Submit();
        while( request.GetRequestState() == nn::nifm::RequestState_OnHold )
        {
            NN_LOG("Request is on hold...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
#endif

        // 要求が受理されれば、通信処理をおこないます
        if( request.GetRequestState() == nn::nifm::RequestState_Accepted )
        {
            NN_LOG("Request is accepted.\n");

            for( int i = 0; i < 10; ++i )
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

                // 要求が外部要因で取り下げられていないか毎処理フレーム確認します
                // 要求の状態変化のシグナル待ち、剥奪予告には未対応です
                if( request.GetRequestState() != nn::nifm::RequestState_Accepted )
                {
                    NN_LOG("Request has been rejected by external factors.\n");
                    break;
                }
            }
        }
        else
        {
            // 将来的には、失敗した場合のエラーハンドリングを行う必要があります
            // エラーハンドリングの方法については、現在検討中です
            NN_LOG("Request is rejected.\n");
        }

        // ネットワーク接続が不要になったら要求を取り下げます
        // 予期しない理由でネットワーク接続が失われていた場合、
        // あらためて明示的に要求を取り下げる必要はありませんが、
        // 重ねて取り下げをおこなっても問題はありません
        request.Cancel();
        NN_ASSERT(request.GetRequestState() == nn::nifm::RequestState_Free);

        // また、明示的にキャンセル操作をおこなわなくても、
        // Request オブジェクトが破棄される際にキャンセルがおこなわれます
    }

    NN_LOG("Request is canceled.\n");
}
