﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NifmNetworkConnectionSimple.cpp,PageSampleNifmNetworkConnectionSimple}
 *
 * @brief
 *  ネットワーク接続の利用要求の提出およびソケット通信処理のサンプルプログラムのメイン処理
 */

/**
 * @page PageSampleNifmNetworkConnectionSimple ネットワーク接続の利用要求の提出およびソケット通信
 * @tableofcontents
 *
 * @brief
 *  ネットワーク接続の利用要求の提出およびソケット通信処理のサンプルプログラムの解説です。
 *
 * @section PageSampleNifmNetworkConnectionSimple_SectionBrief 概要
 *  先ず NIFM の API を利用してネットワーク接続の利用要求をシステムに提出します。
 *  その結果、ネットワーク接続の利用が可能であればソケット通信処理を行います。
 *
 * @section PageSampleNifmNetworkConnectionSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/NifmNetworkConnectionSimple
 *  Samples/Sources/Applications/NifmNetworkConnectionSimple @endlink 以下にあります。
 *
 * @section PageSampleNifmNetworkConnectionSimple_SectionNecessaryEnvironment 必要な環境
 *  ネットワーク接続が可能な環境が必要です。@n
 *  また、事前に設定マネージャを利用して、適切なネットワーク接続設定をインポートしておく必要があります。
 *  ネットワーク接続設定ファイルのサンプルは、 @link ../../../Samples/Sources/Applications/NifmNetworkConnectionSimple
 *  Samples/Sources/Applications/NifmNetworkConnectionSimple @endlink 以下に networkSettings_*.txt という名前で置かれています。
 *  詳細は @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} をご覧ください。
 *
 * @section PageSampleNifmNetworkConnectionSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleNifmNetworkConnectionSimple_SectionPrecaution 注意事項
 *  ネットワーク接続や通信処理に失敗した場合におこなうべき処理の詳細は未定です
 *
 * @section NifmNetworkConnectionSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleNifmNetworkConnectionSimple_SectionDetail 解説
 *
 * @subsection PageSampleNifmNetworkConnectionSimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードの一部を引用します。
 *
 *  NifmNetworkConnectionSimple.cpp
 *  @includelineno NifmNetworkConnectionSimple.cpp
 *
 * @subsection PageSampleNifmNetworkConnectionSimple_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - NIFM の初期化を行います。
 *  - 通信処理の前に、ネットワーク接続の利用要求をシステムに提出し、ネットワーク接続の利用可否を確認します。
 *  - example.com と HTTP 通信をおこない、取得した内容と結果をログに出力します。
 *
 * このサンプルプログラムの実行結果を以下に示します。ただし、Release 版のサンプルでは Curl の出力（受信内容）は表示されません。@n
 * また、実機の出力にも Curl の出力（受信内容）は含まれません。
 *
 * @verbinclude NifmNetworkConnectionSimple_Output.txt
 */

#include "NifmNetworkConnectionSimple.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/nifm.h>

#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <curl/curl.h>

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
}

// ソケット通信処理のサンプル
// example.com と HTTP 通信をおこない、取得した内容と結果をログに出力します
// この通信に成功した場合は true を、失敗した場合は false を返します
// libcurl の API 詳細については libcurl のマニュアルおよびサンプルを参照してください
bool Program::ExecuteTcpIpCommunication() NN_NOEXCEPT
{
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    nn::socket::Initialize(g_SocketConfigWithMemory);

    curl_global_init(CURL_GLOBAL_DEFAULT);

    CURL* curl = curl_easy_init();
    if (curl == nullptr)
    {
        nn::socket::Finalize();

        Printf("curl_easy_init failed.\n");
        return false;
    }

    curl_easy_setopt(curl, CURLOPT_URL, "http://example.com");

    // Debug/Develop 版では、サーバーからの受信内容が表示されます
    CURLcode curlCode = curl_easy_perform(curl);
    Printf("curl_easy_perform returned %d.\n\n", curlCode);

    char *pUrl;
    curl_easy_getinfo(curl, CURLINFO_EFFECTIVE_URL, &pUrl);
    Printf("url: %s\n", pUrl);

    long responseCode = -1;
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &responseCode);
    Printf("response code: %d\n", static_cast<int>(responseCode));

    curl_easy_cleanup(curl);

    curl_global_cleanup();

    nn::socket::Finalize();

    return curlCode == CURLE_OK;
}

// アプリケーションのメイン
void Program::Execute() NN_NOEXCEPT
{
    // nifm の API は、利用前にライブラリの初期化が必要です
    nn::nifm::Initialize();

    while (NN_STATIC_CONDITION(1))
    {
        // 通信処理の前に、ネットワーク接続の利用要求をシステムに提出し、
        // ネットワーク接続の利用可否を確認します
#if 1
    // ブロッキング API を利用する場合
        nn::nifm::SubmitNetworkRequestAndWait();
#else
    // ノンブロッキング API を利用し、ポーリングで完了を確認する場合
        nn::nifm::SubmitNetworkRequest();
        while (nn::nifm::IsNetworkRequestOnHold())
        {
            Printf("Network request is on hold...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
#endif

        // 利用要求提出の結果をハンドリングします
        nn::Result result = nn::nifm::HandleNetworkRequestResult();

        if (result.IsSuccess())
        {
            Printf("Network is available.\n");
            break;
        }
        else if (result <= nn::nifm::ResultErrorHandlingCompleted())
        {
            // 利用要求を再提出することで受理される可能性があります
            Printf("Network is not available. SubmitNetworkRequest again.\n");
            continue;
        }
        else
        {
            Printf("Network is not available.\n");
            break;
        }
    }

    // ネットワーク接続の利用が可能であれば、通信処理をおこないます
    if (nn::nifm::IsNetworkAvailable())
    {
        // 通信処理
        bool isSuccessful = ExecuteTcpIpCommunication();

        if (!isSuccessful)
        {
            // ネットワーク接続が失われたことによる失敗であれば、そのように表示します
            if (nn::nifm::IsNetworkAvailable())
            {
                Printf("Communication failed.\n");
            }
            else
            {
                Printf("Lost network.\n");

                // 切断理由表示のために呼び出すこともできます（必須ではありません）
                nn::nifm::HandleNetworkRequestResult();
            }
        }
    }


    // ネットワーク接続が不要になったら要求を取り下げます
    // 予期しない理由でネットワーク接続が失われていた場合、
    // あらためて明示的に要求を取り下げる必要はありませんが、
    // 重ねて取り下げをおこなっても問題はありません
    nn::nifm::CancelNetworkRequest();
    NN_ASSERT(!nn::nifm::IsNetworkAvailable());
}
