﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{NgcSimple.cpp,PageSampleNgcSimple}

    @brief
    NGC ライブラリのシンプルなサンプルプログラム
 */

/**
 * @page PageSampleNgcSimple NGCライブラリの基本操作
 * @tableofcontents
 *
 * @brief
 *  NGC ライブラリを使用した簡単な不正文字列処理のサンプルプログラムの解説です。
 *
 * @section PageSampleNgcSimple_SectionBrief 概要
 *  NGC ライブラリを使用して、不正文字の判定やマスクを行うサンプルです。
 *
 * @section PageSampleNgcSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/NgcSimple
 *  Samples/Sources/Applications/NgcSimple @endlink 以下にあります。
 *
 * @section PageSampleNgcSimple_SectionNecessaryEnvironment 必要な環境
 *  特にありません。
 *
 * @section PageSampleNgcSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleNgcSimple_SectionPrecaution 注意事項
 *  本サンプルプログラムを実機ターゲット環境で実行する場合、nspd ディレクトリにアプリケーション管理データがないと CheckProfanityWords と MaskProfanityWordsInText の実行でアサートが発生します。@n
 *  「機能／アプリケーションイメージの作成／ツールリファレンス／AuthoringTool」の「nspd に配置するメタデータを作成する」を参照してアプリケーション管理データを作成して下さい。@n
 *  この手順は将来的には、不要になる予定です。
 *
 * @section PageSampleNgcSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleNgcSimple_SectionDetail 解説
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 * - nn::fsの初期化
 * - 数を表す文字のカウント処理のサンプル
 *     - 数値を表す文字を含む文字列を指定してカウント処理を実行
 *     - カウント結果の表示
 * - 不正文字チェックのサンプル
 *     - ProfanityFilterクラスのインスタンス作成と初期化
 *     - 本体にインストールされているNGワードパターンのバージョン情報を取得
 *     - 使用するパターンリストに日本語と中国語を追加指定してチェック処理を実行
 *     - チェック結果の表示
 * - 不正文字マスクのサンプル
 *     - ProfanityFilterクラスのインスタンス作成と初期化
 *     - マスク方法を「上書き」に設定
 *     - 使用するパターンリストに日本語を追加指定してマスク処理を実行
 *     - マスクした文字列の数の表示
 *     - マスク処理後の文字列の表示
 *     - マスク方法を「1文字のアスタリスク記号」に設定
 *     - 使用するパターンリストに日本語を追加指定してマスク処理を実行
 *     - マスクした文字列の数の表示
 *     - マスク処理後の文字列の表示
 * - アットマークチェックのサンプル
 *     - ProfanityFilterクラスのインスタンス作成と初期化
 *     - アットマークチェックをスキップするように設定
 *     - アットマークチェックを実施
 *     - チェック結果を表示
 *     - アットマークチェックをスキップしないように設定
 *     - アットマークチェックを実施
 *     - チェック結果を表示
 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/fs.h>
#include <nn/ngc.h>
#include <nn/util/util_StringUtil.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#pragma execution_character_set("utf-8")
#endif

namespace
{
    /**
       @brief サンプルで使用するヒープ領域
    */
    const size_t MemoryHeapSize         = 128 * 1024 * 1024;
    const size_t MallocHeapSize         =  96 * 1024 * 1024;

    /**
       @brief FS用アロケータ
    */
    void* Allocate(size_t size)
    {
        auto p = std::malloc(size);
        return p;
    }

    /**
       @brief FS用デアロケータ
    */
    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }
} // namespace

NN_ALIGNAS(4096) char g_NgcWorkBuffer[nn::ngc::ProfanityFilter::WorkMemorySize];

void CountNumbersSample()
{
    NN_LOG("\n### 数を表す文字のカウントのサンプルです\n");

    // 数値を表す文字を含む文字列を指定してカウント処理を実行
    const char* CountCheckText = "あ123い４５６う⑦ⅷⅨえ⓪お";
    int num = nn::ngc::CountNumbers(CountCheckText);

    // カウント結果の表示
    NN_LOG("\"%s\" に含まれる数値の数は %d 個です\n",CountCheckText,num);
}

void CheckProfanityWordsSample()
{
    NN_LOG("\n### 文字列チェックのサンプルです\n");

    // ProfanityFilterクラスのインスタンス作成と初期化
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_NgcWorkBuffer,nn::ngc::ProfanityFilter::WorkMemorySize);
    if(result.IsFailure())
    {
        NN_LOG("ProfanityFilterクラスの初期化に失敗しました\n");
        return;
    }

    // 本体にインストールされているNGワードパターンのバージョン情報を取得
    uint32_t nVersion = filter.GetContentVersion();
    if(nVersion == 0)
    {
        NN_LOG("パターンリストのバージョンが不正です\n");
        return;
    }
    NN_LOG( "パターンリストのバージョンは %d です\n", nVersion );

    // 現在の本体リージョンおよび本体言語設定を参照し、
    // テキストデータが他のリージョンと行き来することを想定したNGワードチェックを実行
    const char* ppWords[] =
    {
        "mario",
        "@mario",
        "badword",
        "badwordj",
        "sbadwordjl",
        "testdisney.ne.jptest",
        "クッパ",
        "直アド",
        "hoge直アドhoge",
        "badwordc",
        "bbwczh",
        "tchinesenewsnett",
        "badwordp",
        "bbwpru",
        "badwordc",
        "bbwcom",
    };
    const int WordNum = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 bResults[ WordNum ];
    // 使用するパターンリストに日本語と中国語を追加指定してチェック処理を実行
    nn::Bit32 patterns
        = nn::ngc::ProfanityFilterPatternList_Japanese
          | nn::ngc::ProfanityFilterPatternList_SimplifiedChinese;
    result = filter.CheckProfanityWords( bResults, patterns, ppWords, WordNum );
    if(result.IsFailure())
    {
        NN_LOG("不正文字のチェック処理に失敗しました\n");
        return;
    }

    // チェック結果の表示
    for ( int i = 0; i < WordNum; i++ )
    {
        if ( bResults[ i ] == 0 )
        {
            NN_LOG("\"%s\" はパターンリストにマッチするものがありません\n", ppWords[i]);
        }
        else
        {
            NN_LOG("\"%s\" はパターンリストにマッチするものがあります\n", ppWords[i]);
        }
    }
}

void MaskProfanityWordsSample()
{
    NN_LOG("\n### 文字列マスクのサンプルです\n");

    // ProfanityFilterクラスのインスタンス作成と初期化
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_NgcWorkBuffer,nn::ngc::ProfanityFilter::WorkMemorySize);
    if(result.IsFailure())
    {
        NN_LOG("ProfanityFilterクラスの初期化に失敗しました\n");
        return;
    }

    int nWordCount;
    char checkText[] =
            "こんにちは。はじめまして。\n"
            "一緒にゲームをしていて あなたのことが(以下略)\n"
            "もしよかったらhoge@foobar.netまでメールをくれませんか？\n"
            "偽物のアドレスhoge@foobarはマスクされてはいけません！\n"
            "直アドはマスクされます！";
    char convertBuffer[nn::ngc::TextLengthMax];
    std::memset(convertBuffer,0x00,sizeof(convertBuffer));
    nn::util::Strlcpy(convertBuffer, checkText, sizeof(convertBuffer));

    nn::Bit32 patterns
        = nn::ngc::ProfanityFilterPatternList_Japanese;

    NN_LOG("## 不正な文字列内のすべての文字を*で上書きします\n");

    // マスク方法を「上書き」に設定
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_OverWrite );
    // 使用するパターンリストに日本語を追加指定してマスク処理を実行
    result = filter.MaskProfanityWordsInText( &nWordCount, convertBuffer ,patterns);
    if(result.IsFailure())
    {
        NN_LOG("不正文字のマスク処理に失敗しました\n");
        return;
    }

    // マスクした文字列の数の表示
    NN_LOG("マスクした文字列は %d 個です\n",nWordCount);

    // マスク処理後の文字列の表示
    NN_LOG("%s\n",convertBuffer);

    NN_LOG("## 不正な文字列を1文字の*で置換します\n");
    std::memset(convertBuffer,0x00,sizeof(convertBuffer));
    nn::util::Strlcpy(convertBuffer, checkText, sizeof(convertBuffer));

    // マスク方法を「1文字のアスタリスク記号」に設定
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );
    // 使用するパターンリストに日本語を追加指定してマスク処理を実行
    result = filter.MaskProfanityWordsInText( &nWordCount, convertBuffer ,patterns);
    if(result.IsFailure())
    {
        NN_LOG("不正文字のマスク処理に失敗しました\n");
        return;
    }

    // マスクした文字列の数の表示
    NN_LOG("マスクした文字列は %d 個です\n",nWordCount);

    // マスク処理後の文字列の表示
    NN_LOG("%s\n",convertBuffer);

    return;

}

void CheckAtSignSample()
{
    NN_LOG("\n### アットマークチェックのサンプルです\n");

    // ProfanityFilterクラスのインスタンス作成と初期化
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_NgcWorkBuffer,nn::ngc::ProfanityFilter::WorkMemorySize);
    if(result.IsFailure())
    {
        NN_LOG("ProfanityFilterクラスの初期化に失敗しました\n");
        return;
    }

    const char* ppWords[] =
    {
        "@mario",
        "mario",
    };
    const int WordNum = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 bResults[ WordNum ];
    nn::Bit32 patterns
        = 1 << nn::ngc::ProfanityFilterPatternList_Japanese;

    // アットマークチェックをスキップするように設定
    NN_LOG("## アットマークのスキップを有効にします\n");
    filter.SkipAtSignCheck( nn::ngc::ProfanityFilter::SkipMode_SkipAtSign );
    // アットマークチェックを実施
    result = filter.CheckProfanityWords( bResults, patterns, ppWords, WordNum );
    if(result.IsFailure())
    {
        NN_LOG("不正文字のチェック処理に失敗しました\n");
        return;
    }

    // チェック結果を表示
    for ( int i = 0; i < WordNum; i++ )
    {
        if ( bResults[ i ] == 0 )
        {
            NN_LOG("\"%s\" はアットマークが含まれません\n", ppWords[i]);
        }
        else
        {
            NN_LOG("\"%s\" はアットマークが含まれます\n", ppWords[i]);
        }
    }

    NN_LOG("## アットマークのスキップを無効にします\n");
    // アットマークチェックをスキップしないように設定
    filter.SkipAtSignCheck( nn::ngc::ProfanityFilter::SkipMode_NotSkip );
    // アットマークチェックを実施
    result = filter.CheckProfanityWords( bResults, patterns, ppWords, WordNum );
    if(result.IsFailure())
    {
        NN_LOG("不正文字のチェック処理に失敗しました\n");
        return;
    }

    // チェック結果を表示
    for ( int i = 0; i < WordNum; i++ )
    {
        if ( bResults[ i ] == 0 )
        {
            NN_LOG("\"%s\" はアットマークが含まれません\n", ppWords[i]);
        }
        else
        {
            NN_LOG("\"%s\" はアットマークが含まれます\n", ppWords[i]);
        }
    }
}

// アプリケーションのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// アプリケーションのエントリポイント
extern "C" void nnMain()
{
    // nn::fs の初期化
    nn::fs::SetAllocator(Allocate, Deallocate);

    // 数を表す文字のカウント処理のサンプル
    CountNumbersSample();

    // 不正文字チェックのサンプル
    CheckProfanityWordsSample();

    // 不正文字マスクのサンプル
    MaskProfanityWordsSample();

    // アットマークチェックのサンプル
    CheckAtSignSample();
}
