﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nfp.h>
#include "TagAccessorBase.h"

namespace nns { namespace nfp {

    /*!
        @brief      スレッドに依頼する処理のコマンドです。
     */
    enum Command
    {
        Command_None = TagAccessorBase::Command_None,
        Command_Initialize,
        Command_Finalize,
        Command_ListDevices,
        Command_AttachEvents,
        Command_StartDetection,
        Command_StopDetection,
        Command_Mount,
        Command_Unmount,
        Command_CreateApplicationArea,
        Command_OpenApplicationArea,
        Command_GetApplicationArea,
        Command_SetApplicationArea,
        Command_Flush,
        Command_Restore,
        Command_GetTagInfo,
        Command_GetRegisterInfo,
        Command_GetCommonInfo,
        Command_GetModelInfo,
        Command_StartNicknameAndOwnerSettings,
        Command_StartGameDataEraser,
        Command_StartRestorer,
        Command_GetDeviceState,
        Command_AttachAvailabilityChangeEvent,

        Command_CountMax
    };

    /*!
        @brief      NFP ライブラリの処理に非同期に行うクラスです。

        @details
                    NFP ライブラリで提供される API のほとんどは長時間処理をブロックしてしまいます。
                    このクラスは NFP ライブラリの API を別スレッドで実行する実装のサンプルです。

                    実行する処理に対応するコマンド識別子を @ref Enqueue() で登録し、
                    @ref Run() を呼びだすことで登録した処理をバックグラウンドで実行します。
                    登録できるコマンド識別子の最大数は @ref CommandCountMax です。

                    登録した処理が全て完了した場合、あるいは何らかのエラーが発生して中断された場合には
                    @ref Run() の引数に指定したイベントにシグナルが発行されます。
                    いくつの処理が実行されたかと最後の処理結果は、
                    @ref GetExecutedCount() と @ref GetLastResult() で取得することができます。

                    本クラスのメソッドはスレッドセーフではありませんので、一つのスレッドから呼び出すようにしてください。
     */
    class TagAccessor : public TagAccessorBase
    {
        NN_DISALLOW_COPY( TagAccessor );
        NN_DISALLOW_MOVE( TagAccessor );

    public:

        /*!
            @brief      TagAccessor のインスタンスを生成します。
            @return     TagAccessor のインスタンスです。
        */
        static TagAccessor& CreateInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスを取得します。
            @return     TagAccessor のインスタンスです。
        */
        static TagAccessor& GetInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスを破棄します。
        */
        static void DestroyInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスが存在するか判定します。
            @return     存在する場合は true を返します。
        */
        static bool HasInstance() NN_NOEXCEPT;

        /*!
            @brief      NFP ライブラリを初期化します。
            @return     登録された処理の位置です。
        */
        int Initialize() NN_NOEXCEPT;

        /*!
            @brief      NFP ライブラリを終了します。
            @return     登録された処理の位置です。

            @details    タグがマウントされている場合には自動的にアンマウントされ、検知も停止します。
                        NFC デバイスの状態が DeviceState_Active 状態または DeviceState_Mount 状態であれば、
                        @ref AttachEvents() で設定した Deactivate イベントが発生します。
        */
        int Finalize() NN_NOEXCEPT;

        /*!
            @brief      NFC デバイスのハンドルのリストを取得します。
            @param[out]     pOutBuffer      取得したハンドルのリストを格納するバッファへのポインタを指定します。
            @param[out]     pOutCount       実際に取得したハンドルの数を格納するバッファへのポインタを指定します。
            @param[in]      bufferCount     取得するハンドルの最大数（バッファの要素数）を指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int ListDevices(nn::nfp::DeviceHandle* pOutBuffer, int* pOutCount, int bufferCount) NN_NOEXCEPT;

        /*!
            @brief      タグの発見・喪失を通知するイベントを設定し、NFC デバイスの Npad ID を取得します。
            @param[out]     pOutNpadId          Npad ID です。
            @param[out]     pActivateEvent      タグ発見通知に使用するイベントを指定します。
            @param[out]     pDeactivateEvent    タグ喪失通知に使用するイベントを指定します。
            @param[in]      pDeviceHandle       NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        引数として渡すイベントの初期化と解放は NFP ライブラリ側が担当します。
                        1 つのイベントで Activate と Deactivate の両方の通知を受け取ることはできません。
        */
        int AttachEvents(nn::hid::NpadIdType* pOutNpadId, nn::os::SystemEventType* pActivateEvent, nn::os::SystemEventType* pDeactivateEvent, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの検知を開始します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int StartDetection(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの検知を終了します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        タグがマウントされている場合には自動的にアンマウントされます。
                        DeviceState_Active 状態や DeviceState_Mount 状態であれば Deactivate イベントが発生します。
        */
        int StopDetection(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグをマウントします。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Mount(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグをアンマウントします。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Unmount(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグのアプリケーション専用領域を作成します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @param[in]      accessId        アプリケーション専用領域へのアクセスに使用する ID です。
            @param[in]      pInitialData    アプリケーション専用領域に書き込む初期化データです。
            @param[in]      initialDataSize 初期化データのサイズです。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int CreateApplicationArea(const nn::nfp::DeviceHandle* pDeviceHandle, nn::Bit32 accessId, void* pInitialData, uint32_t initialDataSize) NN_NOEXCEPT;

        /*!
            @brief      マウント時に用意した内部バッファのアプリケーション専用領域へのアクセスを有効化します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @param[in]      accessId        アプリケーション専用領域へのアクセスに使用する ID です。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int OpenApplicationArea(const nn::nfp::DeviceHandle* pDeviceHandle, nn::Bit32 accessId) NN_NOEXCEPT;

        /*!
            @brief      アプリケーション専用領域のデータを、マウント時に用意した内部バッファから取得します。
            @param[out]     pOutBuffer      データの読み込み先バッファです。
            @param[out]     pOutSize        実際に取得できたデータのサイズです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @param[in]      bufferSize      読み込むデータのサイズです。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetApplicationArea(void* pOutBuffer, size_t* pOutSize, const nn::nfp::DeviceHandle* pDeviceHandle, uint32_t bufferSize) NN_NOEXCEPT;

        /*!
            @brief      アプリケーション専用領域のデータを、マウント時に用意した内部バッファに設定します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @param[out]     pBuffer         データの読み込み先バッファです。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int SetApplicationArea(const nn::nfp::DeviceHandle* pDeviceHandle, const void* pData, uint32_t dataSize) NN_NOEXCEPT;

        /*!
            @brief      マウント時に用意した内部バッファの内容をタグに書き込みます。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Flush(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      データが破損したタグをフォーマットし、バックアップデータから復旧します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Restore(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの情報を取得します。
            @param[out]     pOutInfo        TagInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        タグの情報はマウントする前であっても取得できます。
                        また、NFP 以外の NFC タグの情報を取得することも可能です。
        */
        int GetTagInfo(nn::nfp::TagInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグのオーナー登録情報を、マウント時に用意した内部バッファから取得します。
            @param[out]     pOutInfo        RegisterInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetRegisterInfo(nn::nfp::RegisterInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの共用領域の情報を、マウント時に用意した内部バッファから取得します。
            @param[out]     pOutInfo        CommonInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetCommonInfo(nn::nfp::CommonInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの ROM 領域の情報を、マウント時に用意した内部バッファから取得します。
            @param[out]     pOutInfo        ModelInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetModelInfo(nn::nfp::ModelInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      ニックネームとオーナーの設定を行う amiibo 設定を開始します。
            @param[out]     pOutDeviceHandle    タグを更新した NFC デバイスのハンドルを格納するバッファへのポインタを指定します。
            @param[out]     pOutIsRegistered 更新後の登録情報の有無を格納するバッファへのポインタを指定します。
            @param[out]     pOutRegisterInfo 更新後の登録情報を格納するバッファへのポインタを指定します。@a pOutIsRegistered にて登録情報が「有り」を示す場合のみ参照してください。
            @param[in]      pStartParam         amiibo 設定開始時必要な共通パラメータです。
            @param[in]      pTagInfo            更新の対象となるタグを示す情報です。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int StartNicknameAndOwnerSettings(nn::nfp::DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, nn::nfp::RegisterInfo* pOutRegisterInfo, const nn::nfp::AmiiboSettingsStartParam* pStartParam, const nn::nfp::TagInfo* pTagInfo) NN_NOEXCEPT;

        /*!
            @brief      ゲームデータの消去を行う amiibo 設定を開始します。
            @param[out]     pOutDeviceHandle    タグを更新した NFC デバイスのハンドルを格納するバッファへのポインタを指定します。
            @param[in]      pStartParam         amiibo 設定開始時必要な共通パラメータです。
            @param[in]      pTagInfo            更新の対象となるタグを示す情報です。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int StartGameDataEraser(nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam* pStartParam, const nn::nfp::TagInfo* pTagInfo) NN_NOEXCEPT;

        /*!
            @brief      データの復旧を行う amiibo 設定を開始します。
            @param[out]     pOutDeviceHandle    タグを更新した NFC デバイスのハンドルを格納するバッファへのポインタを指定します。
            @param[in]      pStartParam         amiibo 設定開始時必要な共通パラメータです。
            @param[in]      pTagInfo            更新の対象となるタグを示す情報です。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int StartRestorer(nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam* pStartParam, const nn::nfp::TagInfo* pTagInfo) NN_NOEXCEPT;

        /*!
            @brief      デバイスの状態を取得します。
            @param[out]     pOutDeviceState DeviceState の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetDeviceState(nn::nfp::DeviceState* pOutDeviceState, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      デバイスの利用可/不可の変化を通知するイベントを設定します。
            @param[out]     pAvailabilityChangeEvent      通知に使用するイベントを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        引数として渡すイベントの初期化と解放は NFP ライブラリ側が担当します。
        */
        int AttachAvailabilityChangeEvent(nn::os::SystemEventType* pAvailabilityChangeEvent) NN_NOEXCEPT;

        /*!
            @brief      指定番目の登録された処理の結果を取得します。
            @param[out]     pOutResult  結果を格納するバッファへのポインタを指定します。
            @param[in]      index       登録した処理の位置です。
            @return     指定番目の登録された処理の有無です。
                        指定番目の処理が実行されなかった場合には false を返します。
         */
        bool GetResult(nn::Result* pOutResult, int index) const NN_NOEXCEPT
        {
            return TagAccessorBase::GetNthResult(pOutResult, index);
        }

        /*!
            @brief      指定されたコマンド識別子の実行結果を取得します。
            @param[out]     pOutResult  結果を格納するバッファへのポインタを指定します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @return     指定されたコマンド識別子の実行結果の有無です。
                        指定されたコマンド識別子が実行されなかった場合には false を返します。

            @details    複数回実行されたコマンド識別子を指定した場合には、先の実行結果を取得します。
         */
        bool GetResult(nn::Result* pOutResult, Command command) const NN_NOEXCEPT
        {
            return TagAccessorBase::GetCommandResult(pOutResult, static_cast<uint32_t>(command));
        }

        /*!
            @brief      タグに処理を行っているかを判定します。
            @return     タグへの処理中なら真を返します。
         */
        virtual bool IsBusy() NN_NOEXCEPT NN_OVERRIDE;

    protected:

        /*!
            @brief      コマンド識別子で表される処理を実行します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @param[in]      param       実行する処理に渡すパラメータです。
            @return     実行された処理の結果です。
         */
        virtual nn::Result Execute(uint32_t command, const ParameterBuffer& param) NN_NOEXCEPT NN_OVERRIDE;

    private:

        /*!
            @brief      NFP ライブラリの処理をバックグラウンドで実行するスレッドを開始します。
         */
        TagAccessor() NN_NOEXCEPT {}

        /*!
            @brief      スレッドが稼働中であれば終了します。
         */
        virtual ~TagAccessor() NN_NOEXCEPT NN_OVERRIDE {}

        //! 唯一存在する TagAccessor のインスタンスのポインタです。
        static TagAccessor* g_pInstance;

    };

}} // end of namespace nns::nfp
