﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nfp.h>
#include "Graphics.h"
#include "TagUtility.h"
#include "WriteState.h"

namespace nns { namespace nfp { namespace {

    char g_Buffer[64];
    const int LabeledValueOffset = 20;
    const char BlackSquare[] = {'\xe2', '\x96', '\xa0', '\x00'}; //U+25A0 "■"

    inline int ToElapsedFrame(int playerIndex, const ApplicationData& data) NN_NOEXCEPT
    {
        return data.frame - data.playerData[playerIndex].transitFrame;
    }

    inline void WriteLabeledValue(
            int playerIndex,
            int x, int y,
            const char* label,
            const char* value
        ) NN_NOEXCEPT
    {
        WriteText(playerIndex, x, y, label);
        WriteText(playerIndex, x + LabeledValueOffset, y, value);
    }

    void WriteTagInfo(
            int playerIndex,
            int& x, int& y,
            const nn::nfp::TagInfo& tagInfo
        ) NN_NOEXCEPT
    {
        const nn::Bit8* uid = reinterpret_cast<const nn::Bit8*>(tagInfo.tagId.uid);
        char* buffer = g_Buffer;
        std::sprintf(buffer, ":");
        buffer += 1;
        for(int i = 0; i < tagInfo.tagId.length && i < sizeof(tagInfo.tagId.uid); i++)
        {
            std::sprintf(buffer, " %02X", uid[i]);
            buffer += 3;
        }
        WriteLabeledValue(playerIndex, x, y++, "UID", g_Buffer);

        std::sprintf(g_Buffer, ": %d", tagInfo.protocol);
        WriteLabeledValue(playerIndex, x, y++, "PROTOCOL", g_Buffer);

        std::sprintf(g_Buffer, ": %d", tagInfo.type);
        WriteLabeledValue(playerIndex, x, y++, "TYPE", g_Buffer);
    }

    void WriteModelInfo(
            int playerIndex,
            int& x, int& y,
            const nn::nfp::ModelInfo& modelInfo
        ) NN_NOEXCEPT
    {
        std::sprintf(g_Buffer, ": %s", GetCharacterName(modelInfo.characterId));
        WriteLabeledValue(playerIndex, x, y++, "CHARACTER", g_Buffer);

        std::sprintf(g_Buffer, ": %d", modelInfo.nfpType);
        WriteLabeledValue(playerIndex, x, y++, "NFP TYPE", g_Buffer);
    }

    void WriteCaption(
            int playerIndex,
            int x, int y,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        const nn::util::Unorm8x4 backNumberColor = { { 255, 255, 255, 8 } };
        std::sprintf(g_Buffer, "%d", playerIndex + 1);
        WriteText(playerIndex, 1.5f, 0.05f, g_Buffer, backNumberColor, 15.0f);

        if(!data.playerData[playerIndex].isEnabled)
        {
            WriteText(playerIndex, x, y, BlackSquare);
            x += 4;
            WriteLoadingText(playerIndex, x, y, "Waiting for connection from Npad", ToElapsedFrame(playerIndex, data));
        }
        else
        {
            // プレイヤーランプ
            for(int i = 0; i < 4; i++)
            {
                nn::util::Unorm8x4 color;
                if((data.playerData[playerIndex].ledPattern & (1 << i)) == 0)
                {
                    color = Color::Gray;
                }
                else
                {
                    color = Color::Green;
                }
                WriteText(playerIndex, x, y, BlackSquare, color);
                x += 2;
            }

            x++;

            if(!data.playerData[playerIndex].isNfcDevice)
            {
                std::sprintf(g_Buffer, "Npad ID 0x%02X", data.playerData[playerIndex].npadId);
                WriteText(playerIndex, x, y, g_Buffer);
            }
            else
            {
                std::sprintf(g_Buffer, "Npad ID 0x%02X (with NFC device)", data.playerData[playerIndex].npadId);
                WriteText(playerIndex, x, y, g_Buffer);
            }
        }

    }

    void WritePlayerStateNone(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;

        WriteCaption(playerIndex, x, y++, data);
    }

    void WritePlayerStateDeviceInitialize(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;

        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Connecting", ToElapsedFrame(playerIndex, data));
    }

    void WritePlayerStateAttachEvent(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Attaching", ToElapsedFrame(playerIndex, data));
    }

    void WritePlayerStateIdle(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteText(playerIndex, x, y++, "A: start detection");
    }

    void WritePlayerStateStartDetection(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Starting", ToElapsedFrame(playerIndex, data));
    }

    void WritePlayerStateStopDetection(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Suspending", ToElapsedFrame(playerIndex, data));
    }

    void WritePlayerStateTagRelease(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Finished", ToElapsedFrame(playerIndex, data));

        ++y;
        WriteText(playerIndex, x, y++, "Please remove the tag");
    }

    void WritePlayerStateTagSearch(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Searching", ToElapsedFrame(playerIndex, data));

        ++y;
        std::sprintf(g_Buffer, "Please touch the tag to Npad ID 0x%02X", data.playerData[playerIndex].npadId);
        WriteText(playerIndex, x, y++, g_Buffer);
        WriteText(playerIndex, x, y++, "B: cancel");
    }

    void WritePlayerStateDeviceCheck(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Checking", ToElapsedFrame(playerIndex, data));
    }

    void WritePlayerStateTagRead(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Reading", ToElapsedFrame(playerIndex, data));

        ++y;
        WriteText(playerIndex, x, y++, "Do not remove the tag during access");
    }

    void WritePlayerStateTagCheck(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Checking", ToElapsedFrame(playerIndex, data));

        ++y;
        WriteText(playerIndex, x, y++, "Do not remove the tag during access");
    }

    void WritePlayerStateTagWrite(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Writing", ToElapsedFrame(playerIndex, data));

        ++y;
        WriteText(playerIndex, x, y++, "Do not remove the tag during access");
    }

    void WritePlayerStateTagEnabled(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteText(playerIndex, x, y++, "A: update app counter");
        WriteText(playerIndex, x, y++, "B: cancel");

        ++y;
        WriteTagInfo(playerIndex, x, y, data.playerData[playerIndex].tagInfo);

        ++y;
        WriteModelInfo(playerIndex, x, y, data.playerData[playerIndex].modelInfo);

        ++y;
        std::sprintf(g_Buffer, ": %u", data.playerData[playerIndex].counter);
        WriteLabeledValue(playerIndex, x, y++, "APP COUNTER", g_Buffer);
    }

    void WritePlayerStateNeedCreate(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteErrorText(playerIndex, x, y++,  "Application data not found.");

        ++y;
        WriteText(playerIndex, x, y++, "A: create");
        WriteText(playerIndex, x, y++, "B: cancel");
    }

    void WritePlayerStateNeedRestore(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteErrorText(playerIndex, x, y++, "Could not read tag data.");

        ++y;
        WriteText(playerIndex, x, y++, "A: restore");
        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(playerIndex, x, y++, "X: restore (amiibo settings)");
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(playerIndex, x, y++, "B: cancel");
    }

    void WritePlayerStateNeedFormat(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteErrorText(playerIndex, x, y++,  "Could not read tag data.");
        ++y;
        WriteErrorText(playerIndex, x, y++,  "You should format this tag using amiibo settings");
        WriteErrorText(playerIndex, x, y++,  "found in system settings.");
        ++y;
        WriteErrorText(playerIndex, x, y++,  "Alternatively, if you have backup data stored on");
        WriteErrorText(playerIndex, x, y++,  "other console, you can use it to restore the");
        WriteErrorText(playerIndex, x, y++,  "original data.");
        ++y;
        WriteText(playerIndex, x, y++, "B: cancel");
    }

    void WritePlayerStateNeedRegister(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteErrorText(playerIndex, x, y++,  "Registration data not found.");

        ++y;
        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(playerIndex, x, y++, "X: register (amiibo settings)");
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(playerIndex, x, y++, "B: cancel");
    }

    void WritePlayerStateNeedDelete(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteErrorText(playerIndex, x, y++,  "Already exists other application's data.");

        ++y;
        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(playerIndex, x, y++, "X: delete (amiibo settings)");
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(playerIndex, x, y++, "B: cancel");
    }

    void WritePlayerStateAmiiboNotSupported(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteErrorText(playerIndex, x, y++,  "The amiibo can not be used with this application.");
        ++y;
        WriteText(playerIndex, x, y++, "B: cancel");
    }

    void WritePlayerStateAmiiboSettings(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);
        WriteLoadingText(playerIndex, x, y++, "Running Amiibo Settings", ToElapsedFrame(playerIndex, data));
    }

    void WritePlayerStateError(
            int playerIndex,
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteCaption(playerIndex, x, y++, data);

        WriteErrorText(playerIndex, x, y, "ERROR: ");
        x += 7;

        nn::Result result = data.playerData[playerIndex].lastResult;
        if( result <= nn::nfp::ResultNfcDeviceNotFound() )
        {
            WriteErrorText(playerIndex, x, y++, "nfc device not found");
        }
        else if( result <= nn::nfp::ResultNfcDisabled() )
        {
            WriteErrorText(playerIndex, x, y++, "nfc disabled");
        }
        else if( result <= nn::nfp::ResultNeedRetry() )
        {
            WriteErrorText(playerIndex, x, y++, "please retry");
        }
        else if( result <= nn::nfp::ResultNeedRestart() ||
                 result <= nn::nfp::ResultAccessIdMisMatch() ||
                 result <= nn::nfp::ResultNotBroken() )
        {
            WriteErrorText(playerIndex, x, y++, "keep applying the same tag");
        }
        else if( result <= nn::nfp::ResultNeedRegister() )
        {
            WriteErrorText(playerIndex, x, y++, "need register");
        }
        else if( result <= nn::nfp::ResultNeedCreate() )
        {
            WriteErrorText(playerIndex, x, y++, "need create");
        }
        else if( result <= nn::nfp::ResultNeedRestore() )
        {
            WriteErrorText(playerIndex, x, y++, "need restore");
        }
        else if( result <= nn::nfp::ResultNeedFormat() )
        {
            WriteErrorText(playerIndex, x, y++, "need format");
        }
        else if( result <= nn::nfp::ResultAlreadyCreated() )
        {
            WriteErrorText(playerIndex, x, y++, "already created");
        }
        else if( result <= nn::nfp::ResultNotSupported() ||
                 result <= nn::nfp::ResultInvalidFormatVersion() )
        {
            WriteErrorText(playerIndex, x, y++, "this tag is not supported");
        }
        else if( result <= nn::nfp::ResultMaxNfcDeviceActivated() )
        {
            WriteErrorText(playerIndex, x, y++, "nfc device already activated");
        }
        else if( result <= nn::nfp::ResultConflictFunction() )
        {
            WriteErrorText(playerIndex, x, y++, "nfc device is conflicted");
        }
        else if( result <= nn::nfp::ResultNotUpdated() )
        {
            WriteErrorText(playerIndex, x, y++, "not been updated by amiibo setting");
        }
        else
        {
            WriteErrorText(playerIndex, x, y++, "unexpected");
        }

        x  = 1;
        y += 1;
        WriteText(playerIndex, x, y++, "A: restart");
    }

    typedef void (*WritePlayerStateFunc)( int playerIndex, const ApplicationData& data );
    struct WritePlayerStateEntry
    {
        PlayerState           state;
        WritePlayerStateFunc  func;
    };

    const WritePlayerStateEntry WritePlayerStateTable[] = {
        {   PlayerState_None,                 WritePlayerStateNone              },
        {   PlayerState_DeviceInitialize,     WritePlayerStateDeviceInitialize  },
        {   PlayerState_AttachEvent,          WritePlayerStateAttachEvent       },
        {   PlayerState_Idle,                 WritePlayerStateIdle              },
        {   PlayerState_StartDetection,       WritePlayerStateStartDetection    },
        {   PlayerState_StopDetection,        WritePlayerStateStopDetection     },
        {   PlayerState_TagRelease,           WritePlayerStateTagRelease        },
        {   PlayerState_TagSearch,            WritePlayerStateTagSearch         },
        {   PlayerState_DeviceCheck,          WritePlayerStateDeviceCheck       },
        {   PlayerState_TagRead,              WritePlayerStateTagRead           },
        {   PlayerState_TagCheck,             WritePlayerStateTagCheck          },
        {   PlayerState_TagWrite,             WritePlayerStateTagWrite          },
        {   PlayerState_TagEnabled,           WritePlayerStateTagEnabled        },
        {   PlayerState_NeedCreate,           WritePlayerStateNeedCreate        },
        {   PlayerState_NeedRestore,          WritePlayerStateNeedRestore       },
        {   PlayerState_NeedFormat,           WritePlayerStateNeedFormat        },
        {   PlayerState_NeedRegister,         WritePlayerStateNeedRegister      },
        {   PlayerState_NeedDelete,           WritePlayerStateNeedDelete        },
        {   PlayerState_AmiiboNotSupported,   WritePlayerStateAmiiboNotSupported },
        {   PlayerState_AmiiboSettings,       WritePlayerStateAmiiboSettings    },
        {   PlayerState_Error,                WritePlayerStateError             },
    };
    const int WritePlayerStateCountMax = sizeof(WritePlayerStateTable) / sizeof(WritePlayerStateTable[0]);
    NN_STATIC_ASSERT( WritePlayerStateCountMax == static_cast<int>(PlayerState_CountMax) );

    WritePlayerStateFunc GetWritePlayerStateFunc(PlayerState state)
    {
        const int index = static_cast<int>(state);
        NN_ASSERT( index >= 0 && index < WritePlayerStateCountMax );
        NN_ASSERT( WritePlayerStateTable[index].state == state );
        return WritePlayerStateTable[index].func;
    }

}}} // end of namespace nns::nfp::unnamed

namespace nns { namespace nfp {

    void WriteState(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        for(int i = 0; i < PlayerCountMax; i++)
        {
            (*GetWritePlayerStateFunc(data.playerData[i].state))(i, data);
        }
    }

}} // end of namespace nns::nfp
