﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include "HidController.h"
#include "TagUtility.h"
#include "TagAccessor.h"
#include "UpdateState.h"

//------------------------------------------------------------------------------
//  NFP 処理フロー
//------------------------------------------------------------------------------
namespace nns { namespace nfp { namespace {

    //--------------------------------------------------------------------------
    //  定数、型、変数定義
    //--------------------------------------------------------------------------

    // アプリケーション専用領域に書き込まれるカウンタの初期値です。
    const uint32_t InitialCounter = 0;

    // アプリケーション状態処理関数の型です。
    typedef void (*UpdateStateFunc)( ApplicationData& data );

    // プレイヤー状態処理関数の型です。
    typedef void (*UpdatePlayerStateFunc)( int playerIndex, ApplicationData& data );

    // タグ発見、タグ喪失時の状態遷移情報です。
    struct PlayerStateChain
    {
        PlayerState                          prev;
        PlayerState                          next;
        UpdatePlayerStateFunc                writer;
        nn::Result                           result;
    };

    // アプリケーション専用領域に書き込むデータ型です。
    struct ApplicationAreaData
    {
        uint32_t                            counter;
    };

    // amiibo 設定用の作業データ型です。
    struct AmiiboSettingsData
    {
        bool                                isRegistered;
        nn::nfp::AmiiboSettingsStartParam   startParam;
        nn::nfp::DeviceHandle               deviceHandle;
    };

    // プレイヤー用の作業データ型です。
    struct PlayerWorkData
    {
        PlayerStateChain                    chain;
        nn::os::EventType                   processFinishEvent;

        nn::nfp::DeviceHandle               deviceHandle;

        nn::os::SystemEvent*                activateEvent;
        nn::os::SystemEvent*                deactivateEvent;

        nn::nfp::TagInfo                    tagInfo;
        nn::nfp::ModelInfo                  modelInfo;
        nn::nfp::CommonInfo                 commonInfo;
        nn::nfp::RegisterInfo               registerInfo;
        ApplicationAreaData                 applicationAreaData;
        size_t                              applicationAreaSize;
        AmiiboSettingsData                  amiiboSettingsData;
        nn::nfp::DeviceState                deviceState;
    };

    // アプリケーション用の作業データ型です。
    struct ApplicationWorkData
    {
        nn::os::EventType                   processFinishEvent;

        DeviceInfo                          deviceInfo[ DeviceCountMax ];
        int                                 deviceCount;
        int                                 deviceIndex;

        PlayerWorkData                      playerData[ PlayerCountMax ];
        nn::os::SystemEvent*                availabilityChangeEvent;
    };

    // NFP 処理用の作業変数です。
    ApplicationWorkData g_Work;
    NN_ALIGNAS(16) char g_ActivateEventBuffer[ sizeof(nn::os::SystemEvent) * PlayerCountMax ];
    NN_ALIGNAS(16) char g_DeactivateEventBuffer[ sizeof(nn::os::SystemEvent) * PlayerCountMax ];

    //--------------------------------------------------------------------------
    //  通知イベント、デバイスハンドルの生成・破棄
    //--------------------------------------------------------------------------

    // 通知イベント、デバイスハンドルを破棄します。
    void DestroySystemEvents(int playerIndex) NN_NOEXCEPT
    {
        if( g_Work.playerData[playerIndex].activateEvent != nullptr )
        {
            g_Work.playerData[playerIndex].activateEvent->~SystemEvent();
            g_Work.playerData[playerIndex].activateEvent = nullptr;
        }
        if( g_Work.playerData[playerIndex].deactivateEvent != nullptr )
        {
            g_Work.playerData[playerIndex].deactivateEvent->~SystemEvent();
            g_Work.playerData[playerIndex].deactivateEvent = nullptr;
        }
    }

    // 通知イベント、デバイスハンドルを生成します。
    void CreateSystemEvent(int playerIndex) NN_NOEXCEPT
    {
        DestroySystemEvents(playerIndex);
        g_Work.playerData[playerIndex].activateEvent = new(&g_ActivateEventBuffer[sizeof(nn::os::SystemEvent) * playerIndex]) nn::os::SystemEvent;
        g_Work.playerData[playerIndex].deactivateEvent = new(&g_DeactivateEventBuffer[sizeof(nn::os::SystemEvent) * playerIndex]) nn::os::SystemEvent;
    }

    void UpdateNpad(ApplicationData& data) NN_NOEXCEPT
    {
        // 接続されている Npad の ID リストを取得
        nn::hid::NpadIdType npadId[NpadIdCountMax];
        int connectedNpadCount = ListHidControllerConnectedNpadIds(npadId, NpadIdCountMax);

        bool isUsing[NpadIdCountMax] = {false};
        for(auto& playerData : data.playerData)
        {
            bool find = false;
            for(int i = 0; i < connectedNpadCount; i++)
            {
                if(playerData.isEnabled
                   && playerData.npadId == npadId[i])
                {
                    // 使用中
                    find = true;
                    isUsing[i] = true;
                    break;
                }
            }
            if(!find)
            {
                if(playerData.isEnabled)
                {
                    // 空きを作成
                    playerData.isEnabled = false;
                    playerData.isNfcDevice = false;
                    playerData.state = PlayerState_None;
                }
            }
        }

        // 空いているところに新たに使用するコントローラを割り当てる
        for(int i = 0; i < connectedNpadCount; i++)
        {
            if(isUsing[i])
            {
                // 使用中
                continue;
            }

            for(auto& playerData : data.playerData)
            {
                if(!playerData.isEnabled)
                {
                    playerData.isEnabled = true;
                    playerData.isNfcDevice = false;
                    playerData.npadId = npadId[i];
                    playerData.ledPattern = GetHidControllerPlayerLedPattern(playerData.npadId);
                    playerData.state = PlayerState_None;
                    break;
                }
            }
        }
    }

    // 選択中のデバイスハンドルを取得します。
    nn::nfp::DeviceHandle* GetDeviceHandle(int playerIndex) NN_NOEXCEPT
    {
        return &g_Work.playerData[playerIndex].deviceHandle;
    }

    // 選択中のタグ発見通知イベントを取得します。
    nn::os::SystemEvent& GetActivateEvent(int playerIndex) NN_NOEXCEPT
    {
        return *g_Work.playerData[playerIndex].activateEvent;
    }

    // 選択中のタグ喪失通知イベントを取得します。
    nn::os::SystemEvent& GetDeactivateEvent(int playerIndex) NN_NOEXCEPT
    {
        return *g_Work.playerData[playerIndex].deactivateEvent;
    }

    //--------------------------------------------------------------------------
    //  タグ検知開始処理
    //--------------------------------------------------------------------------

    // タグの検出を開始します。
    void DoStartDetection(int playerIndex, ApplicationData& data, PlayerState next, UpdatePlayerStateFunc writer) NN_NOEXCEPT
    {
        NN_ASSERT( (next == PlayerState_TagRead) || (next == PlayerState_TagWrite) );

        g_Work.playerData[playerIndex].chain.prev = data.playerData[playerIndex].state;
        g_Work.playerData[playerIndex].chain.next = next;
        g_Work.playerData[playerIndex].chain.writer = writer;
        g_Work.playerData[playerIndex].chain.result = nn::ResultSuccess();

        // 前回のタグ発見でイベントがシグナルされたままの状態になっている可能性があるため、
        // タグの発見を開始する前にイベントのシグナルをクリアしておくことを推奨します。
        GetActivateEvent(playerIndex).Clear();
        GetDeactivateEvent(playerIndex).Clear();

        PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
        accessor.StartDetection(GetDeviceHandle(playerIndex));
        accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
        data.playerData[playerIndex].state = PlayerState_StartDetection;
    }

    // タグの検出を開始します。タグの読み込みのみを行う場合に呼び出します。
    void DoStartDetectionForTagRead(int playerIndex, ApplicationData& data) NN_NOEXCEPT
    {
        DoStartDetection(playerIndex, data, PlayerState_TagRead, nullptr);
    }

    // タグの検出を開始します。タグの書き込みも行う場合に呼び出します。
    void DoStartDetectionForTagWrite(int playerIndex, ApplicationData& data, UpdatePlayerStateFunc writer) NN_NOEXCEPT
    {
        NN_ASSERT( writer != nullptr );

        DoStartDetection(playerIndex, data, PlayerState_TagWrite, writer);
    }

    // タグ発見時に処理されます。
    void OnTagActivated(int playerIndex, ApplicationData& data) NN_NOEXCEPT
    {
        NN_ASSERT( data.playerData[playerIndex].state == PlayerState_TagSearch );

        // タグを発見したら必要な情報を取得します。
        PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
        switch( g_Work.playerData[playerIndex].chain.next )
        {
        case PlayerState_TagRead:
            {
                accessor.GetTagInfo(&g_Work.playerData[playerIndex].tagInfo, GetDeviceHandle(playerIndex));
                accessor.Mount(GetDeviceHandle(playerIndex));
                accessor.GetModelInfo(&g_Work.playerData[playerIndex].modelInfo, GetDeviceHandle(playerIndex));
                accessor.GetCommonInfo(&g_Work.playerData[playerIndex].commonInfo, GetDeviceHandle(playerIndex));
                accessor.GetRegisterInfo(&g_Work.playerData[playerIndex].registerInfo, GetDeviceHandle(playerIndex));
                accessor.OpenApplicationArea(GetDeviceHandle(playerIndex), AccessId);
                accessor.GetApplicationArea(
                        &g_Work.playerData[playerIndex].applicationAreaData,
                        &g_Work.playerData[playerIndex].applicationAreaSize,
                        GetDeviceHandle(playerIndex),
                        sizeof(g_Work.playerData[playerIndex].applicationAreaData)
                    );
                accessor.Unmount(GetDeviceHandle(playerIndex));
                accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
                data.playerData[playerIndex].state = PlayerState_TagRead;
            }
            break;
        case PlayerState_TagWrite:
            {
                accessor.GetTagInfo(&g_Work.playerData[playerIndex].tagInfo, GetDeviceHandle(playerIndex));
                accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
                data.playerData[playerIndex].state = PlayerState_TagCheck;
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // 書き込み対象のタグ発見時に処理されます。
    void OnTagChecked(int playerIndex, ApplicationData& data) NN_NOEXCEPT
    {
        NN_ASSERT( data.playerData[playerIndex].state == PlayerState_TagCheck );
        NN_ASSERT( g_Work.playerData[playerIndex].chain.next == PlayerState_TagWrite );
        NN_ASSERT( g_Work.playerData[playerIndex].chain.writer != nullptr );

        // 書き込み処理を行って次遷移先を設定します。
        (*g_Work.playerData[playerIndex].chain.writer)(playerIndex, data);
        data.playerData[playerIndex].state = PlayerState_TagWrite;
    }

    //--------------------------------------------------------------------------
    //  タグ検知停止処理
    //--------------------------------------------------------------------------

    // タグの検出を停止します。
    void DoStopDetection(int playerIndex, ApplicationData& data, PlayerState next, nn::Result result) NN_NOEXCEPT
    {
        NN_ASSERT( next != PlayerState_None );

        g_Work.playerData[playerIndex].chain.prev = data.playerData[playerIndex].state;
        g_Work.playerData[playerIndex].chain.next = next;
        g_Work.playerData[playerIndex].chain.result = result;

        PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
        accessor.StopDetection(GetDeviceHandle(playerIndex));
        accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
        data.playerData[playerIndex].state = PlayerState_StopDetection;
    }

    // タグの検出を停止します。停止後に次の状態に遷移する場合に呼び出します。
    void DoStopDetectionForNext(int playerIndex, ApplicationData& data, PlayerState next) NN_NOEXCEPT
    {
        DoStopDetection(playerIndex, data, next, nn::ResultSuccess());
    }

    // タグ検知停止時に処理されます。
    void OnTagStopped(int playerIndex, ApplicationData& data) NN_NOEXCEPT
    {
        NN_ASSERT( data.playerData[playerIndex].state == PlayerState_StopDetection );
        NN_ASSERT( g_Work.playerData[playerIndex].chain.prev != PlayerState_None );
        NN_ASSERT( g_Work.playerData[playerIndex].chain.next != PlayerState_None );

        // 次遷移先と実行結果を設定します。
        data.playerData[playerIndex].state = g_Work.playerData[playerIndex].chain.next;
        data.playerData[playerIndex].lastResult = g_Work.playerData[playerIndex].chain.result;
    }


    //--------------------------------------------------------------------------
    //  タグ喪失待機処理
    //--------------------------------------------------------------------------

    // タグが喪失するまで待機します。
    void WaitTagRelease(int playerIndex, ApplicationData& data, PlayerState next, nn::Result result) NN_NOEXCEPT
    {
        NN_ASSERT( next != PlayerState_None );

        g_Work.playerData[playerIndex].chain.prev = data.playerData[playerIndex].state;
        g_Work.playerData[playerIndex].chain.next = next;
        g_Work.playerData[playerIndex].chain.result = result;

        data.playerData[playerIndex].state = PlayerState_TagRelease;
    }

    // タグが喪失するまで待機します。喪失後に次の状態に遷移する場合に呼び出します。
    void WaitTagReleaseForNext(int playerIndex, ApplicationData& data, PlayerState next) NN_NOEXCEPT
    {
        WaitTagRelease(playerIndex, data, next, nn::ResultSuccess());
    }

    // タグが喪失するまで待機します。喪失後にエラー表示を行う場合に呼び出します。
    void WaitTagReleaseForError(int playerIndex, ApplicationData& data, nn::Result result) NN_NOEXCEPT
    {
        NN_ASSERT( result.IsFailure() );
        WaitTagRelease(playerIndex, data, PlayerState_Error, result);
    }

    // タグ喪失時に処理されます。
    void OnTagDeactivated(int playerIndex, ApplicationData& data) NN_NOEXCEPT
    {
        NN_ASSERT( data.playerData[playerIndex].state == PlayerState_TagRelease );
        NN_ASSERT( g_Work.playerData[playerIndex].chain.prev != PlayerState_None );
        NN_ASSERT( g_Work.playerData[playerIndex].chain.next != PlayerState_None );

        PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
        accessor.StopDetection(GetDeviceHandle(playerIndex));
        accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
        data.playerData[playerIndex].state = PlayerState_StopDetection;
    }

    //--------------------------------------------------------------------------
    //  状態処理
    //--------------------------------------------------------------------------

    // NFP ライブラリを初期化していない状態です。
    void UpdateStateNone(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        nn::os::InitializeEvent(&g_Work.processFinishEvent, false, nn::os::EventClearMode_AutoClear);
        for(int i = 0; i < PlayerCountMax; i++)
        {
            nn::os::InitializeEvent(&g_Work.playerData[i].processFinishEvent, false, nn::os::EventClearMode_AutoClear);
        }

        TagAccessor& accessor = TagAccessor::CreateInstance();
        accessor.Initialize();
        g_Work.availabilityChangeEvent = new nn::os::SystemEvent();
        accessor.AttachAvailabilityChangeEvent(g_Work.availabilityChangeEvent->GetBase());
        accessor.Run(&g_Work.processFinishEvent);
        data.state = State_Initializing;
    }

    // NFP ライブラリを初期中です。
    void UpdateStateInitializing(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 初期化処理が完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                data.state = State_Initialized;
            }
            else
            {
                // 初期化処理に失敗することはありません。
                data.state = State_None;
            }
        }
    }

    // NFP ライブラリが利用可能な状態です。
    void UpdateStateInitialized(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // NFC デバイスの取得を開始します。
        g_Work.deviceCount = 0;

        TagAccessor& accessor = TagAccessor::GetInstance();
        accessor.ListDevices(g_Work.deviceInfo, &g_Work.deviceCount, DeviceCountMax);
        accessor.Run(&g_Work.processFinishEvent);
        data.state = State_DeviceListing;
    }

    // NFC デバイスリストを取得中です。
    void UpdateStateDeviceListing(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // NFC デバイスの取得が完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            bool existResult;
            nn::Result commandResult;
            existResult = accessor.GetResult(&commandResult, Command_ListDevices);
            if( !existResult || commandResult.IsFailure() )
            {
                data.state = State_Initialized;
                return;
            }

            UpdateNpad(data);

            for(int i = 0; i < PlayerCountMax; i++)
            {
                if(!data.playerData[i].isEnabled)
                {
                    continue;
                }

                bool isNfcDevice = false;
                nn::nfp::DeviceHandle deviceHandle = {0};
                for( int j = 0; j < g_Work.deviceCount; j++ )
                {
                    if(data.playerData[i].npadId == g_Work.deviceInfo[j].npadId)
                    {
                        deviceHandle = g_Work.deviceInfo[j].deviceHandle;
                        isNfcDevice = true;
                        break;
                    }
                }

                if(!isNfcDevice)
                {
                    data.playerData[i].isNfcDevice = false;
                    data.playerData[i].state = PlayerState_None;
                    continue;
                }

                if(data.playerData[i].isNfcDevice && data.playerData[i].state != PlayerState_None)
                {
                    if(data.playerData[i].state == PlayerState_TagSearch)
                    {
                        PlayerTagAccessor& playerAccessor = TagAccessor::GetInstance().GetPlayerTagAccessor(i);
                        playerAccessor.GetDeviceState(&g_Work.playerData[i].deviceState, GetDeviceHandle(i));
                        playerAccessor.Run(&g_Work.playerData[i].processFinishEvent);
                        data.playerData[i].state = PlayerState_DeviceCheck;
                    }
                    continue;
                }

                g_Work.playerData[i].deviceHandle = deviceHandle;
                data.playerData[i].state = PlayerState_DeviceInitialize;
                data.playerData[i].isNfcDevice = true;
            }

            data.state = State_Idle;
        }
    }

    // 次回の NFC デイバスリスト取得までの待機状態です。
    void UpdateStateIdle(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        if(g_Work.availabilityChangeEvent->TryWait())
        {
            // コントローラおよび NFC デバイスの状態をチェックします。
            data.state = State_Initialized;
        }
    }

    //--------------------------------------------------------------------------
    // プレイヤー状態処理
    //--------------------------------------------------------------------------
    void UpdatePlayerStateNone(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(playerIndex);
        NN_UNUSED(data);
    }

    // NFC デバイスリストを取得中です。
    void UpdatePlayerStateDeviceInitialize(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);

        CreateSystemEvent(playerIndex);
        accessor.AttachEvents(
            g_Work.playerData[playerIndex].activateEvent->GetBase(),
            g_Work.playerData[playerIndex].deactivateEvent->GetBase(),
            &g_Work.playerData[playerIndex].deviceHandle
                            );

        accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
        data.playerData[playerIndex].state = PlayerState_AttachEvent;
    }

    void UpdatePlayerStateAttachEvent(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // イベントの設定が完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.playerData[playerIndex].processFinishEvent) )
        {
            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                data.playerData[playerIndex].state = PlayerState_Idle;
            }
            else
            {
                data.playerData[playerIndex].state = PlayerState_Error;
                data.playerData[playerIndex].lastResult = result;
            }
        }
    }

    // タグの検知を開始の指示を待っている状態です。
    void UpdatePlayerStateIdle(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンでタグの検知を開始します。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::A::Mask) )
        {
            DoStartDetectionForTagRead(playerIndex, data);
        }
    }

    // タグの検知を開始しています。
    void UpdatePlayerStateStartDetection(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグの検知が開始したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.playerData[playerIndex].processFinishEvent) )
        {
            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                data.playerData[playerIndex].state = PlayerState_TagSearch;
            }
            else
            {
                data.playerData[playerIndex].state = PlayerState_Error;
                data.playerData[playerIndex].lastResult = result;
            }
        }
    }

    // タグの検知を停止しています。
    void UpdatePlayerStateStopDetection(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグの検知が停止したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.playerData[playerIndex].processFinishEvent) )
        {
            OnTagStopped(playerIndex, data);
        }
    }

    // タグの喪失待機中です。
    void UpdatePlayerStateTagRelease(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグが喪失したら状態遷移します。
        if( GetDeactivateEvent(playerIndex).TimedWait(nn::TimeSpan::FromMicroSeconds(100)) )
        {
            OnTagDeactivated(playerIndex, data);
        }
    }

    // タグを検索中です。
    void UpdatePlayerStateTagSearch(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグを発見したら情報を取得して状態遷移します。
        if( GetActivateEvent(playerIndex).TimedWait(nn::TimeSpan::FromMicroSeconds(100)) )
        {
            OnTagActivated(playerIndex, data);
        }

        // B ボタンでタグの検知を停止します。
        else if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::B::Mask) )
        {
            DoStopDetectionForNext(playerIndex, data, PlayerState_Idle);
        }
    }

    // NFC デバイスの状態を確認中です。
    void UpdatePlayerStateDeviceCheck(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        if( nn::os::TryWaitEvent(&g_Work.playerData[playerIndex].processFinishEvent) )
        {
            if(g_Work.playerData[playerIndex].deviceState != nn::nfp::DeviceState_Search
               && g_Work.playerData[playerIndex].deviceState != nn::nfp::DeviceState_Active)
            {
                data.playerData[playerIndex].state = PlayerState_Error;
                if(g_Work.playerData[playerIndex].deviceState == nn::nfp::DeviceState_Deactive)
                {
                    data.playerData[playerIndex].lastResult = nn::nfp::ResultNeedRestart();
                }
                else
                {
                    data.playerData[playerIndex].lastResult = nn::nfp::ResultNfcDeviceNotFound();
                }
            }
            else
            {
                // タグを検索中に戻ります。
                data.playerData[playerIndex].state = PlayerState_TagSearch;
            }
        }
    }

    // タグを読み込み中です。
    void UpdatePlayerStateTagRead(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 読み込みが完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.playerData[playerIndex].processFinishEvent) )
        {
            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            nn::Result  result = accessor.GetLastResult();
            bool isModelUpdated = false;

            // 読み込み結果をコピーして保存します。
            bool existResult;
            nn::Result commandResult;
            existResult = accessor.GetResult(&commandResult, PlayerCommand_GetTagInfo);
            if( existResult && commandResult.IsSuccess() )
            {
                data.playerData[playerIndex].tagInfo = g_Work.playerData[playerIndex].tagInfo;
            }
            else
            {
                std::memset(&data.playerData[playerIndex].tagInfo, 0, sizeof(data.playerData[playerIndex].tagInfo));
            }
            existResult = accessor.GetResult(&commandResult, PlayerCommand_GetModelInfo);
            if( existResult && commandResult.IsSuccess() )
            {
                data.playerData[playerIndex].modelInfo = g_Work.playerData[playerIndex].modelInfo;
                isModelUpdated = true;
            }
            else
            {
                std::memset(&data.playerData[playerIndex].modelInfo, 0, sizeof(data.playerData[playerIndex].modelInfo));
            }
            existResult = accessor.GetResult(&commandResult, PlayerCommand_GetCommonInfo);
            if( existResult && commandResult.IsSuccess() )
            {
                data.playerData[playerIndex].commonInfo = g_Work.playerData[playerIndex].commonInfo;
            }
            else
            {
                std::memset(&data.playerData[playerIndex].commonInfo, 0, sizeof(data.playerData[playerIndex].commonInfo));
            }
            existResult = accessor.GetResult(&commandResult, PlayerCommand_GetRegisterInfo);
            if( existResult && commandResult.IsSuccess() )
            {
                data.playerData[playerIndex].registerInfo = g_Work.playerData[playerIndex].registerInfo;
            }
            else
            {
                std::memset(&data.playerData[playerIndex].registerInfo, 0, sizeof(data.playerData[playerIndex].registerInfo));
            }
            existResult = accessor.GetResult(&commandResult, PlayerCommand_GetApplicationArea);
            if( existResult && commandResult.IsSuccess() )
            {
                data.playerData[playerIndex].counter = g_Work.playerData[playerIndex].applicationAreaData.counter;
                data.playerData[playerIndex].counter = ConvertFromTagByteOrder(data.playerData[playerIndex].counter);
            }
            else
            {
                data.playerData[playerIndex].counter = 0;
            }

            // 読み込み処理の結果に応じて状態遷移します。
            if( isModelUpdated && !IsKnownAmiibo(data.playerData[playerIndex].modelInfo.characterId) )
            {
                // Character ID からアプリが利用可能な amiibo ではないと判断されました。
                WaitTagReleaseForNext(playerIndex, data, PlayerState_AmiiboNotSupported);
            }
            else if( result.IsSuccess() )
            {
                // 利用可能な amiibo の情報を取得しました。
                WaitTagReleaseForNext(playerIndex, data, PlayerState_TagEnabled);
            }
            else if( result <= nn::nfp::ResultNeedCreate() )
            {
                // タグにアプリケーション領域が存在しません。作成する必要があります。
                // ただし、作成する前にアプリが対応する amiibo であることを確認します。
                WaitTagReleaseForNext(playerIndex, data, PlayerState_NeedCreate);
            }
            else if( result <= nn::nfp::ResultNeedRestore() )
            {
                // タグ上のデータが壊れていますがバックアップデータが存在するため復旧可能です。
                WaitTagReleaseForNext(playerIndex, data, PlayerState_NeedRestore);
            }
            else if( result <= nn::nfp::ResultNeedFormat() )
            {
                // タグ上のデータが壊れていますがバックアップデータが存在せず初期化が必要です。
                WaitTagReleaseForNext(playerIndex, data, PlayerState_NeedFormat);
            }
            else if( result <= nn::nfp::ResultNeedRegister() )
            {
                // タグに初期登録情報が登録されていません。登録が必要です。
                WaitTagReleaseForNext(playerIndex, data, PlayerState_NeedRegister);
            }
            else if( result <= nn::nfp::ResultAccessIdMisMatch() )
            {
                // 既にタグ上に他のアプリケーションが作成したデータが存在します。
                // 利用するためにはアプリケーション領域を削除しなければなりません。
                WaitTagReleaseForNext(playerIndex, data, PlayerState_NeedDelete);
            }
            else
            {
                WaitTagReleaseForError(playerIndex, data, result);
            }
        }
    }

    // タグを確認中です。
    void UpdatePlayerStateTagCheck(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグ情報を取得したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.playerData[playerIndex].processFinishEvent) )
        {
            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                if( IsEqualId(g_Work.playerData[playerIndex].tagInfo.tagId, data.playerData[playerIndex].tagInfo.tagId) )
                {
                    OnTagChecked(playerIndex, data);
                }
                else
                {
                    WaitTagReleaseForError(playerIndex, data, nn::nfp::ResultNeedRestart());
                }
            }
            else
            {
                WaitTagReleaseForError(playerIndex, data, result);
            }
        }
    }

    // タグを書き込み中です。
    void UpdatePlayerStateTagWrite(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 書き込みが完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.playerData[playerIndex].processFinishEvent) )
        {
            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                WaitTagReleaseForNext(playerIndex, data, PlayerState_Idle);
            }
            else
            {
                WaitTagReleaseForError(playerIndex, data, result);
            }
        }
    }

    // 利用可能なタグの情報を取得できた状態です。
    void UpdatePlayerStateTagEnabled(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンでタグのカウンタをインクリメントします。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::A::Mask) )
        {
            auto writer = [](int playerIndex, ApplicationData& data)
            {
                g_Work.playerData[playerIndex].applicationAreaData.counter = ConvertToTagByteOrder(data.playerData[playerIndex].counter + 1);

                PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
                accessor.Mount(GetDeviceHandle(playerIndex));
                accessor.OpenApplicationArea(GetDeviceHandle(playerIndex), AccessId);
                accessor.SetApplicationArea(
                        GetDeviceHandle(playerIndex),
                        &g_Work.playerData[playerIndex].applicationAreaData,
                        sizeof(g_Work.playerData[playerIndex].applicationAreaData)
                    );
                accessor.Flush(GetDeviceHandle(playerIndex));
                accessor.Unmount(GetDeviceHandle(playerIndex));
                accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
            };
            DoStartDetectionForTagWrite(playerIndex, data, writer);
        }

        // B ボタンでタグの検知を開始の指示を待っている状態に戻ります。
        else if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::B::Mask) )
        {
            data.playerData[playerIndex].state = PlayerState_Idle;
        }
    }

    // アプリケーション領域を作成する必要があります。
    void UpdatePlayerStateNeedCreate(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンでアプリケーション領域を作成します。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::A::Mask) )
        {
            auto writer = [](int playerIndex, ApplicationData& data)
            {
                NN_UNUSED(data);
                g_Work.playerData[playerIndex].applicationAreaData.counter = ConvertToTagByteOrder(InitialCounter);

                PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
                accessor.Mount(GetDeviceHandle(playerIndex));
                accessor.CreateApplicationArea(
                        GetDeviceHandle(playerIndex),
                        AccessId,
                        &g_Work.playerData[playerIndex].applicationAreaData,
                        sizeof(g_Work.playerData[playerIndex].applicationAreaData)
                    );
                accessor.Unmount(GetDeviceHandle(playerIndex));
                accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
            };
            DoStartDetectionForTagWrite(playerIndex, data, writer);
        }

        // B ボタンでタグの検知を開始の指示を待っている状態に戻ります。
        else if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::B::Mask) )
        {
            data.playerData[playerIndex].state = PlayerState_Idle;
        }
    }

    // タグの復旧が必要です。
    void UpdatePlayerStateNeedRestore(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンでバックアップデータからのタグの復旧を開始します。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::A::Mask) )
        {
            auto writer = [](int playerIndex, ApplicationData& data)
            {
                NN_UNUSED(data);
                PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
                accessor.Restore(GetDeviceHandle(playerIndex));
                accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
            };
            DoStartDetectionForTagWrite(playerIndex, data, writer);
        }

        // B ボタンでタグの検知を開始の指示を待っている状態に戻ります。
        else if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::B::Mask) )
        {
            data.playerData[playerIndex].state = PlayerState_Idle;
        }

        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        // X ボタンで amiibo 設定経由でタグを復旧します。
        else if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::X::Mask) )
        {
            std::memset(&g_Work.playerData[playerIndex].amiiboSettingsData.startParam, 0, sizeof(g_Work.playerData[playerIndex].amiiboSettingsData.startParam));
            g_Work.playerData[playerIndex].amiiboSettingsData.startParam.deviceHandle = *GetDeviceHandle(playerIndex);

            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            accessor.StartRestorer(
                    &g_Work.playerData[playerIndex].amiiboSettingsData.deviceHandle,
                    &g_Work.playerData[playerIndex].amiiboSettingsData.startParam,
                    &g_Work.playerData[playerIndex].tagInfo
                );
            accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
            data.playerData[playerIndex].state = PlayerState_AmiiboSettings;
        }
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
    }

    // タグの初期化が必要です。
    void UpdatePlayerStateNeedFormat(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // この状態に陥るとアプリケーションからタグを復旧することはできません。
        // 以前 amiibo を使用したことのある本体で復旧できる可能性があることと、
        // 復旧せずに使用する場合は本体設定の amiibo設定で初期化できることを通知してください。

        // B ボタンでタグの検知を開始の指示を待っている状態に戻ります。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::B::Mask) )
        {
            data.playerData[playerIndex].state = PlayerState_Idle;
        }
    }

    // 初期登録情報を登録する必要があります。
    void UpdatePlayerStateNeedRegister(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // B ボタンでタグの検知を開始の指示を待っている状態に戻ります。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::B::Mask) )
        {
            data.playerData[playerIndex].state = PlayerState_Idle;
        }

        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        // X ボタンで初期登録情報を登録するために amiibo 設定を実行します。
        else if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::X::Mask) )
        {
            std::memset(&g_Work.playerData[playerIndex].amiiboSettingsData.startParam, 0, sizeof(g_Work.playerData[playerIndex].amiiboSettingsData.startParam));
            g_Work.playerData[playerIndex].amiiboSettingsData.startParam.deviceHandle = *GetDeviceHandle(playerIndex);

            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            accessor.StartNicknameAndOwnerSettings(
                    &g_Work.playerData[playerIndex].amiiboSettingsData.deviceHandle,
                    &g_Work.playerData[playerIndex].amiiboSettingsData.isRegistered,
                    &g_Work.playerData[playerIndex].registerInfo,
                    &g_Work.playerData[playerIndex].amiiboSettingsData.startParam,
                    &g_Work.playerData[playerIndex].tagInfo
                );
            accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
            data.playerData[playerIndex].state = PlayerState_AmiiboSettings;
        }
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
    }

    // アプリケーション領域を削除する必要があります。
    void UpdatePlayerStateNeedDelete(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // B ボタンでタグの検知を開始の指示を待っている状態に戻ります。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::B::Mask) )
        {
            data.playerData[playerIndex].state = PlayerState_Idle;
        }

        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        // X ボタンでアプリケーション領域を削除するために amiibo 設定を実行します。
        else if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::X::Mask) )
        {
            std::memset(&g_Work.playerData[playerIndex].amiiboSettingsData.startParam, 0, sizeof(g_Work.playerData[playerIndex].amiiboSettingsData.startParam));
            g_Work.playerData[playerIndex].amiiboSettingsData.startParam.deviceHandle = *GetDeviceHandle(playerIndex);

            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            accessor.StartGameDataEraser(
                    &g_Work.playerData[playerIndex].amiiboSettingsData.deviceHandle,
                    &g_Work.playerData[playerIndex].amiiboSettingsData.startParam,
                    &g_Work.playerData[playerIndex].tagInfo
                );
            accessor.Run(&g_Work.playerData[playerIndex].processFinishEvent);
            data.playerData[playerIndex].state = PlayerState_AmiiboSettings;
        }
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
    }

    // アプリケーションが対応していない amiibo です。
    void UpdatePlayerStateAmiiboNotSupported(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // サンプルプログラムは amiibo データリストに登録されている一部のキャラクターにしか対応しておらず、
        // 非対応のキャラクターを書き込んだ amiibo に対してはアプリケーション専用領域の読み書きができません。
        // 新しい amiibo に対応する場合は、TagUtility.cpp の CharacterIdList と SampleCharacterName に定義を追加してください。

        // B ボタンでタグの検知を開始の指示を待っている状態に戻ります。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::B::Mask) )
        {
            data.playerData[playerIndex].state = PlayerState_Idle;
        }
    }

    // amiibo 設定の起動中です。
    void UpdatePlayerStateAmiiboSettings(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // amiibo 設定が完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.playerData[playerIndex].processFinishEvent) )
        {
            PlayerTagAccessor& accessor = TagAccessor::GetInstance().GetPlayerTagAccessor(playerIndex);
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                DoStartDetectionForTagRead(playerIndex, data);
            }
            else
            {
                data.playerData[playerIndex].state = PlayerState_Error;
                data.playerData[playerIndex].lastResult = result;
            }
        }
    }

    // エラー状態です。
    void UpdatePlayerStateError(
            int playerIndex,
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンで NFC デバイス取得、または、NFC デバイス選択からやり直します。
        if( HasHidControllerAnyButtonsDown(data.playerData[playerIndex].npadId, nn::hid::NpadButton::A::Mask) )
        {
            // NFC デバイス喪失なので、NFC デバイス取得からやり直します。
            if( data.playerData[playerIndex].lastResult <= nn::nfp::ResultNfcDeviceNotFound() )
            {
                data.playerData[playerIndex].state = PlayerState_None;
                if(data.state == State_Idle)
                {
                    data.state = State_Initialized;
                }
            }
            else
            {
                // NFC デバイス取得済なので、アイドル状態に戻ります。
                data.playerData[playerIndex].state = PlayerState_Idle;
            }
            data.playerData[playerIndex].lastResult = nn::ResultSuccess();
        }
    }

    //--------------------------------------------------------------------------
    //  状態処理関数の取得
    //--------------------------------------------------------------------------
    struct UpdateStateEntry
    {
        State           state;
        UpdateStateFunc func;
    };

    const UpdateStateEntry UpdateStateTable[] = {
        {   State_None,                 UpdateStateNone             },
        {   State_Initializing,         UpdateStateInitializing     },
        {   State_Initialized,          UpdateStateInitialized      },
        {   State_DeviceListing,        UpdateStateDeviceListing    },
        {   State_Idle,                 UpdateStateIdle             },
    };
    const int UpdateStateCountMax = sizeof(UpdateStateTable) / sizeof(UpdateStateTable[0]);
    NN_STATIC_ASSERT( UpdateStateCountMax == static_cast<int>(State_CountMax) );

    // 現在の状態を処理する関数を取得します。
    UpdateStateFunc GetUpdateStateFunc(State state)
    {
        const int index = static_cast<int>(state);
        NN_ASSERT( index >= 0 && index < UpdateStateCountMax );
        NN_ASSERT( UpdateStateTable[index].state == state );
        return UpdateStateTable[index].func;
    }

    //--------------------------------------------------------------------------
    //  プレイヤー状態処理関数の取得
    //--------------------------------------------------------------------------
    struct UpdatePlayerStateEntry
    {
        PlayerState           state;
        UpdatePlayerStateFunc func;
    };

    const UpdatePlayerStateEntry UpdatePlayerStateTable[] = {
        {   PlayerState_None,                 UpdatePlayerStateNone             },
        {   PlayerState_DeviceInitialize,     UpdatePlayerStateDeviceInitialize },
        {   PlayerState_AttachEvent,          UpdatePlayerStateAttachEvent      },
        {   PlayerState_Idle,                 UpdatePlayerStateIdle             },
        {   PlayerState_StartDetection,       UpdatePlayerStateStartDetection   },
        {   PlayerState_StopDetection,        UpdatePlayerStateStopDetection    },
        {   PlayerState_TagRelease,           UpdatePlayerStateTagRelease       },
        {   PlayerState_TagSearch,            UpdatePlayerStateTagSearch        },
        {   PlayerState_DeviceCheck,          UpdatePlayerStateDeviceCheck      },
        {   PlayerState_TagRead,              UpdatePlayerStateTagRead          },
        {   PlayerState_TagCheck,             UpdatePlayerStateTagCheck         },
        {   PlayerState_TagWrite,             UpdatePlayerStateTagWrite         },
        {   PlayerState_TagEnabled,           UpdatePlayerStateTagEnabled       },
        {   PlayerState_NeedCreate,           UpdatePlayerStateNeedCreate       },
        {   PlayerState_NeedRestore,          UpdatePlayerStateNeedRestore      },
        {   PlayerState_NeedFormat,           UpdatePlayerStateNeedFormat       },
        {   PlayerState_NeedRegister,         UpdatePlayerStateNeedRegister     },
        {   PlayerState_NeedDelete,           UpdatePlayerStateNeedDelete       },
        {   PlayerState_AmiiboNotSupported,   UpdatePlayerStateAmiiboNotSupported },
        {   PlayerState_AmiiboSettings,       UpdatePlayerStateAmiiboSettings   },
        {   PlayerState_Error,                UpdatePlayerStateError            },
    };
    const int UpdatePlayerStateCountMax = sizeof(UpdatePlayerStateTable) / sizeof(UpdatePlayerStateTable[0]);
    NN_STATIC_ASSERT( UpdatePlayerStateCountMax == static_cast<int>(PlayerState_CountMax) );

    // 現在の状態を処理する関数を取得します。
    UpdatePlayerStateFunc GetUpdatePlayerStateFunc(PlayerState state)
    {
        const int index = static_cast<int>(state);
        NN_ASSERT( index >= 0 && index < UpdatePlayerStateCountMax );
        NN_ASSERT( UpdatePlayerStateTable[index].state == state );
        return UpdatePlayerStateTable[index].func;
    }
}}} // end of namespace nns::nfp::unnamed

namespace nns { namespace nfp {

    void UpdateState(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        data.frame += 1;
        State lastState = data.state;

        (*GetUpdateStateFunc(data.state))(data);
        if( lastState != data.state )
        {
            data.transitFrame = data.frame;
        }

        for(int i = 0; i < PlayerCountMax; i++)
        {
            PlayerData& playerData = data.playerData[i];
            PlayerState lastPlayerState = playerData.state;

            (*GetUpdatePlayerStateFunc(playerData.state))(i, data);
            if( lastPlayerState != playerData.state )
            {
                playerData.transitFrame = data.frame;
            }
        }
    }

}} // end of namespace nns::nfp
