﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Assert.h>
#include "Config.h"
#include "TagAccessor.h"

namespace nns { namespace nfp { namespace {

    NN_ALIGNAS(16) char g_WorkBuffer[ sizeof(TagAccessor) ];

    typedef nn::Result (*ProcessFunc)( const void* pParam );

    struct ArgDeviceHandle
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    // NFP ライブラリを初期化します。
    nn::Result ProcessInitialize(const void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);
        nn::nfp::Initialize();
        return nn::ResultSuccess();
    }

    // NFP ライブラリを終了します。
    nn::Result ProcessFinalize(const void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);
        nn::nfp::Finalize();
        return nn::ResultSuccess();
    }

    // NFC デバイスのハンドルのリストを取得します。
    struct ArgListDevices
    {
        DeviceInfo*                     pOutBuffer;
        int*                            pOutCount;
        int                             bufferCount;
    };

    nn::Result ProcessListDevices(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgListDevices*>(pParam);
        nn::nfp::DeviceHandle deviceHandle[DeviceCountMax];
        auto result = nn::nfp::ListDevices(deviceHandle, pArg->pOutCount, pArg->bufferCount);
        if(result.IsFailure())
        {
            if(nn::nfp::ResultNfcDeviceNotFound::Includes(result))
            {
                *pArg->pOutCount = 0;
            }
            else
            {
                return result;
            }
        }

        for(int i = 0; i < *pArg->pOutCount; i++)
        {
            pArg->pOutBuffer[i].deviceHandle = deviceHandle[i];
            result = nn::nfp::GetNpadId(&pArg->pOutBuffer[i].npadId, pArg->pOutBuffer[i].deviceHandle);
            if(result.IsFailure())
            {
                return result;
            }
        }

        return nn::ResultSuccess();
    }

    // デバイスの利用可/不可の変化を通知するイベントを設定します。
    struct ArgAttachAvailabilityChangeEvent
    {
        nn::os::SystemEventType*        pAvailabilityChangeEvent;
    };

    nn::Result ProcessAttachAvailabilityChangeEvent(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgAttachAvailabilityChangeEvent*>(pParam);

        nn::nfp::AttachAvailabilityChangeEvent(pArg->pAvailabilityChangeEvent);
        return nn::ResultSuccess();
    }

    // コマンド識別子に対応した処理関数の定義
    struct ProcessEntry
    {
        Command         command;
        ProcessFunc     func;
    };

    const ProcessEntry ProcessTable[] = {
        {   Command_None,                           nullptr                         },
        {   Command_Initialize,                     ProcessInitialize               },
        {   Command_Finalize,                       ProcessFinalize                 },
        {   Command_ListDevices,                    ProcessListDevices              },
        {   Command_AttachAvailabilityChangeEvent,  ProcessAttachAvailabilityChangeEvent },
    };
    const int ProcessCountMax = sizeof(ProcessTable) / sizeof(ProcessTable[0]);
    NN_STATIC_ASSERT( ProcessCountMax == static_cast<int>(Command_CountMax) );

    ProcessFunc GetProcessFunc(Command command)
    {
        const int index = static_cast<int>(command);
        NN_ASSERT( index >= 1 && index < ProcessCountMax );
        NN_ASSERT( ProcessTable[index].command == command );
        return ProcessTable[index].func;
    }

}}} // end of namespace nns::nfp::unnamed

namespace nns { namespace nfp {

    TagAccessor* TagAccessor::g_pInstance;

    TagAccessor& TagAccessor::CreateInstance() NN_NOEXCEPT
    {
        NN_ASSERT( g_pInstance == nullptr );

        g_pInstance = new(g_WorkBuffer) TagAccessor();
        return *g_pInstance;
    }

    TagAccessor& TagAccessor::GetInstance() NN_NOEXCEPT
    {
        NN_ASSERT( g_pInstance != nullptr );

        return *g_pInstance;
    }

    void TagAccessor::DestroyInstance() NN_NOEXCEPT
    {
        if( g_pInstance != nullptr )
        {
            g_pInstance->~TagAccessor();
            g_pInstance = nullptr;
        }
    }

    bool TagAccessor::HasInstance() NN_NOEXCEPT
    {
        return g_pInstance != nullptr;
    }

    PlayerTagAccessor& TagAccessor::GetPlayerTagAccessor(int playerIndex) NN_NOEXCEPT
    {
        return m_PlayerTagAccessor[playerIndex];
    }

    int TagAccessor::Initialize() NN_NOEXCEPT
    {
        return Enqueue(Command_Initialize, nullptr, 0);
    }

    int TagAccessor::Finalize() NN_NOEXCEPT
    {
        return Enqueue(Command_Finalize, nullptr, 0);
    }

    int TagAccessor::ListDevices(
            DeviceInfo* pOutBuffer,
            int* pOutCount,
            int bufferCount
        ) NN_NOEXCEPT
    {
        ArgListDevices arg;
        arg.pOutBuffer = pOutBuffer;
        arg.pOutCount = pOutCount;
        arg.bufferCount = bufferCount;

        return Enqueue(Command_ListDevices, arg);
    }

    int TagAccessor::AttachAvailabilityChangeEvent(
            nn::os::SystemEventType* pAvailabilityChangeEvent
        ) NN_NOEXCEPT
    {
        ArgAttachAvailabilityChangeEvent arg;
        arg.pAvailabilityChangeEvent = pAvailabilityChangeEvent;

        return Enqueue(Command_AttachAvailabilityChangeEvent, arg);
    }

    bool TagAccessor::IsBusy() NN_NOEXCEPT
    {
        return TagAccessorBase::IsBusy();
    }

    nn::Result TagAccessor::Execute(uint32_t command, const ParameterBuffer& param) NN_NOEXCEPT
    {
        return (*GetProcessFunc(static_cast<Command>(command)))(&param);
    }

}} // end of namespace nns::nfp
