﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NetworkServiceDiscoveryMain.cpp,PageSampleNetworkServiceDiscovery}
 *
 * @brief
 *  ネットワークサービスディスカバリの設定データダウンロードのサンプルプログラム(メイン処理)
 */

/**
 * @page PageSampleNetworkServiceDiscovery ネットワークサービスディスカバリの設定データダウンロード
 * @tableofcontents
 *
 * @brief
 *  ネットワークサービスディスカバリの設定データダウンロードのサンプルプログラムの解説です。
 *
 * @section PageSampleNetworkServiceDiscovery_SectionBrief 概要
 *  ネットワーク接続後、HTTP 通信を行ってネットワークサービスディスカバリの設定データをダウンロードします。
 *
 * @section PageSampleNetworkServiceDiscovery_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/NetworkServiceDiscovery Samples/Sources/Applications/NetworkServiceDiscovery @endlink 以下にあります。
 *
 *  他所への組み込みを行いやすくするため、ダウンロードの実行部分においては以下のファイルに切り出しています。
 *  - NetworkServiceDiscoveryDownload.h
 *  - NetworkServiceDiscoveryDownload.cpp
 *
 * @section PageSampleNetworkServiceDiscovery_SectionNecessaryEnvironment 必要な環境
 *  事前に設定マネージャを使用してネットワーク接続設定をインポートする必要があります。
 *  詳細は @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} をご覧ください。
 *
 * @section PageSampleNetworkServiceDiscovery_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleNetworkServiceDiscovery_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleNetworkServiceDiscovery_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleNetworkServiceDiscovery_SectionDetail 解説
 *
 * @subsection PageSampleNetworkServiceDiscovery_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  NetworkServiceDiscoveryMain.cpp
 *  @includelineno NetworkServiceDiscoveryMain.cpp
 *
 *  NetworkServiceDiscoveryDownload.h
 *  @includelineno NetworkServiceDiscoveryDownload.h
 *
 *  NetworkServiceDiscoveryDownload.cpp
 *  @includelineno NetworkServiceDiscoveryDownload.cpp
 *
 * @subsection PageSampleNetworkServiceDiscovery_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 * - NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * - socket ライブラリを初期化します。
 * - libcurl の easy interface を使用して HTTP リクエストを送信します。
 *
 */

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/socket.h>

#include <curl/curl.h>

#include <nn/nsd/nsd_Types.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include "NetworkServiceDiscoveryDownload.h"

namespace
{
    NN_ALIGNAS(4096) uint8_t g_MallocBuffer[1 * 1024 * 1024];

    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    // ネットワークサービスディスカバリの設定ダウンロード用バッファです。
    // 現時点では NetworkServiceDiscoverySample::ResponseBufferSizeMin 以上のサイズで用意してください。
    uint8_t g_DownloadBuffer[networkServiceDiscoverySample::ResponseBufferSizeMin];
}

// ネットワークサービスディスカバリの設定データのダウンロードを実行します
void ExecuteDownload() NN_NOEXCEPT
{
    // curl セッションインスタンスの生成
    CURL* curl = curl_easy_init();
    if(!curl)
    {
        NN_LOG("curl_easy_init failed.\n");
        return;
    }

    // ネットワークサービスディスカバリの設定ダウンロードにはパスコードが必要です。
    // パスコードは、ネットワークサービスディスカバリの設定を行う WEB ページ等で確認できるようになる予定です。
    nn::nsd::PassCode passCode = {{"uNijgVxB"}};// 実際にはユーザが入力したパスコードを格納してください
    size_t downloadSize = 0;

    CURLcode result = networkServiceDiscoverySample::DownloadSettingData(
        &downloadSize,
        g_DownloadBuffer, sizeof(g_DownloadBuffer),
        curl,
        passCode);

    curl_easy_cleanup(curl);

    if( result != CURLE_OK )
    {
        NN_LOG("NetworkServiceDiscoverySample::DownloadSettingData failed. curlcode:%d\n", result);
    }
    else
    {
        NN_LOG("NetworkServiceDiscoverySample::DownloadSettingData returned  CURLE_OK. downloadSize:%d\n", downloadSize);

        //
        // ダウンロード成功後、nn::nsd::ImportSettings() を利用して設定データをデバイスに保存してください。
        //
        static uint8_t importWorkBuffer[nn::nsd::WorkBufferSizeForImportSettings];
        auto importResult = nn::nsd::ImportSettings(
            g_DownloadBuffer, downloadSize,
            importWorkBuffer, sizeof(importWorkBuffer),
            nn::nsd::ImportMode_ApplicationSettings);

        if(importResult.IsSuccess())
        {
            NN_LOG("ImportSettings is succeeded.\n");
        }
        else
        {
            NN_LOG("ImportSettings failed. (%08x, %03d-%04d)\n",
                importResult.GetInnerValueForDebug(),
                importResult.GetModule(), importResult.GetDescription());
        }
    }
}

extern "C" void nninitStartup()
{
    // 本サンプルはアプレット向け desc を利用しており、アプレット向けのリソース制限が適用されます。
    // ここでは、デフォルトの nninitStartup() のデフォルトメモリアロケータのサイズが
    // アプレットで利用できるサイズ上限を超えているため、
    // 自前で nninitStartup() を用意してデフォルトメモリアロケータのサイズを調整しています。

    // 本サンプルでは libcurl がデフォルトのメモリアロケータを利用するよう初期化しているので、
    // 適当に十分なサイズでデフォルトメモリアロケータを設定しています。
    // ここでのメモリアロケータ設定はグローバルな設定となるので、アプレットへの組み込み時には注意してください。
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    // NIFM の初期化(詳しくは nn::nifm のリファレンスを参照してください)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    NN_LOG("Waiting for network interface availability...\n");
    nn::nifm::SubmitNetworkRequestAndWait();

    if( !nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("Network is not available.\n");
        return;
    }

    // SOCKET の初期化(詳しくは nn::socket のリファレンスを参照してください)
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::socket::Initialize(g_SocketConfigWithMemory) );

    // curl の初期化(詳しくは libcurl のリファレンスを参照してください)
    curl_global_init(CURL_GLOBAL_DEFAULT);

    // ネットワークサービスディスカバリの設定データのダウンロード実行
    ExecuteDownload();

    curl_global_cleanup();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Finalize());
}

