﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nns/mm.h>
/**
 * @brief
 * MovieAudioInputHandler class for handling audio decoder input data.
 *
 */
void MovieAudioInputHandlerThreadFunction(void *arg);
class MovieAudioInputHandler
{
public:
   /**
    * @brief
    * MovieAudioInputHandler constructor.
    *
    * @param[in] extractor                      Movie extractor for audio track.
    * @param[in] audioTrackIndex                Audio track index.
    * @param[in] audioDecoder                   Movie decoder for audio track.
    *
    */
    MovieAudioInputHandler(movie::Extractor* extractor, int32_t audioTrackIndex, movie::Decoder* audioDecoder);

   /**
    * @brief
    * Initialize MovieAudioInputHandler.
    *
    * @param[in]  coreMask                      Usable CPU cores in core mask.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * Usable cores in core mask. If Nth bit is set to 1, the thread can run on CPU core number N. The core mask need
    * to be set before creation of thread. MovieAudioInputHandler thread is created. A message queue is created and
    * the thread will wait on this message queue.
    */
    movie::Status Initialize(uint64_t coreMask);

   /**
    * @brief
    * ReadInputDataFromExtractorSendTodecoder. API to read data from extractor and send the same to decoder.
    *
    * @param[in] index                         	Audio decoder input buffer index.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_EndOfStream
    * @retval ::Status_UnknownError
    *
    * @details
    * This API can be used to acquire an input buffer from decoder and read input data from extractor. If data is
    * available, input buffer is filled with data from extractor and send to decoder. Audio time stamp for the input
    * audio frame is retrieved from extractor and passed to decoder.
    *
    */
    movie::Status ReadInputDataFromExtractorSendTodecoder(int32_t index);

   /**
    * @brief
    * CheckForInputBuffersAndReadAudioInputData. API to check for decoder input buffers and fill with audio data to
    * decode.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_EndOfStream
    * @retval ::Status_UnknownError
    *
    * @details
    * This API can be used to check for available input buffer. If an input buffer is available and extractor has data
    * ReadInputDataFromExtractorSendTodecoder() API is called.
    *
    */
    movie::Status CheckForInputBuffersAndReadAudioInputData();

    /**
    * @brief
    * AddBufferToAudioIndexList. API to add input buffer index to @class MovieAudioInputHandler input list
    *
    * @param[in]  bufferIndex                         Audio decoder input buffer index
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to add decoder input buffer index to @class MovieAudioInputHandler input list.
    * @class MovieDecoderEventHandler will acquire buffer index from decoder, when it receives decoder event. Buffer
    * index is passed to @class MovieAudioInputHandler for further processing.
    *
    */
    movie::Status AddBufferToAudioIndexList(int32_t bufferIndex);

   /**
    * @brief
    * GetAudioIndexListSize. API to get current size of input buffer list
    *
    * @param[out] indexListSize                	Audio decoder input buffer list size.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to get current size of @class MovieAudioInputHandler input buffer list. This information
    * can be used whether to proceed with reading data from extractor. If there are no input buffer indices the client
    * need to retry at later time.
    *
    */
    movie::Status GetAudioIndexListSize(int32_t* indexListSize);

   /**
    * @brief
    * RemoveBufferFromVideoIndexList. API to get a input buffer index.
    *
    * @param[out]  bufferIndex                         Video decoder input buffer index
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to remove an input buffer index from the list. An input buffer can be requested from decoder
    * for the given buffer index. Buffer indices are acquired from movie decoder by @class MovieDecoderEventHandler. If
    * no indices are available, -1 value will be set to bufferIndex.
    *
    */
    movie::Status RemoveBufferFromAudioIndexList(int32_t* bufferIndex);

   /**
    * @brief
    * AudioInputDataAvailableEvent. API to indicate input buffer availability.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to create an input buffer available message. The message is sent to @class MovieAudioInputHandler
    * message queue using SignalAudioInputBufferAvailable() API.
    *
    */
    movie::Status AudioInputDataAvailableEvent();

   /**
    * @brief
    * SignalAudioInputBufferAvailable. API to send input buffer available message.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to send input buffer available message to @class MovieAudioInputHandler message queue.
    *
    */
    movie::Status SignalAudioInputBufferAvailable();

   /**
    * @brief
    * Thread function for @class MovieVideoInputHandler thread.
    *
    * @param[in] arg                            Arguments for thread.
    *
    * @return None
    *
    * @details
    * The thread loop function. The thread will wait on a message queue.
    *
    */
    friend void MovieAudioInputHandlerThreadFunction(void *arg);

   /**
    * @brief
    * SignalFlush. API to send flush message to @class MovieAudioInputHandler
    *
    * @return none
    *
    * @details
    * This API is used to send a flush message to @class MovieAudioInputHandler message queue. The message handler will
    * call Flush() API.
    *
    */
    void SignalFlush();

   /**
    * @brief
    * Flush. API to flush input buffer.
    *
    * @return none
    *
    * @details
    * This API is used to flush @class MovieAudioInputHandler input buffer index list. API will simply clear the list.
    *
    */
    void Flush();

   /**
    * @brief
    * MovieAudioInputHandler destructor.
    *
    * @return None
    *
    */
    ~MovieAudioInputHandler();
private:
    movie::Extractor* m_Extractor;
    movie::Decoder* m_AudioDecoder;
    int32_t m_AudioTrackIndex;
    size_t m_ThreadStackSize;
    void* m_ThreadStack;
    nn::os::ThreadType m_ThreadType;

    std::vector<int> m_AudioBufferIndexList;
    nn::os::MutexType m_AudioListMutex;

    size_t m_MessageQueueBufferSize;
    uintptr_t *m_MessageQueueBuffer;
    nn::os::MessageQueueType m_MessageQueue;
    bool m_FlushInput;
    enum MoviePlayerMessage
    {
        MoviePlayerMessage_AudioInputAvailable  = 1,
        MoviePlayerMessage_AudioInputThreadExit = 2,
        MoviePlayerMessage_AudioInputFlush      = 3,
    };
};
