﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
/**
 * @file mm_MovieVideoOutputHandler.h
 * @brief Handle video decoded data from movie::decoder
 */

#include "NvnVideoRenderer.h"
#include <nns/mm/mm_MoviePlayerUtils.h>
#include <map>
/**
 * @brief
 * MovieVideoOutputHandler class for processing video data from movie::decoder
 *
 */
void MovieVideoOutputHandlerThreadFunctionAvSync(void *arg);

void MovieVideoOutputHandlerThreadFunctionRender(void *arg);

class MovieVideoOutputHandler
{
public:
   /**
    * @brief MovieAudioOutputHandler constructor.
    *
    * @param[in] numberOfVideos                 Number of simultaneous videos.
    * @param[in] decoderMode                    Video decoder mode(CPU or Native texture).
    *
    * @details
    * An instance of MovieVideoOutputHandler is created. Also, an instance of NvnVideoRenderer is created and
    * initialized.
    *
    */
    MovieVideoOutputHandler(int numberOfVideos, movie::DecoderMode decoderMode);

   /**
    * @brief
    * MovieVideoOutputHandler destructor.
    *
    */
    ~MovieVideoOutputHandler();
    struct VideoData
    {
        int32_t index;
        int64_t presentationTimeUs;
        uint32_t flags;
    };

    struct RenderContext
    {
        RenderContext(movie::Decoder* decoder, int width, int height, movie::DecoderOutputFormat outputFormat, movie::DecoderMode decoderMode, MediaClock *mediaClock, MovieVideoOutputHandler *movieVideoOutputHandler, nn::os::EventType *renderComplete)
            : m_Decoder(decoder),
            m_OutputFormat(outputFormat),
            m_DecoderMode(decoderMode),
            m_MediaClock(mediaClock),
            m_MovieVideoOutputHandler(movieVideoOutputHandler),
            m_Width(width),
            m_Height(height),
            m_RenderComplete(renderComplete)
        {
            m_ThreadStarted = false;
            m_ThreadDone = false;
            m_ThreadCreated = false;
            m_PresentationTimeUs = 0;
            m_ThreadstackSize = ( 1024 * 256 );
            m_Threadstack = nullptr;
            m_AllFramesRendered = false;
            m_VideoNeedDelayTimeUs = 0;
            m_VideoStartTime = 0;
            m_DroppedVideoFrames = 0;
            m_UpdateVideoStartTime = false;
            m_LastPresentationTimeUs = -1;
            m_VideoOutputBuffersReceived = 0;
            m_IsPaused = false;
            m_PlaybackPositionUs = 0;
            m_LastSleepTimeUs = 0;
            m_LastAvSyncTimeUs = 0;
        }
        movie::Decoder* m_Decoder;
        movie::DecoderOutputFormat m_OutputFormat;
        movie::DecoderMode m_DecoderMode;
        MediaClock *m_MediaClock;
        std::vector<VideoData> m_VideoBuffers;
        nn::os::MutexType m_VideoMutex;
        MovieVideoOutputHandler *m_MovieVideoOutputHandler;
        int64_t m_VideoNeedDelayTimeUs;
        int64_t m_VideoStartTime;
        int64_t m_LastPresentationTimeUs;
        int64_t m_PresentationTimeUs;
        int m_DroppedVideoFrames;
        bool m_AllFramesRendered;
        bool m_UpdateVideoStartTime;
        bool m_ThreadStarted;
        bool m_ThreadDone;
        bool m_ThreadCreated;
        size_t m_ThreadstackSize;
        void *m_Threadstack;
        nn::os::ThreadType m_ThreadType;
        int m_Width;
        int m_Height;
        int64_t m_VideoOutputBuffersReceived;
        bool m_IsPaused;
        int64_t m_PlaybackPositionUs;
        int64_t m_LastSleepTimeUs;
        int64_t m_LastAvSyncTimeUs;
        nn::os::EventType *m_RenderComplete;
    };

  /**
    * @brief Initialize MovieVideoOutputHandler.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError.
    *
    */
    movie::Status Initialize();

   /**
    * @brief
    * Register video decoder with MovieVideoOutputHandler.
    *
    * @param[in] decoder                        movie::Decoder (video).
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    * @param[in] outputFormat                   Video decoder output format.
    * @param[in] mediaClock                     Clock component to do video sync.
    * @param[in] renderComplete                 Client event to be signaled when rendering is complete.
    *
    * @details
    * This API can be used for registering video decoder handle with MovieVideoOutputHandler. Video decoder handle is
    * used for acquiring decoded video frame from decoder. Media clock is used to maintain video frame rate
    * during playback. The client event will be signaled when rendering is completed.
    *
    */
    movie::Status RegisterDecoder(movie::Decoder* decoder, int width, int height, movie::DecoderOutputFormat outputFormat, MediaClock *mediaClock, nn::os::EventType *renderComplete);

   /**
    * @brief
    * Register Update decoder properties(video width and height)
    *
    * @param[in] decoder                        movie::Decoder (video).
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    *
    * @details
    * This API can be used for updating the video width and height. After video decoder is configured, it will report
    * any change in video properties. Video height and width may change due to hardware alignment restrictions.
    *
    */
    movie::Status UpdateDecoderProperties(movie::Decoder* decoder, int width, int height);

    /**
     * @brief
     * Finalize MovieVideoOutputHandler.
     *
     * @return ::movie::Status
     * @retval ::Status_Success
     *
     * @details
     * Rendering thread is stopped and destroyed. All decoder specific video sync threads are stopped and destroyed.
     */
    movie::Status Finalize();

   /**
    * @brief
    * Open MovieVideoOutputHandler.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    * @retval ::Status_UnknownError             Failed to create thread.
    *
    * @details
    * Rendering thread is created. Separate video sync thread is created for all registered video decoders.
    *
    */
    movie::Status Open();

   /**
    * @brief
    * Start processing data from registered video decoders.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Failed to start playback.
    *
    * @details
    * Rendering thread is started. Texture and video buffers are created for each instance of decoder. Video sync
    * threads are started.
    *
    */
    movie::Status Start();

   /**
    * @brief
    * Render video on screen.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    *
    * @details
    * Renderer thread uses @class NvnVideoRenderer to draw video frames at 60 frames per second. The renderer
    * will update the video frames from registered decoders. When no frames are available, previous frames will be used.
    *
    */
    movie::Status Render(int64_t *timeToRenderUs);

   /**
    * @brief
    * DoneRendering.
    *
    * @return true                              All video frames are rendered.
    * @return false                             All video frames are not rendered, playback is not complete.
    *
    * @details
    * An API used to return rendering completion state.
    *
    */
    bool DoneRendering();

   /**
    * @brief
    * Stop processing of video data from video decoders.
    *
    * @return none
    *
    * @details
    * Application can use this API to stop processing of data from video decoders.
    *
    */
    void Stop();

   /**
    * @brief
    * Stop processing video data from a given movie::Decoder
    *
    * @param[in] decoder                        movie::Decoder (video).
    * @return none
    *
    * @details
    * Application can use this API to stop processing video data from a given video decoder.
    *
    */
    void Stop(movie::Decoder* decoder);

   /**
    * @brief
    * Pause processing video data from all video decoders.
    *
    * @return none
    *
    * @details
    * Application can use this API to pause playback. All video sync threads will pause
    * and no frames are submitted for rendering.
    *
    */
    void Pause();

   /**
    * @brief
    * Resume processing video data from all video decoders.
    *
    * @return none
    *
    * @details
    * Application can use this API to resume playback. All video sync threads will resume
    * submitting video frames for rendering.
    *
    */
    void Resume();

   /**
    * @brief
    * Flush video decoder output buffers.
    *
    * @return none
    *
    * @details
    * Application can use this API to flush video decoder output buffers.
    *
    */
    void Flush();

   /**
    * @brief
    * Close MovieVideoOutputHandler and release resources.
    *
    * @return none
    *
    */
    void Close();

   /**
    * @brief
    * Thread function for video sync thread.
    *
    * @param[in] arg                            Arguments for thread.
    * @return None
    *
    * @details
    * The thread loop function for preparing video frames for rendering. For each registered video decoder,
    * one video sync thread is created.
    *
    */
    friend void MovieVideoOutputHandlerThreadFunctionAvSync(void *arg);

   /**
    * @brief
    * Thread function for rendering thread.
    *
    * @param[in] arg                            Arguments for thread.
    *
    * @return None
    *
    * @details
    * The thread loop function for rendering video.
    *
    */
    friend void MovieVideoOutputHandlerThreadFunctionRender(void *arg);

   /**
    * @brief
    * Video output buffer available call back.
    *
    * @param[in] decoder                        movie::Decoder (video).
    * @param[in] index                          Output buffer index from decoder.
    * @param[in] presentationTimeUs             Presentation time for this video buffer.
    * @param[in] flags                          movie:: buffer flags.
    *
    * @return None
    *
    * @details
    * When output buffers are available, callbacks are received by @class MovieVideoOutputHandler. Output buffer index
    * along with presentation time and buffer flags are added to video buffer list.
    *
    */
    void OnOutputAvailable(movie::Decoder* decoder, int32_t index, int64_t presentationTimeUs, uint32_t flags);

   /**
    * @brief
    * DoneProcessing
    *
    * @param[in] renderContext                  Render context of a video decoder.
    *
    * @return boolean
    *
    * @retval true                              Terminate thread.
    * @retval false                             Do not terminate thread.
    *
    * @details
    * Helper method used to determine exit condition from thread loop of a given video sync thread. The sync
    * thread information is obtained using render context.
    *
    */
    bool DoneProcessing(RenderContext *renderContext);

   /**
    * @brief
    * AllFramesRendered
    *
    * @return boolean
    *
    * @retval true                              All video frames are rendered.
    * @retval false                             All frames not rendered, more frames available for rendering.
    *
    * @details
    * Helper method to check render completion state.
    *
    */
    bool AllFramesRendered();

   /**
    * @brief
    * AVSync - Perform video sync with presentation time
    *
    * @param[in] renderContext                  Render context of a video decoder.
    * @param[in] mediaClock                     Clock component to do video sync.
    * @param[in] presentationTimeUs             Presentation time for this video frame.
    *
    * @return boolean
    * @retval true                              Video frame need to be dropped (late).
    * @retval false                             Video frame can be rendered now.
    *
    * @details
    * Helper method to perform video sync. If video is late it will be dropped. Video sync is done to render
    * frames at desired frame rate.
    *
    */
    bool AVSync(RenderContext *renderContext, MediaClock *mediaClock, int64_t presentationTimeUs);

   /**
    * @brief
    * Prepare video frame for rendering
    *
    * @param[in] renderContext                  Render context of a video decoder.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    *
    * @details
    * Get an empty buffer(or texture) from renderer. Send the buffer(or texture) to the decoder. Submit the
    * decoded buffer(or texture) to the renderer.
    */
    movie::Status PrepareFrameForRendering(RenderContext *renderContext);

   /**
    * @brief
    * GetRealTimeUs
    *
    * @param[in] renderContext                  Render context.
    * @param[in] mediaClock                     Media clock for current rendering context.
    * @param[in] mediaTimeUs                    Media time in microseconds.
    * @param[in] nowUs                          Current time in microseconds.
    *
    * @return timeUs                            Real time for the given media time, in microseconds.
    *
    * @details
    * Helper method to find real time for the given media time.
    *
    */
    int64_t GetRealTimeUs(RenderContext *renderContext, MediaClock *mediaClock, int64_t mediaTimeUs, int64_t nowUs);

   /**
    * @brief
    * GetVideoPostDelayUs
    *
    * @param[in] renderContext                  Render context.
    *
    * @return sleepTimeMs                       Sleep time in milliseconds.
    *
    * @details
    * Helper method to get video sync thread sleep time.
    *
    */
    int64_t GetVideoPostDelayUs(RenderContext *renderContext);

   /**
    * @brief
    * IsPaused
    *
    * @param[in] renderContext                  Render context.
    *
    * @return boolean
    * @retval true                              Video is in paused state.
    * @retval false                             Video is in playing state.
    *
    * @details
    * Helper method used to determine video renderer play/pause state.
    *
    */
    bool IsPaused(RenderContext *renderContext);

   /**
    * @brief
    * GetPlaybackPosition
    *
    * @param[in] decoder                        movie::Decoder (video).
    * @param[out] playbackPositionUs            Playback position in microseconds.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    *
    * @details
    * Returns current video playback position in microseconds.
    *
    */
    movie::Status GetPlaybackPosition(movie::Decoder* decoder, int64_t *playbackPositionUs);

   /**
    * @brief
    * CanStartRendering
    *
    * @return boolean
    * @retval true                              Video rendering can be started.
    * @retval false                             Video rendering cannot be started(not enough video data).
    *
    * @details
    * Helper method used to determine starting of video renderer.
    *
    */
    bool CanStartRendering();

public:
    nn::vi::NativeWindowHandle m_NativeWindow;
    NvnVideoRenderer* m_NvnVideoRenderer;
    nn::os::MutexType m_VideoMutexRender;
    std::map <movie::Decoder*, RenderContext* > m_RenderContextMap;
private:
    void CheckEndOfStreamAndSignalPlayer(RenderContext *renderContext, uint32_t flags);
    NvnVideoRenderer::NvnOutputFormat m_nvnOutputFormat;
    int m_NumberOfVideos;
    bool m_ThreadStarted;
    bool m_ThreadDone;
    bool m_ThreadCreated;
    size_t m_ThreadstackSize;
    void *m_Threadstack;
    nn::os::ThreadType m_ThreadType;
    movie::DecoderMode m_DecoderMode;
};

