﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <vector>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Result.h>
#include <map>
#include <movie/Common.h>
#include <movie/Status.h>
#include <movie/Utils.h>
#include <movie/Extractor.h>
#include <movie/Decoder.h>

/**
* @brief
* MovieVideoInputHandler class for handling video decoder input data.
*
*/
void MovieVideoInputHandlerThreadFunction(void *arg);
class MovieVideoInputHandler
{
public:

   /**
    * @brief
    * MovieVideoInputHandler constructor.
    *
    */
    MovieVideoInputHandler();

   /**
    * @brief
    * Initialize MovieVideoInputHandler instance.
    *
    * @param[in]  coreMask                      Usable CPU cores in core mask.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * Usable cores in core mask. If Nth bit is set to 1, the thread can run on CPU core number N. The core mask needs
    * to be set before creation of thread. @class MovieVideoInputHandler thread is created. A message queue is created
    * and the thread will wait on this message queue.
    *
    */
    movie::Status Initialize(uint64_t coreMask);

   /**
    * @brief
    * ReadInputDataFromExtractorSendTodecoder. API to read data from extractor and send the same to decoder.
    *
    * @param[in] extractor                      movie::Extractor.
    * @param[in] decoder                        movie::Decoder (video).
    * @param[in] index                         	Video decoder input buffer index.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_EndOfStream
    * @retval ::Status_UnknownError
    *
    * @details
    * This API can be used to acquire an input buffer from decoder and read input data from extractor. If data is
    * available, input buffer is filled with data from extractor and sent to decoder. Video timestamp for the input
    * video frame is retrieved from extractor and passed to decoder.
    *
    */
    movie::Status ReadInputDataFromExtractorSendTodecoder(movie::Extractor* extractor, movie::Decoder* decoder, int index);

   /**
    * @brief
    * CheckForInputBuffersAndReadVideoInputData. API to check for decoder input buffers and fill with video data to
    * decode.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_EndOfStream
    * @retval ::Status_UnknownError
    *
    * @details
    * This API can be used to check for available input buffer. If an input buffer is available and extractor has data
    * ReadInputDataFromExtractorSendTodecoder() API is called.
    *
    */
    movie::Status CheckForInputBuffersAndReadVideoInputData();

   /**
    * @brief
    * AddBufferToVideoIndexList. API to add input buffer index to @class MovieVideoInputHandler input list.
    *
    * @param[in] decoder                        movie::Decoder (video).
    * @param[in] bufferIndex                    Video decoder input buffer index.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to add decoder input buffer index to  @class MovieVideoInputHandler input list.
    * @class MovieDecoderEventHandler will acquire buffer index from decoder, when it receives decoder event. Buffer
    * index is passed to @class MovieVideoInputHandler for further processing.
    */
    movie::Status AddBufferToVideoIndexList(movie::Decoder *decoder, int32_t bufferIndex);

   /**
    * @brief
    * RemoveBufferFromVideoIndexList. API to get an input buffer index.
    *
    * @param[in] decoder                        movie::Decoder (video).
    * @param[out] bufferIndex                   Video decoder input buffer index.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to remove an input buffer index from the list. Input buffer can be requested from decoder
    * for a given buffer index. Buffer indices are acquired from movie decoder by @class MovieDecoderEventHandler. If
    * no indices are available, -1 value will be set to bufferIndex.
    *
    */
    movie::Status RemoveBufferFromVideoIndexList(movie::Decoder *decoder, int32_t* bufferIndex);

   /**
    * @brief
    * VideoInputBufferAvailableEvent. API to indicate input buffer availability
    *
    * @param[in] extractor                      movie::Extractor.
    * @param[in] decoder                        movie::Decoder (video).
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to create an input buffer available message. The message is sent to @class MovieVideoInputHandler
    * message queue using SignalVideoInputBufferAvailable() API.
    *
    */
    movie::Status VideoInputBufferAvailableEvent(movie::Extractor* extractor, movie::Decoder* decoder);

   /**
    * @brief
    * SignalVideoInputBufferAvailable. API to send input buffer available message.
    *
    * @param[in] extractor                      movie::Extractor.
    * @param[in] decoder                        movie::Decoder (video).
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to send input buffer available message to @class MovieVideoInputHandler message queue.
    *
    */
    movie::Status SignalVideoInputBufferAvailable(movie::Extractor* extractor, movie::Decoder* decoder);

   /**
    * @brief
    * RegisterExtractorAndDecoder. API to register extractor and decoder with @class MovieVideoInputHandler.
    *
    * @param[in] extractor                      movie::Extractor.
    * @param[in] decoder                        movie::Decoder (video).
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to register extractor and video decoder.
    *
    */
    movie::Status RegisterExtractorAndDecoder(movie::Extractor* extractor, movie::Decoder *videoDecoder);

   /**
    * @brief
    * Thread function for MovieVideoInputHandler thread.
    *
    * @param[in] arg                            Arguments for thread.
    *
    * @return None
    *
    * @details
    * The thread loop function. The thread will wait on a message queue.
    *
    */
    friend void MovieVideoInputHandlerThreadFunction(void *arg);

   /**
    * @brief
    * SignalFlush. API to send flush message to @class MovieVideoInputHandler
    *
    * @return None
    *
    * @details
    * This API is used to send a flush message to @class MovieVideoInputHandler message queue. The message handler will
    * call Flush() API.
    *
    */
    void SignalFlush();

   /**
    * @brief
    * Flush. API to flush input buffer.
    *
    * @return None
    *
    * @details
    * This API is used to flush input buffer index list. API will clear the list.
    *
    */
    void Flush();

   /**
    * @brief
    * Flush. API to flush input buffer from a given decoder.
    *
    * @param[in] decoder                        movie::Decoder (video).
    *
    * @return None
    *
    * @details
    * This API is used to flush input buffer index list. It will clear the index list of
    * the given video decoder.
    *
    */
    void Flush(movie::Decoder* decoder);

   /**
    * @brief
    * MovieVideoInputHandler destructor.
    *
    * @return None
    */
    ~MovieVideoInputHandler();

private:
    int32_t m_VideoTrackIndex;
    size_t m_ThreadStackSize;
    void* m_ThreadStack;
    nn::os::ThreadType m_ThreadType;
    nn::os::MutexType m_VideoListMutex;

    size_t m_MessageQueueBufferSize;
    uintptr_t *m_MessageQueueBuffer;
    nn::os::MessageQueueType m_MessageQueue;

    enum MoviePlayerMessage
    {
        MoviePlayerMessage_VideoInputAvailable  = 1,
        MoviePlayerMessage_VideoInputThreadExit = 2,
        MoviePlayerMessage_VideoInputInputFlush = 3,
    };

    struct VideoInputBufferInfo
    {
        VideoInputBufferInfo(movie::Extractor* extractor, movie::Decoder* decoder)
            : m_Extractor(extractor),
              m_Decoder(decoder) { }
        movie::Extractor* m_Extractor;
        movie::Decoder* m_Decoder;
        std::vector<int> m_IndexList;
    };
    std::map <movie::Decoder*, VideoInputBufferInfo* > m_VideoInputBufferInfoMap;
};
