﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @file
 * @brief MovieDecoderPlayer API's for media playback.
 */

#pragma once

#include "MovieDecoderEventHandler.h"
#include <list>
void MoviePlayerThreadFunction(void *arg);

/**
 * @brief MovieDecoderPlayer class for media playback
 */
class MovieDecoderPlayer
{
    public:
    /**
     * @brief
     * Enumeration of MovieDecoderPlayer states.
     */
    enum State
    {
        State_NotInitialized    = 0,            //!< NotInitialized state.
        State_Initialized       = 1,            //!< Initialized state.
        State_Preparing         = 2,            //!< Preparing state.
        State_Prepared          = 3,            //!< Prepared state.
        State_Started           = 4,            //!< Started state.
        State_Stopped           = 5,            //!< Stopped state.
        State_Paused            = 6,            //!< Paused state.
        State_PlaybackCompleted = 7,            //!< PlaybackCompleted state.
        State_Error             = 8,           //!< Error state.
    };
    /**
     * @brief
     * Structure to hold audio track properties
     */
    struct TrackInfoAudio
    {
        std::string mime;                       //!< Mime type of the track.
        int sampleRate;                         //!< Audio sample rate in Hertz.
        int channels;                           //!< Number of audio channels.
        int bitRate;                            //!< Audio bitrate(bits per second).
        movie::DecoderType decoderType;         //!< Audio Decoder type.
    };
    /**
     * @brief
     * Structure to hold video track properties
     */
    struct TrackInfoVideo
    {
        std::string mime;                       //!< Mime type of the track.
        int width;                              //!< Video width (Number of distinct vertical lines).
        int height;                             //!< Video height (Number of distinct horizontal lines).
        int bitRate;                            //!< Video bit rate (bits per second).
        int frameRate;                          //!< Video presentation frame rate, frames per seconds.
        movie::DecoderType decoderType;         //!< Video decoder type.
    };

    /**
     * @brief
     * Structure to hold track properties
     */
    struct TrackInfo
    {
        TrackInfoAudio trackInfoAudio;          //!< Audio track information.
        TrackInfoVideo trackInfoVideo;          //!< Video track information.
    };

    /**
     * @brief
     * Enumeration of Track types.
     * @details
     * Track types recognized by MoviePlayer.
     */
    enum TrackType
    {
        TrackType_Unknown   = 0,                //!< Unknown track.
        TrackType_Audio     = 1,                //!< Audio track.
        TrackType_Video     = 2,                //!< Video track.
    };

    /**
     * @brief
     * Enumeration of EventType.
     */
    enum EventType
    {
        EventType_ErrorEvent        = 0,
        EventType_StateChangedEvent = 1,
    };

    enum PlayerEventType
    {
        PlayerEventType_Unknown             = 0,
        PlayerEventType_Prepare             = 1,
        PlayerEventType_VideoRenderComplete = 2,
        PlayerEventType_ThreadExit          = 3,
    };

public:
    /**
     * @brief
     * Create MovieDecoderPlayer instance.
     *
     * @return                                  MovieDecoderPlayer instantiated.
     *
     * @post
     * MovieDecoderPlayer instance created.
     *
     * @details
     * Create an instance of MovieDecoderPlayer to play media files.
     *
     * MovieDecoderPlayer state transition on success:
     *  -> State_UnInitialized.
     */
    MovieDecoderPlayer() NN_NOEXCEPT;

    /**
     * @brief
     * MovieDecoderPlayer destructor.
     *
     * @return                                  MovieDecoderPlayer instance is destroyed.
     *
     * @pre
     * MovieDecoderPlayer instance is created.
     *
     * @post
     * MovieDecoderPlayer deleted.
     *
     * @details
     * MovieDecoderPlayer instance will be deleted. All resources will be released.
     */
    ~MovieDecoderPlayer();

    /**
     * @brief
     * Initialize MovieDecoderPlayer.
     *
     * @retval Status_Success                   MovieDecoderPlayer is initialized successfully.
     * @retval Status_FailedToInitialize        Failed to initialize MovieDecoderPlayer.
     * @retval Status_OutOfMemory               Memory allocation failed.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is instantiated.
     *
     * @post
     * MovieDecoderPlayer is initialized.
     *
     * @details
     * MovieDecoderPlayer initialized. Non-track specific objects are created and initialized.
     *
     * MovieDecoderPlayer state transition on success:
     * State_UnInitialized -> State_Initialized.
     */
    movie::Status Initialize() NN_NOEXCEPT;

    /**
     * @brief
     * Set native window handle to  MovieDecoderPlayer.
     *
     * @param[in] NativeWindowHandle            NativeWindowHandle to set.
     *
     * @retval Status_Success                   MovieDecoderPlayer is initialized successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is instantiated.
     *
     * @details
     * Native window handle is set to  MovieDecoderPlayer. This window handle will be used by renderer.
     *
     * MovieDecoderPlayer state transition on success:
     * No impact on MovieDecoderPlayer state.
     */
    movie::Status SetViNativeWindowHandle(nn::vi::NativeWindowHandle) NN_NOEXCEPT;

    /**
     * @brief
     * Register client events.
     *
     * @param[in] eventType                     Client event type to register.
     * @param[in] event                         Event to register.
     *
     * @retval Status_Success                   Client listener event is registered successfully.
     *
     * @pre
     * MoviePlayer is initialized.
     *
     * @post
     * Client event is registered with MoviePlayer.
     *
     * @details
     * The MovieDecoderPlayer signals an event for the registered event type, when the specific event trigger
     * conditions are met.
     *
     * MovieDecoderPlayer state transition on success:
     * No impact on MovieDecoderPlayer state.
     */
    movie::Status RegisterEvent(EventType eventType, nn::os::EventType *event) NN_NOEXCEPT;

    /**
     * @brief
     * Sets data source for playback.
     *
     * @param[in] mediaPath                     Media path for playback.
     *
     * @retval Status_Success                   Data source is set successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_TrackNotFound             No valid tracks found.
     * @retval Status_UnknownTrackType          Track type not known to MoviePlayer.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MovieDecoderPlayer is instantiated.
     * MovieDecoderPlayer is State_Initialized state.
     *
     * @details
     * Client can use this API to set data source for playback. The MoviePlayer will probe the media
     * using extractors. MovieDecoderPlayer will return error there are no valid or supported tracks to play.
     *
     * MovieDecoderPlayer state transition on success:
     * No impact on MovieDecoderPlayer state.
     */
    movie::Status SetDataSource(movie::Extractor** xtractor, std::string mediaPath, int *trackIndex, movie::DecoderMode decoderMode = movie::DecoderMode_Cpu) NN_NOEXCEPT;

   /**
     * @brief
     * Prepare MovieDecoderPlayer for playback.
     *
     * @retval Status_Success                   MovieDecoderPlayer is prepared and ready.
     * @retval Status_FailedToPrepare           Failed to prepare.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MovieDecoderPlayer is State_Initialized state.
     * Valid data source is set.
     *
     * @post
     * MovieDecoderPlayer is prepared and ready for playback.
     * If client has registered event for state change, the event is signaled when prepare is complete.
     *
     * @details
     * MovieDecoderPlayer will create decoders for first audio and video track present. If audio track is
     * present, audio renderer will be created.  If video track is present video renderer will be created.
     *
     * MovieDecoderPlayer state transition on success:
     * State_Initialized -> State_Prepared.
     */
    movie::Status Prepare() NN_NOEXCEPT;

    /**
     * @brief
     * Start playback of media.
     *
     * @retval Status_Success                   Playback is started.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MovieDecoderPlayer is State_Prepared state.
     *
     * @post
     * MovieDecoderPlayer is playing the media file.
     * If client has registered event for state change, the event is signaled when playback is started.
     *
     * @details
     * MovieDecoderPlayer will start rendering audio video data.
     *
     * MovieDecoderPlayer state transition on success:
     * State_Prepared -> State_Started.
     */
    movie::Status Start() NN_NOEXCEPT;

    /**
     * @brief
     * Stop playback of media.
     *
     * @retval Status_Success                   Playback is stopped.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MovieDecoderPlayer is State_Started or State_Stopped state
     *
     * @post
     * MovieDecoderPlayer stops playing the media file.
     * If client has registered event for state change, the event is signaled when playback is stopped.
     *
     * @details
     * MovieDecoderPlayer will stop rendering audio and video.
     *
     * MovieDecoderPlayer state transition on success:
     * State_Started -> State_Stopped.
     */
    movie::Status Stop() NN_NOEXCEPT;

    /**
     * @brief
     * Pause media playback.
     *
     * @retval Status_Success                   Pause is successful.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is State_Started or State_Paused state
     *
     * @post
     * MovieDecoderPlayer is paused.
     *
     * @details
     * MovieDecoderPlayer pauses rendering of audio and video data.
     *
     * MovieDecoderPlayer state transition on success:
     * State_Started -> State_Paused.
     * State_Paused  -> State_Paused.
     */
    movie::Status Pause();

    /**
     * @brief
     * Resumes media playback.
     *
     * @retval Status_Success                    Resume is successful.
     * @retval Status_OperationFailed            An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is State_Paused
     *
     * @post
     * MovieDecoderPlayer resumes rendering of audio and video data.
     *
     * @details
     * MovieDecoderPlayer resumes, rendering of audio and video data.
     *
     * MoviePlayer state transition on success:
     * State_Paused -> State_Started.
     */
    movie::Status Resume() NN_NOEXCEPT;

    /**
     * @brief
     * Return current playback position in microseconds.
     *
     * @param[out] playbackPositionUs           Current playback position in time, in microsecond units.
     *
     * @retval Status_Success                   Playback position is retrieved successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is State_Initialized state.
     *
     * @details
     * The playback position of last frame (audio or video) in microseconds is retrieved and
     * returned to client
     *
     * MovieDecoderPlayer state transition on success:
     * No impact on MovieDecoderPlayer state.
     */
    movie::Status GetPlaybackPosition(movie::Extractor* extractor, int64_t *playbackPositionUs);

    /**
    * @brief
    * Select default video track for playback.
    *
    * @param[out] trackNumber                  Selected video track number.
    *
    * @retval Status_Success                   video track is selected successfully.
    * @retval Status_OperationFailed           An unknown error occurred.
    *
    * @pre
    * MovieDecoderPlayer is in State_Initialized state.
    * Valid data source is set.
    *
    * @details
    * Use this API to select video track present in the media file.
    * MovieDecoderPlayer selects first video track.
    *
    * MovieDecoderPlayer state transition on success:
    * No impact on MovieDecoderPlayer state.
    */
    movie::Status SelectVideoTrack(movie::Extractor* extractor, int *trackNumber) NN_NOEXCEPT;

    /**
     * @brief
     * Get track type.
     *
     * @param[in] trackNumber                   Track number to retrieve track type.
     * @param[out] trackType                    Track type for the requested track.
     *
     * @retval Status_Success                   Track type is retrieved successfully.
     * @retval Status_ErrorBadValue             Invalid track number.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is in State_Initialized state.
     * Valid data source is set.
     *
     * @details
     * The track type for the requested track number is retrieved and returned to client.
     *
     * MovieDecoderPlayer state transition on success:
     * No impact on MovieDecoderPlayer state.
     */
    movie::Status GetTrackType(movie::Extractor* extractor, int trackNumber, TrackType *trackType) const NN_NOEXCEPT;

   /**
     * @brief
     * Enable looping mode
     *
     * @param[in] loop                          Infinite looping mode.
     *
     * @retval Status_Success                   Loop flag is set to MoviePlayer.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is instantiated.
     *
     * @details
     * Use this API to set looping mode. Once set, the MovieDecoderPlayer will loop media file until
     * it is reset. This can be used to play a single media file indefinitely. When end of stream
     * is reached MovieDecoderPlayer will start playback from the beginning.
     *
     * MovieDecoderPlayer state transition on success:
     * No impact on MovieDecoderPlayer state.
     */
    movie::Status SetLooping(bool loop) NN_NOEXCEPT;

    /**
     * @brief
     * Get looping mode
     *
     * @param[out] loop                          Infinite looping mode.
     *
     * @retval Status_Success                    Return current playback loop mode.
     * @retval Status_OperationFailed            An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is instantiated.
     *
     * @details
     * Current playback loop mode is returned to client.
     *
     * MovieDecoderPlayer State:
     * No changes to MovieDecoderPlayer state.
     *
     */
    movie::Status GetLooping(bool *loop) const NN_NOEXCEPT;

    /**
     * @brief
     * Get current MovieDecoderPlayer state.
     *
     * @param[out] state                        MovieDecoderPlayer state.
     *
     * @retval Status_Success                   Current state of MovieDecoderPlayer is returned.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MovieDecoderPlayer is instantiated.
     *
     * @details
     * Current state of MovieDecoderPlayer is returned.
     *
     * MoviePlayer state transition on success:
     * No impact on MovieDecoderPlayer state.
     */
    movie::Status GetState(State *state) NN_NOEXCEPT;

    /**
     * @brief
     * Return last movie error
     *
     * @return                                  Return last error.
     *
     * @pre
     * MovieDecoderPlayer is in State_Initialized state.
     *
     * @details
     * Client can use this API to get last error from MovieDecoderPlayer.
     * When client receives error event, they call this API to get the error codes.
     */
    movie::Status GetLastError() NN_NOEXCEPT;

    /**
     * @brief
     * Get string value of Player state enumeration.
     *
     * @return                                  Return string for state enum
     *
     * @pre
     * MovieDecoderPlayer is instantiated.
     *
     * @details
     * Client can use this API to get string value of MovieDecoderPlayer state enumeration.
     */
    const char* StateToString(State state);
private:
    NN_DISALLOW_COPY(MovieDecoderPlayer);
    NN_DISALLOW_MOVE(MovieDecoderPlayer);
    struct EventUserData
    {
        EventUserData(nn::os::EventType *event, PlayerEventType eventType, movie::Decoder* decoder = nullptr)
            : m_Event(event),
            m_EventType(eventType),
            m_Decoder(decoder){}
        nn::os::EventType *m_Event;
        PlayerEventType m_EventType;
        movie::Decoder* m_Decoder;
    };

    friend void MoviePlayerThreadFunction(void *arg);
    movie::Status UnSelectTrack(movie::Extractor* extractor, int trackNumber) NN_NOEXCEPT;
    movie::Status GetTrackCount(movie::Extractor* extractor, int *trackCount) const NN_NOEXCEPT;
    movie::Status SelectTrack(movie::Extractor* extractor, int trackNumber) NN_NOEXCEPT;
    movie::Status GetTrackInfo(movie::Extractor* extractor, int trackNumber, TrackInfo* trackInfo) NN_NOEXCEPT;
    movie::Status RegisterPlayerEvent(PlayerEventType eventType, nn::os::EventType *event, nn::os::MultiWaitHolderType *holder, movie::Decoder* decoder = nullptr) NN_NOEXCEPT;
    movie::Status GetContainerType(std::string mediaPath, movie::ContainerType *containertype);
    movie::Status GetDecoderType(movie::Extractor* extractor, int trackNumber, movie::DecoderType *decoderType);
    movie::Status ValidateStateTransition(State newState);
    movie::Status SetState(State state);
    void UpdateLastErrorAndSignalClient(movie::Status lastError);
    void SetVideoPlaybackComplete(movie::Decoder* decoder = nullptr);
    void ReSetVideoPlaybackComplete(movie::Decoder* decoder = nullptr);
    bool IsMediaPlaybackComplete();
    movie::Status ValidateTrackNumber(movie::Extractor* extractor, int trackNumber);
    void SetLastError(movie::Status movieStatus);
    void DeleteUserData(EventUserData* userData);
    movie::Status CreateExtractorForPlayer(std::string mediaPath, movie::Extractor** extractor);
    movie::Status PrepareForLoopBack(movie::Decoder* decoder);
    bool IsInRequestedState(State state);
    void DeletePlayerOwnedObjects();
    void FlushInputAndOutputHandlers();
    void FlushDecoders();
    State m_State;
    nn::os::EventType *m_ErrorEvent;
    nn::os::EventType *m_StateChangedEvent;
    nn::vi::NativeWindowHandle  m_NativeWindow;
    TrackInfo m_TrackInfo;
    bool m_Loop;
    MovieVideoInputHandler *m_MovieVideoInputHandler;
    MovieVideoOutputHandler *m_MovieVideoOutputHandler;
    MovieDecoderEventHandler *m_MovieDecoderEventHandler;
    uint64_t m_CoreMask;
    nn::os::ThreadType m_ThreadType;
    size_t m_ThreadStackSize;
    void* m_ThreadStack;
    movie::Status m_LastError;
public:
    nn::os::MultiWaitType m_PlayerMultiWait;
private:
    nn::os::EventType m_PrepareEvent;
    nn::os::EventType m_ThreadExitEvent;
    nn::os::MultiWaitHolderType m_PrepareEventHolder;
    nn::os::MultiWaitHolderType m_ThreadExitEventHolder;
    bool m_EventsInitialized;
    bool m_ThreadCreated;
    bool m_ThreadStarted;
    nn::os::Mutex m_StateLock;
    nn::os::Mutex m_ApiLock;
    int64_t m_MediaDurationUs;
    std::vector<std::string> m_MediaPaths;
    struct DecoderInfo
    {
        DecoderInfo() :
            decoder(nullptr),
            extractor(nullptr),
            videoPlaybackComplete(false)
        { }
        movie::Decoder* decoder;
        movie::Extractor* extractor;
        movie::DecoderMode decoderMode;
        movie::DecoderType decoderType;
        MediaClock *mediaClock;
        movie::DecoderOutputFormat decoderOutputFormat;
        movie::Status lastError;
        nn::os::EventType *errorEvent;
        nn::os::EventType *formatChangedEvent;
        nn::os::EventType *inputBufferAvailableEvent;
        nn::os::EventType *outputBufferAvailableEvent;
        nn::os::MultiWaitHolderType *errorEventHolder;
        nn::os::MultiWaitHolderType *formatChangedEventHolder;
        nn::os::MultiWaitHolderType *inputBufferAvailableEventHolder;
        nn::os::MultiWaitHolderType *outputBufferAvailableEventHolder;
        movie::DecoderEvents decoderEvents;
        int trackIndex;
        std::vector<int> bufferIndexList;
        nn::os::MutexType bufferIndexListMutex;
        bool eosReached;
        int width;
        int height;
        movie::MediaData trackFormat;
        nn::os::EventType renderCompleteEvent;
        nn::os::MultiWaitHolderType renderCompleteEventHolder;
        bool videoPlaybackComplete;
    };
    std::vector<DecoderInfo> m_DecoderInfoList;
    int m_TrackIndex;
    movie::DecoderMode m_DecoderMode;
};
