﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "MovieDecoderEventHandler.h"
/**
 * @brief
 * MovieDecoderEventHandler class for handling video decoder events
 *
 * @details
 *
 * Purpose:
 *     The purpose of @class MovieDecoderEventHandler is to respond to video decoder events. @class MovieDecoderEventHandler
 * registers events with video decoder. When any of events are signaled from the decoder, @class MovieDecoderEventHandler
 * acquires information about the event and sends message to input or output handler message queues.
 *
 * Setup:
 *     @class MovieDecoderPlayer will create an instance of @class MovieDecoderEventHandler. The constructor is provided with
 * input/output handles. During initialization a thread is created for receiving events from decoder. The thread waits on
 * a multi wait.
 *
 * Main Processing:
 *     @class MovieDecoderEventHandler thread multi-wait will be signaled by movie decoders when they have input, output
 * format change events or if any error occurs. @class MovieDecoderEventHandler identifies the events based on even user data
 * and delegates the event information to respective input/output handlers.
 *
 * Teardown:
 *    When @class MovieDecoderEventHandler receives thread exit event, the thread will exit and all resources are freed.
 *
 */
MovieDecoderEventHandler::MovieDecoderEventHandler()
    : m_ThreadStackSize(1024 * 256),
    m_ThreadStack(nullptr),
    m_VideoDecoderInFlush(false)
{
}

MovieDecoderEventHandler::~MovieDecoderEventHandler()
{
    nn::os::SignalEvent(&m_ThreadExitEvent);
    nn::os::WaitThread(&m_ThreadType);
    nn::os::DestroyThread(&m_ThreadType);
    nn::os::UnlinkMultiWaitHolder(&m_ThreadExitEventHolder);
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_ThreadExitEventHolder));
    nn::os::FinalizeMultiWaitHolder(&m_ThreadExitEventHolder);
    nn::os::FinalizeEvent(&m_ThreadExitEvent);
    nn::os::FinalizeEvent(&m_FormatChangedPlayerEvent);
    for( auto &element : m_DecoderEventMap )
    {
        nn::os::FinalizeEvent(&element.first->errorEvent);
        nn::os::FinalizeEvent(&element.first->inputBufferAvailableEvent);
        nn::os::FinalizeEvent(&element.first->outputBufferAvailableEvent);
        nn::os::FinalizeEvent(&element.first->formatChangedEvent);
        nn::os::UnlinkMultiWaitHolder(&element.first->errorEventHolder);
        DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&element.first->errorEventHolder));
        nn::os::FinalizeMultiWaitHolder(&element.first->errorEventHolder);
        nn::os::UnlinkMultiWaitHolder(&element.first->formatChangedEventHolder);
        DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&element.first->formatChangedEventHolder));
        nn::os::FinalizeMultiWaitHolder(&element.first->formatChangedEventHolder);
        nn::os::UnlinkMultiWaitHolder(&element.first->inputBufferAvailableEventHolder);
        DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&element.first->inputBufferAvailableEventHolder));
        nn::os::FinalizeMultiWaitHolder(&element.first->inputBufferAvailableEventHolder);
        nn::os::UnlinkMultiWaitHolder(&element.first->outputBufferAvailableEventHolder);
        DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&element.first->outputBufferAvailableEventHolder));
        nn::os::FinalizeMultiWaitHolder(&element.first->outputBufferAvailableEventHolder);
        delete element.first;
    }
    nn::os::FinalizeMultiWait(&m_DecoderMultiWait);
    if( m_ThreadStack != NULL )
        free(m_ThreadStack);
}

movie::Status MovieDecoderEventHandler::Initialize(uint64_t coreMask)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();
    if( coreMask )
        movie::SetCoreMask(coreMask);
    nn::os::InitializeMultiWait(&m_DecoderMultiWait);
    nn::os::InitializeEvent(&m_ThreadExitEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_FormatChangedPlayerEvent, false, nn::os::EventClearMode_ManualClear);
    EventUserData *userData = new EventUserData(&m_ThreadExitEvent, DecoderEventType_ThreadExit, nullptr, nullptr);
    if( userData == nullptr )
        return movie::Status_OutOfMemory;
    nn::os::InitializeMultiWaitHolder(&m_ThreadExitEventHolder, &m_ThreadExitEvent);
    nn::os::SetMultiWaitHolderUserData(&m_ThreadExitEventHolder, ( uintptr_t ) userData);
    nn::os::LinkMultiWaitHolder(&m_DecoderMultiWait, &m_ThreadExitEventHolder);
    m_ThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
    if( m_ThreadStack == nullptr )
        return movie::Status_OutOfMemory;
    nnResult = nn::os::CreateThread(&m_ThreadType,
        &MovieDecoderEventHandlerThreadFunction,
        ( void* )this,
        m_ThreadStack,
        m_ThreadStackSize,
        nn::os::DefaultThreadPriority);
    if( nnResult.IsSuccess() )
        nn::os::SetThreadName(&m_ThreadType, "MovieDecoderEventHandler");
    else
        movieStatus = movie::Status_FailedToCreateThread;
    return movieStatus;
}

void MovieDecoderEventHandler::Start()
{
    nn::os::StartThread(&m_ThreadType);
}

movie::Status MovieDecoderEventHandler::RegisterInputHandler(MovieVideoInputHandler *inputHandler)
{
    m_MovieVideoInputHandler = inputHandler;
    return movie::Status_Success;
}

movie::Status MovieDecoderEventHandler::RegisterOutputHandler(MovieVideoOutputHandler *outputHandler)
{
    m_MovieVideoOutputHandler = outputHandler;
    return movie::Status_Success;
}

movie::Status MovieDecoderEventHandler::RegisterExtractorAndDecoder(movie::Extractor* extractor, movie::Decoder *decoder, movie::DecoderEvents *decoderEvents)
{
    movie::Status movieStatus = movie::Status_Success;
    DecoderEventInfo *decoderEventInfo = new DecoderEventInfo();
    if( decoderEventInfo == nullptr )
        return movie::Status_OutOfMemory;
    movieStatus = RegisterEvent(extractor, decoder, &decoderEventInfo->errorEvent, &decoderEventInfo->errorEventHolder, DecoderEventType::DecoderEventType_Error);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    movieStatus = RegisterEvent(extractor, decoder, &decoderEventInfo->formatChangedEvent, &decoderEventInfo->formatChangedEventHolder, DecoderEventType::DecoderEventType_FormatChanged);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    movieStatus = RegisterEvent(extractor, decoder, &decoderEventInfo->inputBufferAvailableEvent, &decoderEventInfo->inputBufferAvailableEventHolder, DecoderEventType::DecoderEventType_InputBuffer);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    movieStatus = RegisterEvent(extractor, decoder, &decoderEventInfo->outputBufferAvailableEvent, &decoderEventInfo->outputBufferAvailableEventHolder, DecoderEventType::DecoderEventType_OutputBuffer);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    decoderEvents->errorEvent = &decoderEventInfo->errorEvent;
    decoderEvents->formatChangedEvent = &decoderEventInfo->formatChangedEvent;
    decoderEvents->inputBufferAvailableEvent = &decoderEventInfo->inputBufferAvailableEvent;
    decoderEvents->outputBufferAvailableEvent = &decoderEventInfo->outputBufferAvailableEvent;
    m_DecoderEventMap.insert(std::make_pair(decoderEventInfo, decoder));
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::RegisterEvent(movie::Extractor* extractor, movie::Decoder *decoder, nn::os::EventType *event, nn::os::MultiWaitHolderType *holder, DecoderEventType eventType)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::InitializeEvent(event, false, nn::os::EventClearMode_ManualClear);
    EventUserData *userData = new EventUserData(event, eventType, extractor, decoder);
    if( userData == nullptr )
        return movie::Status_OutOfMemory;
    nn::os::InitializeMultiWaitHolder(holder, event);
    nn::os::SetMultiWaitHolderUserData(holder, ( uintptr_t ) userData);
    nn::os::LinkMultiWaitHolder(&m_DecoderMultiWait, holder);
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::HandleInputBufferAvailableEvent(movie::Extractor* extractor, movie::Decoder *decoder)
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    movie::Buffer buffer;
    int32_t numberOfBuffersAvailable = 0;
    if( ( decoder != nullptr ) && ( m_MovieVideoInputHandler != nullptr ) )
    {
        movieStatus = decoder->AcquireInputBufferIndex(&index, &numberOfBuffersAvailable);
        if( movieStatus == movie::Status_Success )
        {
            m_MovieVideoInputHandler->AddBufferToVideoIndexList(decoder, index);
            for( int32_t i = 0; i < numberOfBuffersAvailable; i++ )
            {
                int32_t remainingBuffers = 0;
                movieStatus = decoder->AcquireInputBufferIndex(&index, &remainingBuffers);
                if( movieStatus == movie::Status_Success )
                    m_MovieVideoInputHandler->AddBufferToVideoIndexList(decoder, index);
            }
            m_MovieVideoInputHandler->SignalVideoInputBufferAvailable(extractor, decoder);
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::HandleOutputBufferAvailableEvent(movie::Decoder *decoder)
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    int64_t presentationTimeUs = 0ll;
    uint32_t flags = 0;
    int32_t numberOfBuffersAvailable = 0;
    movie::Buffer buffer;
    if( ( decoder != nullptr ) && ( m_MovieVideoOutputHandler != nullptr ) )
    {
        movieStatus = decoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &numberOfBuffersAvailable);
        if( ( movieStatus == movie::Status_Success ) || ( movieStatus == movie::Status_EndOfStream ) )
        {
            m_MovieVideoOutputHandler->OnOutputAvailable(decoder, index, presentationTimeUs, flags);
            for( int32_t i = 0; i < numberOfBuffersAvailable; i++ )
            {
                int32_t remainingBuffers = 0;
                movieStatus = decoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &remainingBuffers);
                if( ( movieStatus == movie::Status_Success ) || ( movieStatus == movie::Status_EndOfStream ) )
                    m_MovieVideoOutputHandler->OnOutputAvailable(decoder, index, presentationTimeUs, flags);
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::HandleFormatChangedEvent()
{
    m_VideoFormatChangeEventReceived = true;
    CheckAndSignalFormatChangeWaiter();
    return movie::Status_Success;
}

movie::Status MovieDecoderEventHandler::HandleErrorEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    return movieStatus;
}

void MovieDecoderEventHandler::DeleteUserData(EventUserData* userData)
{
    if( userData != nullptr )
        delete userData;
}

void MovieDecoderEventHandler::CheckAndSignalFormatChangeWaiter()
{
    if( m_VideoFormatChangeEventReceived == true )
    {
        nn::os::SignalEvent(&m_FormatChangedPlayerEvent);
        m_VideoFormatChangeEventReceived = false;
        if( m_VideoDecoder == nullptr )
            m_VideoFormatChangeEventReceived = true;
    }
}

void MovieDecoderEventHandler::SignalInputBufferAvailableEvent()
{
    for( auto element : m_DecoderEventMap )
        nn::os::SignalEvent(&element.first->inputBufferAvailableEvent);
}

void MovieDecoderEventHandler::SignalOutputBufferAvailableEvent()
{
    for( auto element : m_DecoderEventMap )
        nn::os::SignalEvent(&element.first->outputBufferAvailableEvent);
}

void MovieDecoderEventHandler::WaitForFormatChange()
{
    nn::os::TimedWaitEvent(&m_FormatChangedPlayerEvent, nn::TimeSpan::FromMilliSeconds(5000));
    nn::os::ClearEvent(&m_FormatChangedPlayerEvent);
}

void MovieDecoderEventHandlerThreadFunction(void *arg)
{
    MovieDecoderEventHandler *movieDecoderEventHandler = ( MovieDecoderEventHandler*) arg;
    bool exitThread = false;
    if( movieDecoderEventHandler != nullptr )
    {
        // Wait for events from movie::decoder and process them.
        for( ;; )
        {
            nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&movieDecoderEventHandler->m_DecoderMultiWait);
            uintptr_t userData = GetMultiWaitHolderUserData(holder);
            MovieDecoderEventHandler::EventUserData *eventUserData = ( MovieDecoderEventHandler::EventUserData* ) userData;
            if( eventUserData == nullptr )
                break;
            MovieDecoderEventHandler::DecoderEventType eventype = eventUserData->m_EventType;
            nn::os::EventType *event = eventUserData->m_Event;
            nn::os::ClearEvent(event);
            switch( eventype )
            {
            case MovieDecoderEventHandler::DecoderEventType_InputBuffer:
                    movieDecoderEventHandler->HandleInputBufferAvailableEvent(eventUserData->m_Extractor, eventUserData->m_Decoder);
                break;
            case MovieDecoderEventHandler::DecoderEventType_OutputBuffer:
                    movieDecoderEventHandler->HandleOutputBufferAvailableEvent(eventUserData->m_Decoder);
                break;
            case MovieDecoderEventHandler::DecoderEventType_FormatChanged:
                movieDecoderEventHandler->HandleFormatChangedEvent();
                break;
            case MovieDecoderEventHandler::DecoderEventType_Error:
                movieDecoderEventHandler->HandleErrorEvent();
                break;
            case MovieDecoderEventHandler::DecoderEventType_ThreadExit:
                exitThread = true;
                break;
            default:
                break;
            }
            if( exitThread == true )
                break;
        }
    }
}
