﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
 * @examplesource{MediaPlayerMseSample.cpp,PageSampleMediaPlayerMseSample}
 *
 * @brief
 *  movie player API による動画再生サンプル
 */

#include <getopt.h>
#include "MediaPlayerMseSample.h"
#include "MediaPlayerObserver.h"
#include "MediaPlayerUtilities.h"
#include "HidHandler.h"
#include "HeapTracker.h"
#include "MseSourceBuffer.h"
#include "MseExtractor.h"
#include <movie/Utils.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>

#include <nv/nv_MemoryManagement.h>
#include <sys/time.h>

#include <libxml/xmlreader.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include <string>
#include <cinttypes>
#include <limits>
#include <cmath>

#ifdef ENABLE_PROFILER
    #include <nn/profiler.h>
#endif

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>

// Instantiate Globals
uint8_t g_GraphicsHeap[g_GraphicsSystemMemorySize] __attribute__(( aligned(4096) ));
uint8_t g_FsHeapBuffer[FsHeapSize];
nn::lmem::HeapHandle g_FsHeap;
bool USE_PROXY = false;
std::string PROXY_SETTING_HOST = "proxy-host.com";
int PROXY_SETTING_PORT = 8080;
std::string PROXY_SETTING_USERPWD = "UserName:Password";
unsigned long PROXY_SETTING_AUTH = CURLAUTH_BASIC;
bool USE_HEAP_TRACKING = false;
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

void FsInitHeap()
{
    g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize,
        nn::lmem::CreationOption_DebugFill);
}

void* FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

/* MediaPlayer Sample Usage */
static void Usage()
{
    NN_LOG( "\n ERROR:: Invalid options See Readme.MediaPlayerMseSample.txt \n");
}

struct TrackInfo
{
    std::string initSegment;
    std::vector<std::string> dataSegments;
    uint32_t initSegmentSize;
};

struct MPDInfo
{
    std::vector<TrackInfo> trackList;
};


void print_element_names(xmlNode * a_node, TrackInfo& trackInfo)
{
    xmlNode *cur_node = NULL;

    for (cur_node = a_node; cur_node; cur_node = cur_node->next)
    {
        if (cur_node->type == XML_ELEMENT_NODE) {
            NN_LOG("node type: Element, name: %s,\n", cur_node->name);
            if (!strcmp((const char*)cur_node->name, "Initialization"))
            {
                xmlAttr *cur_attr = NULL;
                for (cur_attr = cur_node->properties; cur_attr; cur_attr = cur_attr->next)
                {
                    if (!strcmp((const char*)cur_attr->name, "sourceURL"))
                    {
                        trackInfo.initSegment = (const char*)cur_attr->children->content;
                    }
                }
            }
            if (!strcmp((const char*)cur_node->name, "SegmentURL"))
            {
                xmlAttr *cur_attr = NULL;
                for (cur_attr = cur_node->properties; cur_attr; cur_attr = cur_attr->next)
                {
                    if (!strcmp((const char*)cur_attr->name, "media"))
                    {
                        trackInfo.dataSegments.push_back((const char*)cur_attr->children->content);
                    }
                }
            }
            if (!strcmp((const char*)cur_node->name, "SegmentList"))
            {
                //clear the filename list if a new SegmentList is reached
                //fileNames.clear();
            }
        }

        print_element_names(cur_node->children, trackInfo);
    }
}

/* Process startup setup */
extern "C" void nninitStartup()
{
    const size_t heapSize = 256 << 20;
    const size_t mallocSize = 128 << 20;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    //ASSERT(result.IsSuccess());

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    //ASSERT(result.IsSuccess());

    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

extern "C" int* __errno()
{
    static int i = 0;
    return &i;
}

/* Process main entry */
extern "C" void nnMain()
{
    { // We use an embedded scope to make sure local destructors have been called
      // before we print our final heap tracking statistics at the end of nnMain()

#ifdef ENABLE_PROFILER
    std::vector<char> profiler_buffer__(nn::profiler::MinimumBufferSize);
    nn::profiler::Initialize(profiler_buffer__.data(), profiler_buffer__.size());
    NN_UTIL_SCOPE_EXIT { nn::profiler::Finalize(); };
#endif

    bool isNetworkFile = false;
    nn::hid::InitializeDebugPad();

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    // Set allocator callback functions

    movie::SetAllocator(mmAllocate, mmDeallocate, mmReallocate, &MMHeap());
    // Donate memory for graphics driver to work in
    nv::InitializeGraphics(g_GraphicsHeap, g_GraphicsSystemMemorySize);
    nv::SetGraphicsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::SetGraphicsDevtoolsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());

    if( argc <= 1 )
    {
       Usage();
       return;
    }

    bool playAudio = true;
    bool playVideo = true;
    float playRate = 1.0; //Normal;
    bool setLooping = false;
    bool invalidOptions = false;
    bool isVideoTrack = false;
    bool unrestrictedResolution = false;
    const char *configFile = NULL;
    const char *urlFile = NULL;
    movie::PlayerResMax maxPlayerRes = movie::PlayerResMax_None;
    movie::VideoDecodeMode videoDecodeMode = movie::VideoDecodeMode_NativeTexture;
    movie::OutputFormat videoOutputFormat = movie::OutputFormat_VideoColorAbgr;
    std::string urlString;
    float volume = 1.0; //Normal

    for(int options = 0; options < argc; options ++)
    {
        if( !strcmp(argv[options], "-a" ) )
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "0") ) ||  ( !strcmp(argv[options], "1") ) )
                {
                    playAudio = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if( !strcmp(argv[options], "-v") )
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "0") ) ||  ( !strcmp(argv[options], "1") ) )
                {
                    playVideo = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if( !strcmp(argv[options], "-ur") )
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "0") ) ||  ( !strcmp(argv[options], "1") ) )
                {
                    invalidOptions = false;
                    unrestrictedResolution = atoi(argv[options]);
                    continue;
                }
            }
            break;
        }

        if( !strcmp(argv[options], "-r"))
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                playRate = atof(argv[options]);
                if (playRate < 0.0001f) {   // This limit is just to rule out negative and close-to-zero values, as well as values that
                    invalidOptions = true;  // atof() could not recognize as floats (in which case it returns 0.0)
                } else {
                    invalidOptions = false;
                }
                continue;
            }
            break;
        }

        if( !strcmp(argv[ options ], "-vol") )
        {
            options++;
            invalidOptions = true;
            if( options < argc )
            {
                volume = atof(argv[ options ]);
                if( ( volume < 0.0 ) || ( volume > 4.0 ) )
                    invalidOptions = true;
                else
                    invalidOptions = false;
                continue;
            }
            break;
        }

        if( !strcmp(argv[options], "-l"))
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "0") ) ||  ( !strcmp(argv[options], "1") ) )
                {
                    setLooping = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if (!strcmp(argv[options], "-h"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                {
                    USE_HEAP_TRACKING = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if( !strcmp(argv[options], "-c"))
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                configFile = argv[options];
                invalidOptions = false;
                continue;
            }
            break;
        }

        if (!strcmp(argv[options], "-u"))
        {
            options++;
            invalidOptions = true;
            if (options < argc) {
                urlFile = argv[options];
                invalidOptions = false;
                continue;
            }
            break;
        }

        if (!strcmp(argv[options], "-dm"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "1")) || (!strcmp(argv[options], "2")) )
                {
                    int videoDecodeModeSelection = atoi(argv[options]);
                    switch( videoDecodeModeSelection )
                    {
                    case 1:
                        videoDecodeMode = movie::VideoDecodeMode_Cpu;
                        videoOutputFormat = movie::OutputFormat_VideoColorNv12;
                        break;
                    case 2:
                        videoDecodeMode = movie::VideoDecodeMode_NativeTexture;
                        videoOutputFormat = movie::OutputFormat_VideoColorAbgr;
                        break;
                    default:
                        break;
                    }
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }
    }

    if( invalidOptions == true )
    {
        Usage();
        return;
    }

    switch( videoDecodeMode )
    {
    case movie::VideoDecodeMode_Cpu:
        NN_LOG("\n MediaPlayerMseSample:: Using [movie::VideoDecodeMode_Cpu] video decode mode \n");
        break;
    case movie::VideoDecodeMode_NativeTexture:
        NN_LOG("\n MediaPlayerMseSample:: Using [movie::VideoDecodeMode_NativeTexture] video decode mode \n");
        break;
    default:
        NN_LOG("\n MediaPlayerMseSample:: Using [movie::VideoDecodeMode_NativeTexture] video decode mode \n");
        break;
    }

    if( unrestrictedResolution )
    {
        maxPlayerRes = movie::PlayerResMax_None;
        NN_LOG("\n MediaPlayerMseSample:: Unrestricted maximum resolution\n");
    }

    timeval playerTimeVal;
    int64_t startTime = 0ll;
    int64_t endTime = 0ll;

    const char* inputFileName = inputFileName = argv[1];
    std::string path = inputFileName;
    std::string delimiter = ":";
    std::string token = path.substr(0, path.find(delimiter));
    if ( token == "http" || token == "https" || urlFile )
    {
        NN_LOG("\nError! Sample does not support network files!\n\n");
        return;
    }
    nn::Result resultHostMount = nn::fs::MountHostRoot();
    if( resultHostMount.IsFailure() )
    {
        NN_LOG( "\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
            resultHostMount.GetModule(),
            resultHostMount.GetDescription() );
        return;
    }
    if(configFile && !ProcessConfigFile(configFile))
        return;
    /* Check whether media exists if it is a local file */
    if (!isNetworkFile && token != "file")
    {
        nn::fs::FileHandle fileHandle;
        fileHandle.handle = NULL;
        nn::Result result = nn::fs::OpenFile(&fileHandle, inputFileName, nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            NN_LOG("\n Failed to open %s \n \n Terminating PlayerSample MediaPlayerMseSample\n", inputFileName);
            return;
        }
        else
        {
            nn::fs::CloseFile(fileHandle);
        }
    }

    if (urlFile)
    {
        if (ProcessUrlFile(urlFile, &urlString))
        {
            inputFileName = urlString.c_str();
        }
        else
        {
            return;
        }
    }

    NN_LOG( "\n MediaPlayerMseSample:: Input File Name : %s\n", inputFileName );

    movie::PlayerConfig config;
    config.returnAudioDataToClient = false;
    config.returnVideoDataToClient = true;
    // Usage of "useNativeBuffers" is being deprecated, use movie::videoDecodeMode
    // config.useNativeBuffers = useNativeBuffers;
    config.videoDecodeMode = videoDecodeMode;
    config.returnSyncedData = true;
    config.useClientMemoryForOutput = true;
    config.autoReleaseOutputBuffers = true;
    config.audioFormat = movie::OutputFormat_AudioPcm16;
    config.videoFormat = videoOutputFormat;
    config.maxResolution = maxPlayerRes;
    movie::MediaData bufferAlignment;
    int nv12YAlignment = 32;
    int nv12UvAlignment = 32;
    int pcmAlignment = 32;
    bufferAlignment.SetInt32("nv12-y-alignment", nv12YAlignment);
    bufferAlignment.SetInt32("nv12-uv-alignment", nv12UvAlignment);
    bufferAlignment.SetInt32("pcm-alignment", pcmAlignment);

    /*Set browser config, below are some sample values set to test this feature*/
    movie::BrowserConfig browserConfig;
    browserConfig.coreMask = 0x00;
    browserConfig.sfThreadPriority = 16;
    browserConfig.audioRendererThreadPriority = 16;
    browserConfig.videoRendererThreadPriority = 16;


    MediaPlayerObserver* mediaPlayerObserver = new MediaPlayerObserver;
    if( mediaPlayerObserver == NULL )
    {
        NN_LOG( "\n Failed to Create MediaPlayerMseSample - Terminating PlayerSample \n" );
        return;
    }
    mediaPlayerObserver->Initialize(&config);

    NN_LOG( "\n MediaPlayerMseSample:: Create \n" );
    movie::BrowserPlayer* player = NULL;
    movie::BrowserPlayer::Create(&player, &config, &browserConfig);
    if( player == NULL )
    {
        NN_LOG( "\n MediaPlayerMseSample:: Failed to Create Player  - Terminating PlayerSample \n" );
        mediaPlayerObserver->Finalize(&config);
        nn::fs::UnmountHostRoot();
        delete mediaPlayerObserver;
        return;
    }
    mediaPlayerObserver->SetMoviePlayer(player);

    HidHandler hidHandler(player, mediaPlayerObserver, setLooping);
    hidHandler.Initialize();

    movie::Status movieStatus = movie::Status_Success;
    bool errorCleanUp = false;


    movie::sample::MseSourceBuffer mseSourceBuffer;
    NN_LOG("\n BrowserPlayerSample :: Create MSESource \n");

    int64_t fileLength = 0;
    char* fileBuffer = nullptr;
    char* directoryPath = nullptr;
    char* file = nullptr;
    char* segmentName = nullptr;
    nn::fs::FileHandle fileHandle;
    fileHandle.handle = NULL;
    nn::Result result;

    int64_t segmentLength = 0;
    char* segmentBuffer = nullptr;

    bool initializationSegmentSet = false;
    bool bailTest = false;

    result = nn::fs::OpenFile(&fileHandle, inputFileName, nn::fs::OpenMode_Read);

    if (result.IsFailure())
    {
        NN_LOG("\n Failed to open %s \n \n Exiting BrowserPlayerSample \n", inputFileName);
        bailTest = true;
    }
    else
    {
        // Read the file into a buffer
        nn::fs::GetFileSize(&fileLength, fileHandle);

        fileBuffer = new char[fileLength];

        nn::fs::ReadFile(fileHandle, 0, fileBuffer, fileLength);
        nn::fs::CloseFile(fileHandle);

        // Determine the path of the file
        int dirPos = 0;
        int directoryLength = 0;
        while (inputFileName[dirPos] != '\0')
        {
            if (inputFileName[dirPos] == '\\')
            {
                directoryLength = dirPos + 1; //Include '\'
            }
            ++dirPos;
        }

        // Copy the path
        if (directoryLength)
        {
            directoryPath = new char[directoryLength + 1];
            memcpy(directoryPath, inputFileName, directoryLength);
            directoryPath[directoryLength] = 0;
        }

        xmlDoc *doc = NULL;
        xmlNode *root_element = NULL;
        std::vector<std::string> fileNames;
        doc = xmlReadMemory(fileBuffer, fileLength, inputFileName, NULL, 0);

        root_element = xmlDocGetRootElement(doc);

        TrackInfo trackInfo;

        print_element_names(root_element, trackInfo);

        xmlFreeDoc(doc);

        xmlCleanupParser();

        // Get each segment in the MPD file
        std::string dirPath = directoryPath;
        std::string catPath;

        catPath = dirPath + trackInfo.initSegment;
        result = nn::fs::OpenFile(&fileHandle, catPath.c_str(), nn::fs::OpenMode_Read);

        if (result.IsFailure())
        {
            NN_LOG("\n Failed to open Segment File %s \n \n Exiting BrowserPlayerSample Test \n", catPath.c_str());
            bailTest = true;
        }
        else
        {
            // Read the Segment file into a buffer
            nn::fs::GetFileSize(&segmentLength, fileHandle);

            if (segmentBuffer)
            {
                delete[] segmentBuffer;
            }
            segmentBuffer = new char[segmentLength];

#if VERBOSE_LOGGING
            NN_LOG("Reading segment file %s\n", file);
            NN_LOG("Reading segment file size %d\n", segmentLength);
#endif

            nn::fs::ReadFile(fileHandle, 0, segmentBuffer, segmentLength);
            nn::fs::CloseFile(fileHandle);

            initializationSegmentSet = true;
            mseSourceBuffer.appendData(segmentBuffer, segmentLength, movie::sample::MseDataType::INITIALIZATION_SEGMENT);
        }

        if (!bailTest)
        {
            for (int i = 0; i < trackInfo.dataSegments.size(); i++)
            {
                if (!bailTest)
                {
                    catPath = dirPath + trackInfo.dataSegments[i];
                    // Open the Data Segment file
                    result = nn::fs::OpenFile(&fileHandle, catPath.c_str(), nn::fs::OpenMode_Read);
                    if (result.IsFailure())
                    {
                        NN_LOG("\n Failed to open Segment File %s \n \n Exiting BrowserPlayerSample Test \n", catPath.c_str());
                        bailTest = true;
                    }
                    else
                    {
                        // Read the Segment file into a buffer
                        nn::fs::GetFileSize(&segmentLength, fileHandle);

                        if (segmentBuffer)
                        {
                            delete[] segmentBuffer;
                        }
                        segmentBuffer = new char[segmentLength];

#if VERBOSE_LOGGING
                        NN_LOG("Reading segment file %s\n", file);
                        NN_LOG("Reading segment file size %d\n", segmentLength);
#endif

                        nn::fs::ReadFile(fileHandle, 0, segmentBuffer, segmentLength);
                        nn::fs::CloseFile(fileHandle);

                        // Just set the first segment as the intialization segment
                        mseSourceBuffer.appendData(segmentBuffer, segmentLength, movie::sample::MseDataType::DATA_SEGMENT);
                    }
                }
            }
        }
        //This is still necessary to handle the case of only having a single segment playlist
        mseSourceBuffer.appendData(0, 0, movie::sample::MseDataType::END_OF_FILE_SEGMENT);
    }

    NN_LOG("\n BrowserPlayerSample:: MSE Source buffer filled\n");

    if (directoryPath)
    {
        delete[] directoryPath;
        directoryPath = nullptr;
    }

    if (file)
    {
        delete[] file;
        file = nullptr;
    }

    if (segmentName)
    {
        delete[] segmentName;
        segmentName = nullptr;
    }

    if (segmentBuffer)
    {
        delete[] segmentBuffer;
        segmentBuffer = nullptr;
    }

    if (fileBuffer)
    {
        delete[] fileBuffer;
        fileBuffer = nullptr;
    }


    ///////////////

    NN_UTIL_SCOPE_EXIT {
        nn::fs::UnmountHostRoot(); // this needs to happen after extractor_'s dtor is called
        delete mediaPlayerObserver;
        nv::FinalizeGraphics();
    };

    std::string path_{inputFileName};
    movie::sample::MseExtractor extractor_;
    mseSourceBuffer.processAppendedData();
    extractor_.AddSourceBuffer(&mseSourceBuffer);

    ///////////////

    if( player != NULL )
    {
        NN_LOG( "\n MediaPlayerMseSample:: Set SetObserver \n" );
        movieStatus = player->SetObserver(mediaPlayerObserver);
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to Set SetObserver: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_LOG("\n MediaPlayerMseSample :: Set Output Buffer Alignment \n");
        movieStatus = player->SetOutputBufferAlignment(&bufferAlignment);
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG("\n MediaPlayerMseSample:: Failed to Set Output Buffer Alignment Error : 0x%x \n", movieStatus);
        }

        NN_LOG( "\n MediaPlayerMseSample :: Set extractor \n" );
        movieStatus = player->SetExtractor(extractor_);
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to Set extractor: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_LOG( "\n MediaPlayerMseSample:: Prepare \n" );
        movieStatus = player->Prepare();
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to Prepare: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_LOG( "\n MediaPlayerMseSample:: Test API: GetPlaybackDuration() + \n");
        int64_t trackDuration = 0;
        movieStatus = player->GetPlaybackDuration(&trackDuration);
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to GetPlaybackDuration: Error : 0x%x \n", movieStatus );
        }

        trackDuration = trackDuration / 1000;
        NN_LOG( "   GetPlaybackDuration (In msec) : %" PRId64 "\n", trackDuration);
        NN_LOG( " MediaPlayerMseSample:: Test API: GetPlaybackDuration() - \n");

        NN_LOG( "\n MediaPlayerMseSample:: Test API: GetTrackCount() + \n");
        int32_t trackCount = 0;
        movieStatus = player->GetTrackCount(&trackCount);
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to GetTrackCount: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }
        NN_LOG( "   GetTrackCount: %d \n" , trackCount);
        NN_LOG( " MediaPlayerMseSample:: Test API: GetTrackCount() - \n");

        NN_LOG( "\n MediaPlayerMseSample:: Test API: GetVideoDimensions() + \n");
        int32_t width = 0;
        int32_t height = 0;
        movieStatus = player->GetVideoDimensions(&width, &height);
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to GetVideoDimensions: Error : 0x%x, Detail: %s\n", movieStatus );
        }
        NN_LOG( "   GetVideoDimensions: width: %d height: %d \n", width, height);
        NN_LOG( " MediaPlayerMseSample:: Test API: GetVideoDimensions() - \n");

        NN_LOG( "\n MediaPlayerMseSample:: Test API: GetTrackType(), GetTrackInfo() + \n");

        for( int32_t track = 0; track < trackCount; track ++)
        {
            movie::TrackType trackType = movie::TrackType_Unknown;
            movie::TrackInfo trackInfo;
            movieStatus = player->GetTrackType(track, &trackType);
            if( movieStatus != movie::Status_Success )
            {
                NN_LOG( "\n MediaPlayerMseSample:: Failed to GetTrackType: Error : 0x%x \n", movieStatus );
            }

            switch (trackType )
            {
            case movie::TrackType_Audio:
                NN_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_Audio \n" , track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if( movieStatus != movie::Status_Success )
                {
                    NN_LOG( "\n MediaPlayerMseSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus );
                }
                else
                {
                    NN_LOG( "\n GetTrackInfo() Track Index : %d Track Type: TrackType_Audio \n" , track);
                    NN_LOG( "                  mime        : %s \n", trackInfo.audioTrackInfo.mime);
                    NN_LOG( "                  sampleRate  : %d \n", trackInfo.audioTrackInfo.sampleRate);
                    NN_LOG( "                  channels    : %d \n", trackInfo.audioTrackInfo.channels);
                    NN_LOG( "                  bitRate     : %d \n", trackInfo.audioTrackInfo.bitRate);
                    NN_LOG( "                  codec       : 0x%x \n", trackInfo.audioTrackInfo.codecType);
                    NN_LOG( "                  container   : 0x%x \n", trackInfo.containerType);
                }
                break;

            case movie::TrackType_Video:
                if (playVideo)
                    isVideoTrack = true;
                NN_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_Video \n" , track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);

                {
                    auto& fps = trackInfo.videoTrackInfo.frameRate;
                    if (fps == 60 || fps == 30 || fps == 20 || fps == 15) {
                        nvnWindowSetPresentInterval(&mediaPlayerObserver->GetNvnVideoRenderer()->GetDevice()->GetWindow(), 60 / fps);
                        NN_LOG("\n MediaPlayerMseSample:: presentation interval set for %d fps\n",fps);
                    }
                }

                if( movieStatus != movie::Status_Success )
                {
                    NN_LOG( "\n MediaPlayerMseSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus );
                }
                else
                {
                    NN_LOG( "\n GetTrackInfo() Track Index : %d Track Type: TrackType_Video \n" , track);
                    NN_LOG( "                  mime        : %s \n", trackInfo.videoTrackInfo.mime);
                    NN_LOG( "                  frameRate   : %d \n", trackInfo.videoTrackInfo.frameRate);
                    NN_LOG( "                  height      : %d \n", trackInfo.videoTrackInfo.height);
                    NN_LOG( "                  width       : %d \n", trackInfo.videoTrackInfo.width);
                    NN_LOG( "                  bitRate     : %d \n", trackInfo.videoTrackInfo.bitRate);
                    NN_LOG( "                  codec       : 0x%x \n", trackInfo.videoTrackInfo.codecType);
                    NN_LOG( "                  container   : 0x%x \n", trackInfo.containerType);
                }
                break;

            case movie::TrackType_Subtitle:
                NN_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_Subtitle \n" , track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if( movieStatus != movie::Status_Success )
                {
                    NN_LOG( "\n MediaPlayerMseSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus );
                }
                else
                {
                    NN_LOG( "\n GetTrackInfo() Track Index : %d Track Type: TrackType_Subtitle \n" , track);
                    NN_LOG( "                  mime        : %s \n", trackInfo.subtitleTrackInfo.mime);
                }
                break;

            case movie::TrackType_ClosedCaption:
                NN_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_ClosedCaption \n" , track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if( movieStatus != movie::Status_Success )
                {
                    NN_LOG( "\n MediaPlayerMseSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus );
                }
                else
                {
                    NN_LOG( "\n GetTrackInfo() Track Index : %d Track Type: TrackType_ClosedCaption \n" , track);
                    NN_LOG( "                  mime        : %s \n", trackInfo.closedCaptionTrackInfo.mime);
                }
                break;

            default:
                NN_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_Unknown \n" , track);
                break;
            }
        }
        NN_LOG( " MediaPlayerMseSample:: Test API: GetTrackType(), GetTrackInfo() - \n");

        /* Play only selected track */
        for( int32_t track = 0; track < trackCount; track ++)
        {
            movie::TrackType trackType = movie::TrackType_Unknown;
            movieStatus = player->GetTrackType(track, &trackType);
            if( movieStatus != movie::Status_Success )
            {
                NN_LOG( "\n MediaPlayerMseSample:: Failed to GetTrackType: Error : 0x%x \n", movieStatus );
            }
            switch (trackType )
            {
            case movie::TrackType_Audio:
                if( playAudio == false )
                {
                    movieStatus = player->DeSelectTrack(track);
                    if( movieStatus != movie::Status_Success )
                    {
                        NN_LOG( "\n MediaPlayerMseSample:: Failed to DeSelectTrack: Error : 0x%x \n", movieStatus );
                    }
                }
                mediaPlayerObserver->SetAudioTrackNumber(track);
                break;

            case movie::TrackType_Video:
                if( playVideo == false )
                {
                    movieStatus = player->DeSelectTrack(track);
                    if( movieStatus != movie::Status_Success )
                    {
                        NN_LOG( "\n MediaPlayerMseSample:: Failed to DeSelectTrack: Error : 0x%x \n", movieStatus );
                    }
                }
                mediaPlayerObserver->SetVideoTrackNumber(track);
                break;

            default:
                break;
            }
        }

        /* Set loop mode */
        if( setLooping == true )
        {
            movieStatus = player->SetLooping(setLooping);
            if( movieStatus != movie::Status_Success )
            {
                NN_LOG( "\n MediaPlayerMseSample:: Failed to SetLooping: Error : 0x%x \n", movieStatus );
            }
        }

        NN_LOG( "\n MediaPlayerMseSample:: setting playback rate to %f\n", playRate);
        player->SetPlaybackRate(playRate);

        movieStatus = player->SetVolume(volume);
        if( movieStatus != movie::Status_Success )
            NN_LOG("\n MediaPlayerMseSample::SetVolume Error = 0x%x \n", movieStatus);

        float vol = 1.0;
        movieStatus = player->GetVolume(&vol);
        if( movieStatus != movie::Status_Success )
            NN_LOG("\n MediaPlayerMseSample::GetVolume Error = 0x%x \n", movieStatus);
        NN_LOG("\n MediaPlayerMseSample:: Volume = %f \n", vol);

        NN_LOG( "\n MediaPlayerMseSample:: start \n" );
        movieStatus = player->Start();
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to Start: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }
        if( isVideoTrack )
        {
            nn::os::WaitEvent(&mediaPlayerObserver->m_StartPlaybackEvent);
            nn::os::ClearEvent(&mediaPlayerObserver->m_StartPlaybackEvent);
            player->GetVideoDimensions(&width, &height);
            if( ( config.returnVideoDataToClient == true ) && ( config.useClientMemoryForOutput == true ) )
            {
                size_t videoOutputBufferSize = width * height;
                size_t extraForAlignment = 1024 * 4;
                if( config.videoFormat == movie::OutputFormat_VideoColorAbgr )
                {
                    videoOutputBufferSize = videoOutputBufferSize * 4 + extraForAlignment;
                }
                else if( config.videoFormat == movie::OutputFormat_VideoColorNv12 )
                {
                    videoOutputBufferSize = videoOutputBufferSize + ( 2 * ( ( width / 2 ) * ( height / 2 ) ) ) + extraForAlignment;
                }

                mediaPlayerObserver->m_VideoOutputBufferSize = videoOutputBufferSize;
            }

        }

        gettimeofday(&playerTimeVal, NULL);
        startTime = (int64_t)playerTimeVal.tv_sec * 1000 + (int64_t)playerTimeVal.tv_usec / 1000;

        if( playAudio == false && playVideo == false )
        {
            mediaPlayerObserver->SetPlaybackComplete();
        }

        int64_t lastPlaybackPosition = 0;
        int64_t getPlaybackPosition = 0;

        while ( !mediaPlayerObserver->IsPlaybackComplete() )
        {

#ifdef ENABLE_PROFILER
            nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main);
#endif
            getPlaybackPosition = 0;
            movieStatus = player->GetCurrentPlaybackPosition(&getPlaybackPosition);
            if( movieStatus != movie::Status_Success )
            {
                NN_LOG( "\n MediaPlayerMseSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
            }
            getPlaybackPosition = getPlaybackPosition / (1000 * 1000);
            if( ( (getPlaybackPosition % 5 ) == 0ll) && ( lastPlaybackPosition != getPlaybackPosition))
            {
                NN_LOG( "Track Duration : %" PRId64 " Seconds ", trackDuration / 1000);
                NN_LOG( " Playback Position : %" PRId64 " Seconds\n", getPlaybackPosition);
                lastPlaybackPosition = getPlaybackPosition;
            }

            hidHandler.Update();

            if( isVideoTrack )
            {
                int yOffset = 0;
                int uvOffset = 0;
                int ystride = 0;
                int nativeWidth = 0;
                int nativeHeight = 0;
                int cropWidth = 0;
                int cropHeight = 0;
                int colorSpace = 0;
                int32_t bufferIndex = -1;
                int32_t trackNumber = -1;
                int64_t presentationTimeUs = 0;
                uint32_t flags = 0;
                int32_t bufferIndexForProperties = -1;

                if( ( config.returnVideoDataToClient == true ) && ( config.useClientMemoryForOutput == true ) )
                {
                    nn::os::LockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );
                    uint32_t videoListSize = mediaPlayerObserver->m_VideoOutputBuffersList.size();

                    if( videoListSize > 0 )
                    {
                        bool videoAvailableForRendering = false;
                        MediaPlayerObserver::OutputBufferInfo videoBuffer;

                        videoBuffer = mediaPlayerObserver->m_VideoOutputBuffersList.front();
                        mediaPlayerObserver->m_VideoOutputBuffersList.erase( mediaPlayerObserver->m_VideoOutputBuffersList.begin());

                        nn::os::UnlockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );

                        bufferIndex = videoBuffer.bufferIndex;
                        trackNumber = videoBuffer.trackNumber;

                        if( config.videoDecodeMode == movie::VideoDecodeMode_Cpu )
                        {
                            bufferIndexForProperties = bufferIndex;
                        }
                        else
                        {
                            //make sure the BufferIndex is one of the valid ones
                            for( int i = 0; i < mediaPlayerObserver->m_videoBufferCount; ++i )
                            {
                                if( bufferIndex == mediaPlayerObserver->m_videoBufferIndexArray[ i ] )
                                {
                                    bufferIndexForProperties = bufferIndex;
                                    mediaPlayerObserver->m_PresentationIndex = bufferIndex;
                                    videoAvailableForRendering = true;
                                    break;
                                }
                            }
                        }

                        player->GetOutputBufferProperties(bufferIndexForProperties, &mediaPlayerObserver->m_BufferProperty);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-y-stride", &ystride);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-y-offset", &yOffset);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-uv-offset", &uvOffset);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("width", &nativeWidth);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("height", &nativeHeight);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("crop-width", &cropWidth);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("crop-height", &cropHeight);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-colorspace", &colorSpace);
                        if( mediaPlayerObserver->m_CropWidth != cropWidth || mediaPlayerObserver->m_CropHeight != cropHeight )
                        {
                            mediaPlayerObserver->m_CropWidth = cropWidth;
                            mediaPlayerObserver->m_CropHeight = cropHeight;
                            if( config.videoFormat == movie::OutputFormat_VideoColorNv12 )
                            {
                                NN_LOG("Video resolution changed, ResizeTextures : WxH %dx%d ystride %d\n", cropWidth, cropHeight, ystride);
                                if( ( cropWidth > 0 ) && ( cropHeight > 0 ) )
                                {
                                    mediaPlayerObserver->ResizeTextures(cropWidth, cropHeight);
                                }
                            }
                        }

                        if( config.videoDecodeMode == movie::VideoDecodeMode_Cpu )
                        {
                            movie::Status movieStatus = player->FillOutputBuffer(trackNumber, bufferIndex, &(mediaPlayerObserver->m_videobufferArray[0]), &presentationTimeUs, &flags);
                            if( movieStatus == movie::Status_Success )
                            {
                                videoAvailableForRendering = true;
                            }

                            uvOffset = nn::util::align_up(uvOffset, nv12UvAlignment);   // If "nv12-uv-alignment" is specified in the
                        }                                                               // buffer properties then uvOffset needs to be
                                                                                        // up-aligned accordingly.
                        if( videoAvailableForRendering )
                        {
                            mediaPlayerObserver->DrawVideoNvn(mediaPlayerObserver->m_PresentationIndex, cropWidth, cropHeight, yOffset, uvOffset, ystride, colorSpace);
                        }
                    }
                    else
                    {
                        nn::os::UnlockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );
                    }
                }
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }

        NN_LOG( "\n MediaPlayerMseSample:: Playback is complete! \n" );

        NN_LOG( "\n MediaPlayerMseSample:: Playback Stop \n" );
        movieStatus = player->Stop();
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to Stop: Error : 0x%x \n", movieStatus );
        }
        gettimeofday(&playerTimeVal, NULL);
        endTime = (int64_t)playerTimeVal.tv_sec * 1000 + (int64_t)playerTimeVal.tv_usec / 1000;

        NN_LOG(" \n Track Duration : %" PRId64 " msec ******* Total Playback Time : %" PRId64 " msec", trackDuration , (endTime - startTime));

        player->Reset();
        movie::Player::Destroy(player);
        //mediaPlayerObserver->DestroyNvn();
        NN_LOG( "\n MediaPlayerMseSample:: Player destroyed \n" );
    }
    else
    {
        NN_LOG( "\n MediaPlayerMseSample:: Could not create player \n" );
    }

Exit_PlayerSample:

    if( errorCleanUp == true )
    {
        movieStatus = player->Stop();
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG( "\n MediaPlayerMseSample:: Failed to Stop: Error : 0x%x \n", movieStatus );
            player->Reset();
        }
        movie::Player::Destroy(player);
        NN_LOG( "\n MediaPlayerMseSample:: Player destroyed \n" );
    }

    mediaPlayerObserver->Finalize(&config);

    } // End of embedded scope

    if (USE_HEAP_TRACKING) {
        NN_LOG("mm heap\n");
        MMHeap().OutputUsage();
        NN_LOG("core heap\n");
        CoreHeap().OutputUsage();
        NN_LOG("malloc heap\n");
        MallocHeap().OutputUsage();
        NN_LOG("new heap\n");
        NewHeap().OutputUsage();
    }

} //NOLINT(impl/function_size)
