﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

#include <upnp/upnp.h>
#include <upnp/upnpdebug.h>
#include <upnp/upnptools.h>
#include <upnp/nnlibupnp_VirtualDir.h>

#include <cstdlib>

#include "UpnpLanDevice.h"

/**
 * @examplesource{UpnpControlRouter.cpp,PageSampleUpnpControlRouter}
 *
 * @brief
 *  Use libupnp to control a network router and forward a TCP port
 */

/**
 * @page PageSampleUpnpControlRouter UpnpControlRouter
 * @tableofcontents
 *
 * @brief
 *  Documentation about the Sample program which uses libupnp to control a network router and forward a TCP port
 *
 * @section PageSampleUpnpControlRouter_SectionBrief Overview
 *  First this sample initializes network connectivity by using the NIFM library.  Then the libupnp library is
 *  initialized.  The sample will then search the local LAN using libupnp for network routers that are
 *  advertising themselves as supporting the UPnP deviceType "urn:schemas-upnp-org:device:InternetGatewayDevice:1.",
 *  or the InternetGatewayDevice (IGW).  The first network device found on the LAN that supports the InternetGatewayDevice
 *  (IGW) deviceTpe will be selected by the sample.  The sample will then invoke the AddPortMapping(),
 *  GetSpecificPortMappingEntry(), and DeletePortMapping() functions to add, display and remove a "forwarded" TCP
 *  port from the network device.
 *
 * @section UpnpControlRouter_SectionFileStructure File Structure
 *  This sample program can be found at
 * @link ../../../Samples/Sources/Applications/Libupnp/UpnpControlRouter Samples/Sources/Applications/Libupnp/UpnpControlRouter @endlink.
 *
 * @section PageSampleUpnpControlRouter_SectionNecessaryEnvironment System Requirements
 *  It is required to import network setting by SettingManager prior to running this sample.
 *  Please refer @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} for further information.
 *
 *  It is required that the network router device on the LAN that the sample will connect to supports UPnP configuration
 *  changes. The option "Allow users to make Configuration Changes using UPnP" should be enabled/on in the network router.
 *
 *  By default this sample will forward all network traffic recieved by the selected network router for TCP port "9000"
 *  to the sample application's IP address to TCP port "22000".  If a different TCP port assignment is desired it can be
 *  tuned in the settings below.  NOTE: This sample application does not monitor or use TCP port "22000" so any network
 *  data received on that port is assumed to be discarded.
 *
 * @section PageSampleUpnpControlRouter_SectionHowToOperate Operation Procedure
 *  In order for the sample to communicate with a network router on the LAN the network router will need to have UPnP
 *  "enabled" under its setup.  See the network router manufactures setup instructions for how to enable UPnP support
 *  in the router.
 *
 *  NOTE: By default the sample will use the Nifm negotiated network interface to send and receive libupnp generated
 *  multicast network traffic.  If the sample is running on a host that supports multiple network interfaces the
 *  desired network interface to use can be selected by passing the command line argument "-interface <Ipv4 Address>",
 *  where <Ipv4 Address> is assigned to the host network interface that the sample should use.

 * @section PageSampleUpnpControlRouter_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleUpnpControlRouter_SectionHowToExecute Execution Procedure
 *  Build the Visual Studio Solution in the desired configuration and run it.
 *
 * @section PageSampleUpnpControlRouter_SectionDetail Detail
 *  Here is the sequence of this sample program.
 *
 *  - Sends a request to use the network to the NIFM library
 *  - Initializes the socket library
 *  - The libupnp library is initialized with the host network interface to use for sending and receiving multicast traffic
 *  - Initializes a "virtual filesystem" to support the libupnp "web server".
 *    Enables the libupnp "web server" functionality.
 *  - Joins the UPnP multicast group (ip=255.255.255.250, UDP port=1900) for the local LAN and searches for UPnP devices
 *    which support the deviceType "urn:schemas-upnp-org:device:InternetGatewayDevice:1." (IGW)
 *  - If a network router is found then it will be used by this sample to enable a network "port forward" of TCP port 9000
 *    to the IP address of the sample and port "22000".
 *  - If 30 seconds elapse, and no network router supporting deviceType is found, then the sample will exit, indicating
 *    that no supported UPnP network routers were found on the LAN.
 *  - Sample will call AddPortMapping() to add/create a TCP port mapping in the network router.
 *  - Sample will call GetSpecificPortMappingEntry() to display the TCP port mapping setting in the network router.
 *  - Sample will call DeletePortMapping() to delete the TCP port mapping setting in the network router.
 *  - Sample exits.
 *
 *  Please enable "#define DETAILED_LIBUPNP_DEBUGGING" if you are interested in seeing debugging output from libupnp.
 *  The libupnp library supports the displaying of detailed debugging information to NN_LOG.  By default only CRITICAL
 *  error information from the libupnp library is displayed.
 *
 *  Please enable "#define KEEP_PORT_MAPPING" if you are interested in keeping the network router port forwarding set
 *  by the sample in the network router.  Otherwise the sample will delete the libupnp set port forwarding from the
 *  network router when the sample program terminates.
 *
 *  Please enable "#define PRINT_ADVERTISED_SERVICE" if you are interested in seeing what UPnP services were discovered on
 *  the LAN.  As libupnp searches the local LAN and discovers UPnP capable devices it can display what devices were
 *  found/discovered via an NN_LOG if defined.
 *
 *  Please "#define ROUTER_TCP_PORT" to a legal TCP port value if you would like to forward a TCP port from the network
 *  router other than TCP port 9000.  By default this sample will forward TCP port 9000 from the network router to the
 *  sample.
 *
 *  Please "#define LOCAL_TCP_PORT" to a legal TCP port value if you would like to receive network traffic on the sample
 *  using a TCP port other than 22000.  By default this sample will receive network traffic from the network router on
 *  TCP port 22000.  Please note that this sample does not actually process any received network traffic from the network
 *  router.
 *
 * @subsection PageSampleUpnpControlRouter_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  UpnpControlRouter.cpp
 *  @includelineno UpnpControlRouter.cpp

 * @subsection PageUpnpControlRouter_SectionExpectedOutput Expected Output
 * @verbinclude UpnpControlRouter_OutputExample.txt
 */

// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
// #define KEEP_PORT_MAPPING
// #define DETAILED_LIBUPNP_DEBUGGING
// #define PRINT_ADVERTISED_SERVICE
// #define ROUTER_TCP_PORT "9000"
// #define LOCAL_TCP_PORT "22000"

#ifndef ROUTER_TCP_PORT
#define ROUTER_TCP_PORT "9000"
#endif

#ifndef LOCAL_TCP_PORT
#define LOCAL_TCP_PORT "22000"
#endif

namespace nns {
namespace libupnp {

// Constants
const int                           TimeoutOnNicConfigurationInSec = 15;
const size_t                        ThreadStackSize = 8192;
const uint16_t                      UpnpServerPort = 49152;

// Socket memory
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

// Thread Memory
NN_OS_ALIGNAS_THREAD_STACK char     g_ThreadStack[ ThreadStackSize ];

// Nifm Values
struct in_addr                      g_NifmIpv4Addr = { 0 };
struct in_addr                      g_NifmIpv4Mask = { 0 };
struct in_addr                      g_NifmIpv4Gateway = { 0 };
struct in_addr                      g_NifmIpv4Dns1 = { 0 };
struct in_addr                      g_NifmIpv4Dns2 = { 0 };
bool                                g_IsNetworkInitialized = false;

// UPNP Values
struct UpnpVirtualDirCallbacks      g_Callbacks;
nn::libupnp::VirtualDir             g_VirtualDirectory;
UpnpIGWDevices                      g_UpnpIGWDevices;
UpnpClient_Handle                   g_UpnpCtrlptHandle = -1;
nn::os::MutexType                   g_UpnpAdvertiseMutex;
bool                                g_IsUpnpInitialized = false;
bool                                g_IsUpnpClientRegistered = false;
bool                                g_IsUpnpSyncFlag = true;

void (*g_UpnpSOAPActionCallback)(struct Upnp_Action_Complete* pEvent);

// ------------------------------------------------------------------------------------------------
// Setup network interface
// ------------------------------------------------------------------------------------------------
int SetupNetwork() NN_NOEXCEPT
{
    int            ret = 0;
    nn::Result     result = nn::ResultSuccess();
    uint32_t       timeoutSec = 0;
    bool           isSocketLibInitialized = false;
    bool           isNifmLibInitialized = false;

    do
    {
        // If already initialized
        if (g_IsNetworkInitialized == true)
        {
            NN_LOG("Network is already initialized!\n");
            break;
        }

        // Initialize NIFM
        result = nn::nifm::Initialize();
        if(result.IsFailure())
        {
            NN_LOG("Failed calling: 'nn::nifm::Initialize()'\n");
            ret = -1;
            break;
        }

        // Submit the NIFM Network Request
        nn::nifm::SubmitNetworkRequest();

        // Wait for NIFM to go online
        while(nn::nifm::IsNetworkRequestOnHold())
        {
            NN_LOG("Waiting for network interface availability...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            if(timeoutSec++ > TimeoutOnNicConfigurationInSec)
            {
                NN_LOG("Failed to setup NIFM network interface.\n");
                ret = -1;
                break;
            }
        }

        // Did NIFM request timeout?
        if (ret < 0)
        {
            break;  // Yes
        }

        // Is the NIFM Network Available?
        if(nn::nifm::IsNetworkAvailable() == 0)
        {
            NN_LOG("Network is not available.\n");
            ret = -1;
            break;
        }
        isNifmLibInitialized = true;

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &g_NifmIpv4Addr, &g_NifmIpv4Mask, &g_NifmIpv4Gateway, &g_NifmIpv4Dns1, &g_NifmIpv4Dns2 );

        NN_LOG( "Network Interface is UP!\n" );
        NN_LOG( "NIFM Address Ipv4      : %s\n", nn::socket::InetNtoa( g_NifmIpv4Addr ));
        NN_LOG( "NIFM Network Mask Ipv4 : %s\n", nn::socket::InetNtoa( g_NifmIpv4Mask ));
        NN_LOG( "NIFM Gateway Ipv4      : %s\n", nn::socket::InetNtoa( g_NifmIpv4Gateway));
        NN_LOG( "NIFM DNS1 Ipv4         : %s\n", nn::socket::InetNtoa( g_NifmIpv4Dns1 ));
        NN_LOG( "NIFM DNS2 Ipv4         : %s\n", nn::socket::InetNtoa( g_NifmIpv4Dns2 ));

        // Initialize the socket library
        result = nn::socket::Initialize(g_SocketConfigWithMemory);
        if(result.IsFailure())
        {
            NN_LOG("Failed to initialize socket library.\n");
            ret = -1;
            break;
        }
        isSocketLibInitialized = true;

        // Successfully initialized the network
        g_IsNetworkInitialized = true;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    // If failed - disable network
    if (ret != 0)
    {
        if (isSocketLibInitialized == true)
        {
            nn::socket::Finalize();
            isSocketLibInitialized  = false;
            ret = -1;
        }

        if (isNifmLibInitialized == true)
        {
            nn::nifm::CancelNetworkRequest();
            isNifmLibInitialized = false;
            ret = -1;
        }

        g_IsNetworkInitialized = false;
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Finalize Network
// ------------------------------------------------------------------------------------------------
int FinalizeNetwork() NN_NOEXCEPT
{
    nn::Result     result = nn::ResultSuccess();
    int            ret = 0;

    if (g_IsNetworkInitialized == true)
    {
        result = nn::socket::Finalize();
        if (result.IsFailure())
        {
            NN_LOG("Failed calling 'nn::socket::Finalize()'!\n");
        }

        nn::nifm::CancelNetworkRequest();
        g_IsNetworkInitialized = false;
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// UPNP: Event Callback
// ------------------------------------------------------------------------------------------------
int UpnpCallbackEventHandler(Upnp_EventType eventType, void* pEvent, void* pCookie) NN_NOEXCEPT
{
    switch (eventType)
    {
    case UPNP_DISCOVERY_ADVERTISEMENT_ALIVE:
    case UPNP_DISCOVERY_SEARCH_RESULT:
         {
             struct Upnp_Discovery* pDiscover = reinterpret_cast<struct Upnp_Discovery*>(pEvent);
             IXML_Document*         pDescDoc = nullptr;
             int                    ret = -1;
             bool                   isIGWDevice = false;

             nn::os::LockMutex(&g_UpnpAdvertiseMutex);

             do
             {
                 // If map is already populated..
                 if (g_UpnpIGWDevices.GetMapSize() > 0)
                 {
                     break;
                 }

                 // Skip Devkits for now
                 if ( (nn::util::Strnlen(pDiscover->DeviceId, 11) == 11)     &&
                      (nn::util::Strncmp(pDiscover->DeviceId, "uuid:DevKit", 11) == 0)  )
                 {
                     break;
                 }

#ifdef PRINT_ADVERTISED_SERVICE
                 UpnpPrintDiscoveryStructure(pDiscover);
#endif

                 if (pDiscover->ErrCode != UPNP_E_SUCCESS)
                 {
                     NN_LOG( "Error in Discovery Callback -- %d/%s\n",
                                      pDiscover->ErrCode, UpnpGetErrorMessage(pDiscover->ErrCode));
                     break;
                 }

                 ret = UpnpDownloadXmlDoc(pDiscover->Location, &pDescDoc);
                 if (ret != UPNP_E_SUCCESS)
                 {
                     NN_LOG( "Error obtaining device description from [%s] -- error = %d/%s\n",
                                 pDiscover->Location, ret, UpnpGetErrorMessage(ret));
                     break;
                 }

                 ret = g_UpnpIGWDevices.AddDevice(pDescDoc, pDiscover, &isIGWDevice);
                 if (ret != 0)
                 {
                     NN_LOG( "Failed calling: UpnpAddDevice -- error = [%d]/[%s]\n",
                                 ret, UpnpGetErrorMessage(ret));
                     break;
                 }

                 // All done
                 break;

             } while (NN_STATIC_CONDITION(false));

             nn::os::UnlockMutex(&g_UpnpAdvertiseMutex);

             if (pDescDoc != nullptr)
             {
                 ixmlDocument_free(pDescDoc);
                 pDescDoc = nullptr;
             }
         }
         break;

    case UPNP_DISCOVERY_SEARCH_TIMEOUT:
        /* Nothing to do here... */
        break;

    case UPNP_DISCOVERY_ADVERTISEMENT_BYEBYE:
         /* Nothing to do here... */
         break;

    /* SOAP Stuff */
    case UPNP_CONTROL_ACTION_COMPLETE:
         {
             struct Upnp_Action_Complete* pActionCompleteEvent = reinterpret_cast<struct Upnp_Action_Complete*>(pEvent);

             g_UpnpSOAPActionCallback(pActionCompleteEvent);
         }
         break;

    /* Not an Event we process */
    default:
        break;
    }

    return 0;
}

#ifndef KEEP_PORT_MAPPING
// ------------------------------------------------------------------------------------------------
// DeletePortMappingCallBack()
//
// ------------------------------------------------------------------------------------------------
void DeletePortMappingCallBack(struct Upnp_Action_Complete* pEvent) NN_NOEXCEPT
{
    if (pEvent->ErrCode != UPNP_E_SUCCESS)
    {
        NN_LOG( "DeletePortMappingCallBack():  Error in Action Complete Callback -- %d/%s\n",
                              pEvent->ErrCode, UpnpGetErrorMessage(pEvent->ErrCode));
        return;
    }

    NN_LOG("DeletePortMappingCallBack(): Action Completed Successfully\n");

    g_IsUpnpSyncFlag = false;

    return;
}

// ------------------------------------------------------------------------------------------------
// DeletePortMapping:  [xmlns:urn:schemas-upnp-org:service:WANIPConnection:1]
//
// ------------------------------------------------------------------------------------------------
int DeletePortMapping(UpnpLanDevice& lanDevice) NN_NOEXCEPT
{
    IXML_Document*   pActionNode = nullptr;
    const char*      pParamNames[]  = { "NewRemoteHost", "NewExternalPort", "NewProtocol" };
    const char*      pParamValues[] = { nullptr, ROUTER_TCP_PORT, "TCP" };
    int              idx = 0;
    int              ret = 0;

    do
    {
        for(idx = 0; idx < 3; idx++)
        {
            ret = UpnpAddToAction( &pActionNode, "DeletePortMapping", g_UpnpIGWDevices.IGWWANIPConnection,
                                   pParamNames[idx], pParamValues[idx]);

            if (ret != UPNP_E_SUCCESS)
            {
                NN_LOG( "Failed to UpnpAddToAction() - Name: [%s], Value [%s] - Error: [%s]\n",
                                        pParamNames[idx], pParamValues[idx], UpnpGetErrorMessage(ret));
                break;
            }
        }

        if (ret != UPNP_E_SUCCESS)
        {
            break;
        }

        ret = UpnpSendActionAsync(g_UpnpCtrlptHandle, lanDevice.m_WanControlUrl, g_UpnpIGWDevices.IGWWANIPConnection,
                                  nullptr, pActionNode, UpnpCallbackEventHandler, nullptr);

        if (ret != UPNP_E_SUCCESS)
        {
            NN_LOG( "Failed UpnpSendActionAsync() - Error: [%s]\n", UpnpGetErrorMessage(ret));
            break;
        }

        NN_LOG( "Successfully sent DeletePortMapping!\n" );

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    return(ret);
}
#endif

// ------------------------------------------------------------------------------------------------
// GetSpecificPortMappingEntryCallBack() - [xmlns:urn:schemas-upnp-org:service:WANIPConnection:1]
//
// ------------------------------------------------------------------------------------------------
void GetSpecificPortMappingEntryCallBack(struct Upnp_Action_Complete* pEvent) NN_NOEXCEPT
{
    int           ret = 0;
    int           idx = 0;
    const char*   pParamNames[]  = { "NewRemoteHost", "NewExternalPort", "NewProtocol", "NewInternalPort",
                                     "NewInternalClient", "NewEnabled", "NewPortMappingDescription", "NewLeaseDuration" };
    const char*   pParamValues[] = { nullptr, ROUTER_TCP_PORT, "TCP" };
    char          value[200];

    if (pEvent->ErrCode != UPNP_E_SUCCESS)
    {
        NN_LOG( "GetSpecificPortMappingEntry: Error in Action Complete Callback -- %d/%s\n",
                              pEvent->ErrCode, UpnpGetErrorMessage(pEvent->ErrCode));
        return;
    }

    NN_LOG( "*************************************************************\n" );
    NN_LOG( "GetSpecificPortMappingEntry - Action Completeted Successfully\n" );

    for(idx = 0; idx < 8; idx++)
    {
        if (idx < 3)
        {
            memset(value, 0, sizeof(value));
            if (pParamValues[idx] != nullptr)
            {
                strcpy(value, pParamValues[idx]);
            }
        }
        else
        {
            ret = g_UpnpIGWDevices.GetFirstDocumentItem(value, sizeof(value), pEvent->ActionResult, pParamNames[idx]);
            if (ret != 0)
            {
                NN_LOG("Failed to get 'NewInternalPort' from ActionResult\n" );
                return;
            }
        }

        NN_LOG( "** Name: [%s], Value [%s]\n", pParamNames[idx], value);
    }
    NN_LOG( "*************************************************************\n" );

    g_IsUpnpSyncFlag = false;

    return;
}

// ------------------------------------------------------------------------------------------------
// GetSpecificPortMappingEntry:  [xmlns:urn:schemas-upnp-org:service:WANIPConnection:1]
//
// ------------------------------------------------------------------------------------------------
int GetSpecificPortMappingEntry(UpnpLanDevice& lanDevice) NN_NOEXCEPT
{
    IXML_Document*  pActionNode = nullptr;
    const char*     pParamNames[]  = { "NewRemoteHost", "NewExternalPort", "NewProtocol" };
    const char*     pParamValues[] = { nullptr, ROUTER_TCP_PORT, "TCP" };
    int             idx = 0;
    int             ret = 0;

    do
    {
        for(idx = 0; idx < 3; idx++)
        {
            ret = UpnpAddToAction( &pActionNode, "GetSpecificPortMappingEntry", g_UpnpIGWDevices.IGWWANIPConnection,
                                   pParamNames[idx], pParamValues[idx]);

            if (ret != UPNP_E_SUCCESS)
            {
                NN_LOG( "Failed to UpnpAddToAction() - Name: [%s], Value [%s] - Error: %d/%s\n",
                                        pParamNames[idx], pParamValues[idx], ret, UpnpGetErrorMessage(ret));
                break;
            }
        }

        if (ret != UPNP_E_SUCCESS)
        {
            break;
        }

        ret = UpnpSendActionAsync(g_UpnpCtrlptHandle, lanDevice.m_WanControlUrl, g_UpnpIGWDevices.IGWWANIPConnection,
                                  nullptr, pActionNode, UpnpCallbackEventHandler, nullptr);

        if (ret != UPNP_E_SUCCESS)
        {
            NN_LOG( "Failed UpnpSendActionAsync() - Error: %d/%s\n", ret, UpnpGetErrorMessage(ret));
            break;
        }

        NN_LOG( "Successfully sent GetSpecificPortMappingEntry!\n" );

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// AddPortMappingCallBack()
//
// ------------------------------------------------------------------------------------------------
void AddPortMappingCallBack(struct Upnp_Action_Complete* pEvent) NN_NOEXCEPT
{
    if (pEvent->ErrCode != UPNP_E_SUCCESS)
    {
        NN_LOG( "AddPortMappingCallBack():  Error in Action Complete Callback -- %d/%s\n",
                              pEvent->ErrCode, UpnpGetErrorMessage(pEvent->ErrCode));
        return;
    }

    NN_LOG("AddPortMappingCallBack(): Action Completed Successfully\n");

    g_IsUpnpSyncFlag = false;

    return;
}

// ------------------------------------------------------------------------------------------------
// AddPortMapping:  [xmlns:urn:schemas-upnp-org:service:WANIPConnection:1]
//
// ------------------------------------------------------------------------------------------------
int AddPortMapping(UpnpLanDevice& lanDevice) NN_NOEXCEPT
{
    IXML_Document*   pActionNode = nullptr;
    const char*      pParamNames[]  = { "NewRemoteHost", "NewExternalPort", "NewProtocol", "NewInternalPort",
                                        "NewInternalClient", "NewEnabled", "NewPortMappingDescription", "NewLeaseDuration" };
    const char*      pParamValues[] = { nullptr, ROUTER_TCP_PORT, "TCP", LOCAL_TCP_PORT, nullptr, "1", "example", "0" };
    char*            pActualValue = nullptr;
    int              idx = 0;
    int              ret = 0;

    do
    {
        for(idx = 0; idx < 8; idx++)
        {
            if (idx == 4)
            {
                pActualValue = nn::socket::InetNtoa(g_NifmIpv4Addr);
            }
            else
            {
                pActualValue = (char *) pParamValues[idx];
            }

            ret = UpnpAddToAction( &pActionNode, "AddPortMapping", g_UpnpIGWDevices.IGWWANIPConnection,
                                   pParamNames[idx], pActualValue);

            if (ret != UPNP_E_SUCCESS)
            {
                NN_LOG( "Failed to UpnpAddToAction() - Name: [%s], Value [%s] - Error: [%s]\n",
                                      pParamNames[idx], pActualValue, UpnpGetErrorMessage(ret));
                break;
            }
        }

        if (ret != UPNP_E_SUCCESS)
        {
            break;
        }

        ret = UpnpSendActionAsync(g_UpnpCtrlptHandle, lanDevice.m_WanControlUrl, g_UpnpIGWDevices.IGWWANIPConnection,
                                  nullptr, pActionNode, UpnpCallbackEventHandler, nullptr);

        if (ret != UPNP_E_SUCCESS)
        {
            NN_LOG( "Failed UpnpSendActionAsync() - Error: %d/%s\n", ret,  UpnpGetErrorMessage(ret));
            break;
        }

        NN_LOG( "Successfully sent AddPortMapping!\n" );

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    return(ret);
}

#ifdef PRINT_ADVERTISED_SERVICE
// ------------------------------------------------------------------------------------------------
// UPNP: Print Discovery Structure
// ------------------------------------------------------------------------------------------------
void UpnpPrintDiscoveryStructure(struct Upnp_Discovery* pDiscover) NN_NOEXCEPT
{
    NN_LOG("ErrCode     =  %s(%d)\n", UpnpGetErrorMessage(pDiscover->ErrCode), pDiscover->ErrCode);
    NN_LOG("Expires     =  %d\n",  pDiscover->Expires);
    NN_LOG("DeviceId    =  %s\n",  pDiscover->DeviceId);
    NN_LOG("DeviceType  =  %s\n",  pDiscover->DeviceType);
    NN_LOG("ServiceType =  %s\n",  pDiscover->ServiceType);
    NN_LOG("ServiceVer  =  %s\n",  pDiscover->ServiceVer);
    NN_LOG("Location    =  %s\n",  pDiscover->Location);
    NN_LOG("OS          =  %s\n",  pDiscover->Os);
    NN_LOG("Ext         =  %s\n",  pDiscover->Ext);
}
#endif


// ------------------------------------------------------------------------------------------------
// Device Command Loop
// ------------------------------------------------------------------------------------------------
void CommandLoop(void* args) NN_NOEXCEPT
{
    UpnpLanDevice   lanDevice;
    int             idx = 0;
    int             ret = 0;;

    NN_LOG( "CommandLoop - Begin!\n" );

    do
    {
        NN_LOG ("Searching for UPnP Devices matching to DeviceType: [%s]\n", g_UpnpIGWDevices.IGWDeviceType);

        for(idx = 0; idx < 30; idx++)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

            ret = g_UpnpIGWDevices.GetDevice(lanDevice);
            if (ret < 0)
            {
                continue;
            }

            NN_LOG( "Match found for UPnP DeviceType: [%s]\n", g_UpnpIGWDevices.IGWDeviceType);
            break;
        }

        if (idx == 30)
        {
            NN_LOG( "Sample Program [TIMED_OUT] after [30] seconds\n" );
            NN_LOG( "Sample program did not UPnP Discover any network router devices supporting DeviceType: [%s] attached to the local LAN: [%s]\n",
                                         g_UpnpIGWDevices.IGWDeviceType, nn::socket::InetNtoa(g_NifmIpv4Addr));
            NN_LOG( "Please make sure UPnP support is (ENABLED) on the setup screen of the network router you wish to run the sample with.\n" );
            NN_LOG( "The network router on this LAN (might) be located at IP Address: [%s]\n", nn::socket::InetNtoa(g_NifmIpv4Gateway));

            break;
        }

        // Add the Port Mapping to the Router
        g_UpnpSOAPActionCallback = AddPortMappingCallBack;
        g_IsUpnpSyncFlag = true;

        ret = AddPortMapping(lanDevice);
        if (ret != 0)
        {
            NN_LOG( "Failed calling: AddPortMapping()\n" );
            break;
        }

        while(g_IsUpnpSyncFlag == true)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            NN_LOG( "Waiting for AddPortMapping response!\n" );
        }

        // Display the Port Mapping currently (Set) in the Router
        g_UpnpSOAPActionCallback = GetSpecificPortMappingEntryCallBack;
        g_IsUpnpSyncFlag = true;

        ret = GetSpecificPortMappingEntry(lanDevice);
        if (ret != 0)
        {
            NN_LOG( "Failed calling: GetSpecificPortMappingEntry()\n" );
            break;
        }

        while(g_IsUpnpSyncFlag == true)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            NN_LOG( "Waiting for GetSpecificPortMappingEntry response!\n" );
        }

#ifndef KEEP_PORT_MAPPING
        // Delete the Port Mapping in the Router
        g_UpnpSOAPActionCallback = DeletePortMappingCallBack;
        g_IsUpnpSyncFlag = true;

        ret = DeletePortMapping(lanDevice);
        if (ret != 0)
        {
            NN_LOG( "Failed calling: DeletePortMapping()\n" );
            break;
        }

        while(g_IsUpnpSyncFlag == true)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            NN_LOG( "Waiting for DeletePortMapping response!\n" );
        }
#endif

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG( "CommandLoop - End!\n" );

    return;
}

// ------------------------------------------------------------------------------------------------
// Create libupnp virtual filsystem
// ------------------------------------------------------------------------------------------------
int MakeVirtualFileSystem() NN_NOEXCEPT
{
    int  rc = 0;

    g_Callbacks.get_info = &g_VirtualDirectory.GetInfo;
    g_Callbacks.open     = &g_VirtualDirectory.Open;
    g_Callbacks.read     = &g_VirtualDirectory.Read;
    g_Callbacks.write    = &g_VirtualDirectory.Write;
    g_Callbacks.seek     = &g_VirtualDirectory.Seek;
    g_Callbacks.close    = &g_VirtualDirectory.Close;

    rc = UpnpSetVirtualDirCallbacks(&g_Callbacks);
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpSetVirtualDirCallbacks(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    rc = UpnpEnableWebserver(1);
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpEnableWebserver(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    rc = UpnpAddVirtualDir( "/" );
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpAddVirtualDir(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    return(0);
}
// ------------------------------------------------------------------------------------------------
// Parse Arguments
// ------------------------------------------------------------------------------------------------
bool ParseArgs(char** pInterface) NN_NOEXCEPT
{
    size_t        ifaceSize = 0;
    int           options = 0;
    int           argc = nn::os::GetHostArgc();
    char**        argv = nn::os::GetHostArgv();
    const char*   pChoosenInterface = nullptr;
    bool          ret = true;

    // Initialize passed variables
    *pInterface = nullptr;

    do
    {
        // Look for a passed command line option
        for(options = 0; options < argc; options++)
        {
            if(strcmp(argv[options], "-interface" ) == 0)  // Optional: Network Interface
            {
                options++;
                if(options >= argc)
                {
                    NN_LOG( "Arg: -interface requires an additional option, which is an IPv4 address assigned to this host\n");
                    ret = false;
                    break;
                }

                // Save choosen network interface
                pChoosenInterface = argv[options];
                break;
            }
        }

        // If parsing options failed..
        if (ret == false)
        {
            break;
        }

        // If NO network interface specified
        if (pChoosenInterface == nullptr)
        {
            break;
        }

        // Figure out size of the network Interface
        ifaceSize = strlen(pChoosenInterface) + 1;

        // Allocate Memory to hold network Interface
        *pInterface = new char[ifaceSize];
        if (*pInterface == nullptr)
        {
            NN_LOG("Failed calling `new` for network interface argument [%s], size [%d]\n", pChoosenInterface, ifaceSize);
            ret = false;
            break;
        }

        // Copy the network Interface
        memset(*pInterface, 0, ifaceSize);
        memcpy(*pInterface, pChoosenInterface, strlen(pChoosenInterface));

    } while (NN_STATIC_CONDITION(false));

    if (ret == false)
    {
        if (*pInterface != nullptr)
        {
            delete [] *pInterface;
            *pInterface = nullptr;
        }
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Main Loop
// ------------------------------------------------------------------------------------------------
void mainLoop() NN_NOEXCEPT
{
    nn::Result          result;
    nn::os::ThreadType  cmdloopThread;
    int                 rc = UPNP_E_SUCCESS;
    int                 len = 0;
    char*               pNetworkIface = nullptr;
    char*               pTmpNetworkIface = nullptr;
    char*               pUpnpIpAddress = nullptr;
    uint16_t            upnpPort = 0;

    do
    {
        // Initialize the Mutex
        nn::os::InitializeMutex(&g_UpnpAdvertiseMutex, false, 0);

       // Determine if caller specified a network interface
       if (ParseArgs(&pNetworkIface) == false)
        {
            NN_LOG("\nError: Invalid arguments.\n");
            NN_LOG("\nUsage:\n");
            NN_LOG("  -interface <ipv4 address> - Network interface to initialize libupnp on. (optional)\n");
            break;
        }

        NN_LOG ( "Initializing Network Interface ... \n");

        // Setup network interface
        rc = SetupNetwork();
        if(rc != 0)
        {
            NN_LOG("Failed to enable network interface\n");
            break;
        }

        // If caller (DID NOT) specify a network interface..
        if (pNetworkIface == nullptr)
        {
            // Turn the Nifm negotiated IPv4 Address into ASCII
            pTmpNetworkIface = nn::socket::InetNtoa(g_NifmIpv4Addr);
            if (pTmpNetworkIface == nullptr)
            {
                NN_LOG( "Failed to translate IPv4 address: [0x%08x] into an ASCII IP Address\n", nn::socket::InetNtohl(g_NifmIpv4Addr.s_addr));
                break;
            }

            // Figure out size of the network Interface
            len = strlen(pTmpNetworkIface) + 1;

            // Allocate Memory to hold network Interface
            pNetworkIface = new char[len];
            if (pNetworkIface == nullptr)
            {
                NN_LOG("Failed calling `new` for network interface argument [%s], size [%d]\n", pTmpNetworkIface, len);
                break;
            }

            // Copy the network Interface
            memset(pNetworkIface, 0, len);
            memcpy(pNetworkIface, pTmpNetworkIface, strlen(pTmpNetworkIface));

            NN_LOG( "Sample will use [Nifm] assigned network interface [%s]\n", pNetworkIface);
        }
        else
        {
            NN_LOG( "Sample will use [Command Line] assigned network interface [%s]\n", pNetworkIface);
        }

        // libupnp debugging flags:  UPNP_CRITICAL, UPNP_PACKET, UPNP_INFO, UPNP_ALL
#ifdef DETAILED_LIBUPNP_DEBUGGING
        UpnpSetLogLevel(UPNP_ALL);
#else
        UpnpSetLogLevel(UPNP_CRITICAL);
#endif

        NN_LOG ( "Initializing libupnp: (Requesting) IPv4 Network Interface Address:Port: [%s:%d]\n", pNetworkIface, UpnpServerPort);

        rc = UpnpInit(pNetworkIface, UpnpServerPort);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG ("** ERROR UpnpInit(): %d/%s", rc, UpnpGetErrorMessage(rc));
            break;
        }
        g_IsUpnpInitialized = true;

        pUpnpIpAddress = UpnpGetServerIpAddress();
        if (pUpnpIpAddress == nullptr)
        {
            NN_LOG( "UpnpGetServerIpAddress() returned a (NULL) IPv4 Address - Can't continue\n");
            break;
        }

        upnpPort = UpnpGetServerPort();
        if (upnpPort < 1)
        {
            NN_LOG( "UpnpGetServerPort() returned a (NULL) Port - Can't continue\n");
            break;
        }
        NN_LOG ("Successfully Initialized libupnp: (Actual) IPv4 Network Interface Address:Port: [%s:%d]\n", pUpnpIpAddress, upnpPort);

        rc = MakeVirtualFileSystem();
        if(rc != 0)
        {
            NN_LOG ("** ERROR: Failed calling MakeFileSystem()\n" );
            break;
        }

        // Act like a Control Point
        rc = UpnpRegisterClient(UpnpCallbackEventHandler, &g_UpnpCtrlptHandle, &g_UpnpCtrlptHandle);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** ERROR UpnpRegisterClient(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
            break;
        }
        g_IsUpnpClientRegistered = true;

        NN_LOG ( "Control Point Initialized\n" );

        // Search specifically for devices that support: urn:schemas-upnp-org:device:InternetGatewayDevice:1
        rc = UpnpSearchAsync(g_UpnpCtrlptHandle, 5, g_UpnpIGWDevices.IGWDeviceType, nullptr);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** ERROR UpnpSearchAsync(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
            break;
        }
        // Create the Device Command Loop Thread
        result = nn::os::CreateThread(&cmdloopThread, CommandLoop, nullptr, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        if (result.IsFailure())
        {
            NN_LOG ("** ERROR Failed calling 'nn::os::CreateThread()'\n");
            break;
        }

        // Start the thread
        nn::os::StartThread(&cmdloopThread);

        // Wait for the Thread
        nn::os::WaitThread(&cmdloopThread);

        // Destroy the Thread
        nn::os::DestroyThread(&cmdloopThread);

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG ("Exiting!\n");

    if (g_IsUpnpClientRegistered == true)
    {
        rc = UpnpUnRegisterClient(g_UpnpCtrlptHandle);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** ERROR UpnpUnRegisterClient(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
        }
        g_IsUpnpClientRegistered = false;
    }

    if (g_IsUpnpInitialized == true)
    {
        (void) UpnpFinish();
        g_IsUpnpInitialized = false;
    }

    if (pNetworkIface != nullptr)
    {
        delete [] pNetworkIface;
        pNetworkIface = nullptr;
    }

    FinalizeNetwork();

    // Destroy the Mutex
    nn::os::FinalizeMutex(&g_UpnpAdvertiseMutex);

} // NOLINT(impl/function_size)

}}  // Namespace nns / libupnp

// ------------------------------------------------------------------------------------------------
// Startup function
// ------------------------------------------------------------------------------------------------
extern "C" void nninitStartup() NN_NOEXCEPT
{
    // Setup the size of total memory heap
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    NN_ASSERT( result.IsSuccess() );

    // Allcate memory block from the heap for malloc
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock(&address, MemoryHeapSize);
    NN_ASSERT( result.IsSuccess() );

    // Set memory chunk for malloc
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// nnMain
// ------------------------------------------------------------------------------------------------
extern "C" void nnMain() NN_NOEXCEPT
{
    nns::libupnp::mainLoop();
}
