﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LibcurlBasic.cpp,PageSampleLibcurlBasic}
 *
 * @brief
 * libcurl による HTTP(S) 通信
 */

/**
 * @page PageSampleLibcurlBasic libcurl による HTTP(S) 通信
 * @tableofcontents
 *
 * @brief
 * libcurl による HTTP(S) 通信のサンプルプログラムの解説です。
 *
 * @section PageSampleLibcurlBasic_SectionBrief 概要
 * まず、 NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * その結果、ネットワーク接続が利用可能であれば libcurl を使用して HTTP(S) リクエストを送信します。
 *
 * @section PageSampleLibcurlBasic_SectionFileStoructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/Libcurl/Basic
 * Samples/Sources/Applications/Libcurl/Basic @endlink 以下にあります。
 *
 * @section PageSampleLibcurlBasic_SectionNecessaryEnvironment 必要な環境
 * 事前に設定マネージャを使用してネットワーク接続設定をインポートする必要があります。
 *  詳細は @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} をご覧ください。
 *
 * また、既定では http://www.example.com に対して HTTP リクエストを送信するため、
 * インターネットに接続できる環境が必要です。インターネットに接続できる環境を準備できない場合、
 * サンプルプログラムの URL 部分をアクセス可能な URL に置き換えてください。
 *
 * @section PageSampleLibcurlBasic_SectionHowToOperate 操作方法
 * 特にありません。
 *
 * @section PageSampleLibcurlBasic_SectionPrecaution 注意事項
 * ネットワーク接続に失敗した場合のハンドリング方法は未定です
 *
 * @section PageSampleLibcurlBasic_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleLibcurlBasic_SectionDetail 解説
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * - socket ライブラリを初期化します。
 * - URLがhttpsを含む場合SSLの設定をCURLOPT_SSL_CTX_FUNCTIONを利用して行います。
 * - libcurl の easy interface を使用して HTTP(S) リクエストを送信します。
 * - HTTP(S) レスポンスの内容をログ出力します。
 *
 * このサンプルプログラムの実行結果を以下に示します。
 * ただし、 Release ビルドではログメッセージが出力されません。
 *
 * @verbinclude LibcurlBasic_Output.txt
 */

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/nifm.h>
#include <nn/socket.h>
#include <curl/curl.h>

#include "../Common/HttpsHelper.h"

namespace
{
    class CurlProgress {
    public:
        CURL *curl;
        curl_off_t lastDownload;
        curl_off_t lastUpload;
        uint32_t lastDownloadPercent;
        uint32_t lastUploadPercent;
        nn::os::Tick lastTick;
    };

    // Socket configuration and memory pool
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    const uint32_t TransactionTimeoutSec = 10; // Cancel transaction if no data has been sent/recved within 'TransactionTimeoutSec' seconds

    // CurlWriteFunction
    size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
    {
        int32_t count = static_cast<int32_t>(blobsize * blobcount);

        if (count > 0)
        {
            NN_LOG("%.*s", count, pData);
        }

        return count;
    }

    size_t CurlHeaderFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
    {
        int32_t count = static_cast<int32_t>(blobsize * blobcount);

        if (count > 0)
        {
            NN_LOG("%.*s", count, pData);
        }

        return count;
    }

    int CurlTransferInfo(void *clientp, curl_off_t downloadTotal, curl_off_t downloadCurrent, curl_off_t uploadTotal, curl_off_t uploadCurrent) NN_NOEXCEPT
    {
        CurlProgress *pProgress = static_cast<CurlProgress *>(clientp);
        nn::os::Tick currentTick       = nn::os::GetSystemTick();

        if( downloadTotal > 0 )
        {
            uint32_t nowPercent = uint32_t(double(downloadCurrent) / double(downloadTotal) * 100.0);
            if( nowPercent >= pProgress->lastDownloadPercent + 10 ) // Only print every 10 percent or more
            {
                char *url = NULL;
                // In the case of a redirect, this URL will be different than the original request
                CURLcode code = curl_easy_getinfo(pProgress->curl, CURLINFO_EFFECTIVE_URL, &url);
                if( code == CURLE_OK && url && strlen(url) > 0 )
                {
                    NN_LOG("\n * URL: %s", url);
                }
                pProgress->lastDownloadPercent = nowPercent;
                NN_LOG("\n * Download Progress: %d%%\n", nowPercent);
            }
        }

        if( uploadTotal > 0 )
        {
            uint32_t nowPercent = uint32_t(double(uploadCurrent) / double(uploadTotal) * 100.0);
            if( nowPercent >= pProgress->lastUploadPercent + 10 ) // Only print every 10 percent or more
            {
                char *url = NULL;
                // In the case of a redirect, this URL will be different than the original request
                CURLcode code = curl_easy_getinfo(pProgress->curl, CURLINFO_EFFECTIVE_URL, &url);
                if( code == CURLE_OK && url && strlen(url) > 0 )
                {
                    NN_LOG("\n * URL: %s", url);
                }
                pProgress->lastUploadPercent = nowPercent;
                NN_LOG("\n * Upload Progress: %d%%\n", nowPercent);
            }
        }

        // If we have sent or received anything, reset the timer.
        if( (downloadCurrent > pProgress->lastDownload) || (uploadCurrent > pProgress->lastUpload) )
        {
            pProgress->lastDownload = downloadCurrent;
            pProgress->lastUpload = uploadCurrent;
            pProgress->lastTick = currentTick;
        }
        // If we have not sent or received data in the last 'TransactionTimeoutSec' seconds, cancel the transaction.
        else if( nn::os::ConvertToTimeSpan(currentTick - pProgress->lastTick).GetSeconds() >= TransactionTimeoutSec )
        {
            NN_LOG("\n * [ERROR] No data sent or received for %d seconds. Canceling transaction...\n\n", TransactionTimeoutSec);
            return -1; // Return non-zero to cancel transaction.
        }

        return 0;
    }
} // Un-named namespace

//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    CURL *curl = NULL;
    CURLcode res = CURLE_OK;
    nn::Result ret = nn::Result();

    if( nn::os::GetHostArgc() < 2 )
    {
        NN_LOG("\n[ERROR] Must pass parameter <url>.\n");
        return;
    }

    /* since the following will utilize the system network interface, we must initialize
       network interface manager (NIFM) */
    nn::nifm::Initialize();

    /* this application is now requesting to use network interface */
    nn::nifm::SubmitNetworkRequest();

    /* wait for network interface availability, while providing status */
    while( nn::nifm::IsNetworkRequestOnHold() )
    {
        NN_LOG("Waiting for network interface availability...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    if( !nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("\n[ERROR] network is not available.\n");
        goto CLEANUP_NIFM;
    }

    /* Initialize socket library, while supplying configuration and memory pool */
    ret = nn::socket::Initialize(g_SocketConfigWithMemory);

    if( ret.IsFailure() )
    {
        NN_LOG("\n[ERROR] nn::socket::Initialize() failed. Err Desc: %d\n\n", ret.GetDescription());
        goto CLEANUP_NIFM;
    }

    /* initialize using system malloc for libcurl for testing */
    res = curl_global_init(CURL_GLOBAL_DEFAULT);
    if( res != CURLE_OK )
    {
        NN_LOG("\n[ERROR] curl_global_init failed. Err: %d\n\n", res);
        goto CLEANUP_SOCKET_AND_NIFM;
    }

    /* initialize with application-supplied allocator (intended application mode of use) */
    /*
    curl_global_init_mem(CURL_GLOBAL_DEFAULT, mymalloc, myfree, myrealloc, mystrdup, mycalloc);
    */

    curl = curl_easy_init();

    if( curl )
    {
        CurlProgress progress = {curl, 0, 0, 0, 0, nn::os::GetSystemTick()};
        char *url = nn::os::GetHostArgv()[1];

        /* currently using host resolver for DNS resolution */
        curl_easy_setopt(curl, CURLOPT_URL, url);

        /* optionally set verbose for debugging */
        curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
        curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 15L);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,  CurlWriteFunction);
        curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, CurlHeaderFunction);
        curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        curl_easy_setopt(curl, CURLOPT_XFERINFOFUNCTION, CurlTransferInfo);
        curl_easy_setopt(curl, CURLOPT_XFERINFODATA, &progress);

#if 0
        /*
          Configure proxy if needed.
          When configuring proxy via curl options, CURLOPT_PROXYAUTOCONFIG must be set to 0.
          Otherwise, these settings will be overwritten with the proxy configuration in
          the system settings (nifm).
        */
        curl_easy_setopt(curl, CURLOPT_PROXYAUTOCONFIG, 0);
        curl_easy_setopt(curl, CURLOPT_PROXY, "<Proxy_hostname_or_ip>");
        curl_easy_setopt(curl, CURLOPT_PROXYPORT, <Proxy_port>);
        curl_easy_setopt(curl, CURLOPT_PROXYUSERNAME, "<Proxy_username>");
        curl_easy_setopt(curl, CURLOPT_PROXYPASSWORD, "<Proxy_password>");
        curl_easy_setopt(curl, CURLOPT_PROXYAUTH, <Proxy_auth_method>);
#endif

        /* Setup for SSL if URL contains https */
        HttpsHelperForCallback httpsHelper;
        if( HttpsHelperBase::IsUrlHttps(url) == true )
        {
            HttpsHelperForCallback::VerifyOption verifyOption;

            if (httpsHelper.Initialize(curl) < 0)
            {
                NN_LOG("\n[ERROR] Failed to configure CURL handle for HTTPS\n\n");
                curl_easy_cleanup(curl);
                goto CLEANUP_SOCKET_AND_NIFM;
            }

            /* Configure verify options */
            verifyOption.isVerifyPeer = true;
            verifyOption.isVerifyName = true;
            verifyOption.isVerifyTime = false;
            if (httpsHelper.ConfigureVerifyOption(curl, &verifyOption) < 0)
            {
                NN_LOG("\n[ERROR] Failed to configure verify option for HTTPS\n\n");
                curl_easy_cleanup(curl);
                httpsHelper.Finalize();
                goto CLEANUP_CURL_AND_SOCKET_AND_NIFM;
            }
        }

        /* Perform the request, res will get the return code */
        NN_LOG("Fetching URL: %s\n", url);
        res = curl_easy_perform(curl);

        /* Check for errors */
        if( res != CURLE_OK )
        {
            NN_LOG("\n[ERROR] CURL_EASY_PERFORM() FAILED! curl error: %d\n", res);
            if( HttpsHelperBase::IsUrlHttps(url) == true )
            {
                httpsHelper.PrintErrorMessage(curl, res);
            }
        }
        else
        {
            NN_LOG("\nCURL_EASY_PERFORM() SUCCEEDED!\n");
        }

        /* always cleanup */
        curl_easy_cleanup(curl);

        httpsHelper.Finalize();
    }
    else
    {
        NN_LOG("[ERROR] CURL_EASY_INIT() FAILED\n");
    }

CLEANUP_CURL_AND_SOCKET_AND_NIFM:
    /* cleanup libcurl library */
    curl_global_cleanup();

CLEANUP_SOCKET_AND_NIFM:
    /* cleanup socket library */
    nn::socket::Finalize();

CLEANUP_NIFM:
    /* this application no longer requires use of network interface */
    nn::nifm::CancelNetworkRequest();

    return;
} // NOLINT(impl/function_size)

