﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include "NetworkTask.h"
#include "LdnTask.h"
#include "LdnUpdater.h"

namespace nns { namespace ldn { namespace
{
    NN_OS_ALIGNAS_GUARDED_STACK char g_RunnerStack[16 * 1024];
    NN_OS_ALIGNAS_GUARDED_STACK char g_SenderStack[4 * 1024];
    NN_OS_ALIGNAS_GUARDED_STACK char g_ReceiverStack[4 * 1024];

    template <int N>
    nn::Result GetResult(const SequentialTaskRunner<nn::Result, N>& runner) NN_NOEXCEPT
    {
        for (int i = 0; i < runner.GetTaskCount(); ++i)
        {
            const auto &task = *runner.GetTask(i);
            NN_ASSERT_EQUAL(task.GetState(), TaskState_Finished);
            if (task.GetResult().IsFailure())
            {
                return task.GetResult();
            }
        }
        return nn::ResultSuccess();
    }

    template <typename T, int N>
    void Clear(SequentialTaskRunner<T, N>* pRunner) NN_NOEXCEPT
    {
        for (int i = 0; i < pRunner->GetTaskCount(); ++i)
        {
            auto pTask = pRunner->GetTask(i);
            delete pTask;
        }
        pRunner->Clear();
    }

    template <int N>
    void Clear(SequentialVoidTaskRunner<N>* pRunner) NN_NOEXCEPT
    {
        for (int i = 0; i < pRunner->GetTaskCount(); ++i)
        {
            auto pTask = pRunner->GetTask(i);
            delete pTask;
        }
        pRunner->Clear();
    }

    const nn::ldn::ScanFilter CreateScanFilter(const Config& config) NN_NOEXCEPT
    {
        nn::ldn::ScanFilter filter = { };
        filter.flag = nn::ldn::ScanFilterFlag_IntentId;
        filter.networkId.intentId = nn::ldn::MakeIntentId(
            config.localCommunicationId, config.sceneId);
        return filter;
    }

    const nn::ldn::NetworkConfig CreateNetworkConfig(const Config& config) NN_NOEXCEPT
    {
        nn::ldn::NetworkConfig network = { };
        network.intentId = nn::ldn::MakeIntentId(
            config.localCommunicationId, config.sceneId);
        network.channel = config.channel;
        network.nodeCountMax = config.nodeCountMax;
        network.localCommunicationVersion = config.localCommunicationVersion;
        return network;
    }

    const nn::ldn::SecurityConfig CreateSecurityConfig(const Config& config) NN_NOEXCEPT
    {
        const uint8_t passphrase[16] = {
            0xce, 0x83, 0x3a, 0x55, 0xd6, 0x92, 0xca, 0x7d,
            0x57, 0x88, 0x7e, 0x2e, 0x2f, 0xce, 0x7a, 0xb7
        };
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(config.isEncrypted ?
            nn::ldn::SecurityMode_Product : nn::ldn::SecurityMode_Debug);
        std::memcpy(security.passphrase, passphrase, sizeof(passphrase));
        security.passphraseSize = static_cast<uint16_t>(sizeof(passphrase));
        return security;
    }

    const nn::ldn::UserConfig CreateUserConfig(const Config& config) NN_NOEXCEPT
    {
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, config.userName, nn::ldn::UserNameBytesMax);
        return user;
    }

    PadState IntegratePadState(const PadState (&padStates)[PadCountMax]) NN_NOEXCEPT
    {
        PadState out = { };
        for (int i = 0; i < PadCountMax; ++i)
        {
            const auto& in = padStates[i];
            out.button.down |= in.button.down;
            out.button.trigger |= in.button.trigger;
        }
        return out;
    }

}}} // namespace nns::ldn::<unnamed>

namespace nns { namespace ldn
{
    LdnUpdater::LdnUpdater() NN_NOEXCEPT
        : m_NextScene(Scene_None),
          m_NextSubScene(LdnSubScene_None),
          m_Runner(g_RunnerStack, sizeof(g_RunnerStack), nn::os::DefaultThreadPriority),
          m_Sender(g_SenderStack, sizeof(g_SenderStack), nn::os::DefaultThreadPriority),
          m_Receiver(g_ReceiverStack, sizeof(g_ReceiverStack), nn::os::DefaultThreadPriority),
          m_Socket(nn::socket::InvalidSocket)
    {
    }

    LdnUpdater::~LdnUpdater() NN_NOEXCEPT
    {
    }

    void LdnUpdater::Setup(const ApplicationStatus& app) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        std::memset(&m_Data, 0, sizeof(LdnData));
        SetNextSubScene(LdnSubScene_Initializing);
    }

    void LdnUpdater::Cleanup(const ApplicationStatus& app) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
        m_NextScene = Scene_None;
        m_NextSubScene = LdnSubScene_None;
    }

    const void* LdnUpdater::GetSceneData() const NN_NOEXCEPT
    {
        return &m_Data;
    }

    Scene LdnUpdater::GetNextScene() const NN_NOEXCEPT
    {
        return m_NextScene;
    }

    void LdnUpdater::Update(
        const ApplicationStatus& app,
        const PadState (&padStates)[PadCountMax]) NN_NOEXCEPT
    {
        // 更新フラグは落としておきます。
        m_Data.isUpdated = false;

        // サブシーンの遷移です。
        if (m_Data.subScene != m_NextSubScene)
        {
            m_Data.subScene = static_cast<int16_t>(m_NextSubScene);
            m_Data.x = 0;
            m_Data.y = 0;
            m_Data.subSceneFrame = 0;
            m_Data.isUpdated = true;
        }
        else
        {
            ++m_Data.subSceneFrame;
        }

        // サブシーンに応じた処理です。
        const PadState padState = IntegratePadState(padStates);
        switch (m_Data.subScene)
        {
        case LdnSubScene_Initializing:
            UpdateInitializing(app, padState);
            break;
        case LdnSubScene_Initialized:
            UpdateInitialized(app, padState);
            break;
        case LdnSubScene_Station:
            UpdateStation(app, padState);
            break;
        case LdnSubScene_StationScanning:
            UpdateStationScanning(app, padState);
            break;
        case LdnSubScene_StationConnecting:
            UpdateStationConnecting(app, padState);
            break;
        case LdnSubScene_StationConnected:
            UpdateStationConnected(app, padState);
            break;
        case LdnSubScene_StationDisconnected:
            UpdateStationDisconnected(app, padState);
            break;
        case LdnSubScene_StationClosing:
            UpdateStationClosing(app, padState);
            break;
        case LdnSubScene_AccessPointCreating:
            UpdateAccessPointCreating(app, padState);
            break;
        case LdnSubScene_AccessPointCreated:
            UpdateAccessPointCreated(app, padState);
            break;
        case LdnSubScene_AccessPointScanning:
            UpdateAccessPointScanning(app, padState);
            break;
        case LdnSubScene_AccessPointClosing:
            UpdateAccessPointClosing(app, padState);
            break;
        case LdnSubScene_AccessPointRejecting:
            UpdateAccessPointRejecting(app, padState);
            break;
        case LdnSubScene_Finalizing:
            UpdateFinalizing(app, padState);
            break;
        case LdnSubScene_Error:
            UpdateError(app, padState);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void LdnUpdater::UpdateInitializing(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // LDN ライブラリの初期化を開始します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            m_Runner.AddTask(new LdnInitializeTask());
            m_Runner.Run();
        }

        // LDN ライブラリの初期化処理が完了したら結果に応じて状態遷移します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::Result result = GetResult(m_Runner);
            if (result.IsSuccess())
            {
                nn::ldn::AttachStateChangeEvent(&m_StateChangeEvent);
                SetNextSubScene(LdnSubScene_Initialized);
            }
            else
            {
                SetNextSubScene(LdnSubScene_Error);
            }
            m_Data.lastError = result;
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateInitialized(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);

        // メニュー項目の総数です。
        const int menuCount = 2;

        // スリープなどの要因により、通信を継続できなくなった場合には最初からやり直します。
        if (nn::ldn::GetState() != nn::ldn::State_Initialized)
        {
            SetNextSubScene(LdnSubScene_Finalizing);
        }

        // 先頭のコントローラからの入力のみを処理します。
        else if (IsTriggeredAny(padState, Button_A))
        {
            if (m_Data.y == 0)
            {
                // ステーションとして周囲のネットワークのスキャンを開始します。
                m_Data.scanResultCount = 0;
                SetNextSubScene(LdnSubScene_StationScanning);
            }
            else if (m_Data.y == 1)
            {
                // アクセスポイントとしてネットワークを構築します。
                SetNextSubScene(LdnSubScene_AccessPointCreating);
            }
        }
        else if (IsTriggeredAny(padState, Button_Up))
        {
            m_Data.y = (m_Data.y + menuCount - 1) % menuCount;
            m_Data.isUpdated = true;
        }
        else if (IsTriggeredAny(padState, Button_Down))
        {
            m_Data.y = (m_Data.y + 1) % menuCount;
            m_Data.isUpdated = true;
        }
        else if (IsTriggeredAny(padState, Button_Plus))
        {
            m_NextScene = Scene_Exit;
        }
    }

    void LdnUpdater::UpdateStation(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT

    {
        NN_UNUSED(app);

        // スリープなどの要因により、通信を継続できなくなった場合には最初からやり直します。
        if (nn::ldn::GetState() != nn::ldn::State_Station)
        {
            SetNextSubScene(LdnSubScene_Finalizing);
        }

        // 先頭のコントローラからの入力のみを処理します。
        else if (IsTriggeredAny(padState, Button_A) && 0 < m_Data.scanResultCount)
        {
            m_Data.network = m_Data.scanResult[m_Data.y];
            SetNextSubScene(LdnSubScene_StationConnecting);
        }
        else if (IsTriggeredAny(padState, Button_B))
        {
            SetNextSubScene(LdnSubScene_StationClosing);
        }
        else if (IsTriggeredAny(padState, Button_X))
        {
            SetNextSubScene(LdnSubScene_StationScanning);
        }
        else if (2 <= m_Data.scanResultCount)
        {
            if (IsTriggeredAny(padState, Button_Up))
            {
                m_Data.y = (m_Data.y + m_Data.scanResultCount - 1) % m_Data.scanResultCount;
                m_Data.isUpdated = true;
            }
            else if (IsTriggeredAny(padState, Button_Down))
            {
                m_Data.y = (m_Data.y + 1) % m_Data.scanResultCount;
                m_Data.isUpdated = true;
            }
        }
    }

    void LdnUpdater::UpdateStationScanning(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT

    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // スキャンを開始します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            m_Runner.AddTask(new LdnOpenStationTask());
            auto pScanTask = new LdnScanTask();
            pScanTask->SetChannel(app.config.channel);
            pScanTask->SetScanFilter(CreateScanFilter(app.config));
            m_Runner.AddTask(pScanTask);
            m_Runner.Run();
        }

        // スキャン結果を取得します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::Result result = GetResult(m_Runner);
            if (result.IsSuccess())
            {
                auto pScanTask = static_cast<LdnScanTask*>(m_Runner.GetTask(1));
                for (int i = 0; i < pScanTask->GetNetworkCount(); ++i)
                {
                    m_Data.scanResult[i] = pScanTask->GetNetwork(i);
                }
                m_Data.scanResultCount = static_cast<int8_t>(pScanTask->GetNetworkCount());
                SetNextSubScene(LdnSubScene_Station);
            }
            else
            {
                SetNextSubScene(LdnSubScene_Error);
            }
            m_Data.lastError = result;
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateStationConnecting(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // ネットワークへの接続を試行します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            m_Runner.AddTask(new LdnOpenStationTask());
            auto pConnectTask = new LdnConnectTask();
            pConnectTask->SetNetworkInfo(m_Data.network);
            pConnectTask->SetSecurityConfig(CreateSecurityConfig(app.config));
            pConnectTask->SetUserConfig(CreateUserConfig(app.config));
            pConnectTask->SetLocalCommunicationVersion(app.config.localCommunicationVersion);
            pConnectTask->SetRetryCount(3);
            m_Runner.AddTask(pConnectTask);
            m_Runner.Run();
        }

        // ネットワーク接続の結果を取得します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::Result result = GetResult(m_Runner);
            if (result.IsSuccess())
            {
                auto pConnectTask = static_cast<LdnConnectTask*>(m_Runner.GetTask(1));
                m_Data.address = pConnectTask->GetIpv4Address();
                m_Data.mask = pConnectTask->GetSubnetMask();
                StartCommunication(app.config);
                SetNextSubScene(LdnSubScene_StationConnected);
            }
            else if (nn::ldn::ResultConnectionFailed::Includes(result))
            {
                SetNextSubScene(LdnSubScene_Station);
            }
            else
            {
                SetNextSubScene(LdnSubScene_Error);
            }
            m_Data.lastError = result;
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateStationConnected(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT

    {
        NN_UNUSED(app);

        // 通信状態に変化があった場合には新しいネットワーク情報を取得します。
        nn::Result result;
        if (nn::os::TryWaitSystemEvent(&m_StateChangeEvent))
        {
            result = nn::ldn::GetNetworkInfo(
                &m_Data.network, m_Data.updates, nn::ldn::NodeCountMax);
            if (nn::ldn::ResultInvalidState::Includes(result))
            {
                SetNextSubScene(LdnSubScene_StationDisconnected);
                m_Data.lastError = result;
                return;
            }
            else if (result.IsFailure())
            {
                SetNextSubScene(LdnSubScene_Error);
                m_Data.lastError = result;
                return;
            }
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (m_Data.updates[i].stateChange != nn::ldn::NodeStateChange_None)
                {
                    std::memset(&m_Data.counters[i], 0, sizeof(Counter));
                }
            }
            m_Data.isUpdated = true;
        }

        // 先頭のコントローラからの入力のみを処理します。
        if (IsTriggeredAny(padState, Button_B))
        {
            SetNextSubScene(LdnSubScene_StationClosing);
        }
        else if (IsTriggeredAny(padState, Button_X))
        {
            // 将来的にスキャンを実装します。
        }
    }

    void LdnUpdater::UpdateStationDisconnected(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT

    {
        NN_UNUSED(app);

        // 切断理由を取得します。
        if (m_Data.subSceneFrame == 0)
        {
            m_Data.disconnectReason = static_cast<int8_t>(nn::ldn::GetDisconnectReason());
        }

        // 先頭のコントローラからの入力のみを処理します。
        if (IsTriggeredAny(padState, Button_B))
        {
            SetNextSubScene(LdnSubScene_StationClosing);
        }
    }

    void LdnUpdater::UpdateStationClosing(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT

    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // ステーションとしての動作を終了します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            m_Runner.AddTask(new LdnCloseStationTask());
            m_Runner.Run();
        }

        // ステーションの終了処理が完了するまで待機します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::Result result = GetResult(m_Runner);
            if (result.IsSuccess())
            {
                SetNextSubScene(LdnSubScene_Initialized);
            }
            else
            {
                SetNextSubScene(LdnSubScene_Error);
            }
            StopCommunication();
            m_Data.lastError = result;
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateAccessPointCreating(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // ネットワークを構築します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            m_Runner.AddTask(new LdnOpenAccessPointTask());
            auto pCreateTask = new LdnCreateNetworkTask();
            pCreateTask->SetNetworkConfig(CreateNetworkConfig(app.config));
            pCreateTask->SetSecurityConfig(CreateSecurityConfig(app.config));
            pCreateTask->SetUserConfig(CreateUserConfig(app.config));
            m_Runner.AddTask(pCreateTask);
            m_Runner.Run();
        }

        // ネットワーク構築の完了まで待機します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::Result result = GetResult(m_Runner);
            if (result.IsSuccess())
            {
                auto pCreateTask = static_cast<LdnCreateNetworkTask*>(m_Runner.GetTask(1));
                m_Data.address = pCreateTask->GetIpv4Address();
                m_Data.mask = pCreateTask->GetSubnetMask();
                m_Data.scanResultCount = 0;
                StartCommunication(app.config);
                SetNextSubScene(LdnSubScene_AccessPointCreated);
            }
            else
            {
                SetNextSubScene(LdnSubScene_Error);
            }
            m_Data.lastError = result;
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateAccessPointCreated(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT

    {
        NN_UNUSED(app);
        NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);

        // 通信状態に変化があった場合には新しいネットワーク情報を取得します。
        nn::Result result;
        if (nn::os::TryWaitSystemEvent(&m_StateChangeEvent))
        {
            result = nn::ldn::GetNetworkInfo(
                &m_Data.network, m_Data.updates, nn::ldn::NodeCountMax);
            if (result.IsFailure())
            {
                SetNextSubScene(LdnSubScene_Error);
                m_Data.lastError = result;
                return;
            }
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (m_Data.updates[i].stateChange != nn::ldn::NodeStateChange_None)
                {
                    std::memset(&m_Data.counters[i], 0, sizeof(Counter));
                }
            }
            m_Data.isUpdated = true;
        }

        // 先頭のコントローラからの入力のみを処理します。
        if (IsTriggeredAny(padState, Button_A))
        {
            // 指定されたステーションを切断します。
            const auto& node = m_Data.network.ldn.nodes[m_Data.y];
            if (0 < node.nodeId && node.isConnected)
            {
                m_Data.rejectNodeId = node.nodeId;
                SetNextSubScene(LdnSubScene_AccessPointRejecting);
            }
        }
        else if (IsTriggeredAny(padState, Button_B))
        {
            // ネットワークを破棄します。
            SetNextSubScene(LdnSubScene_AccessPointClosing);
        }
        else if (IsTriggeredAny(padState, Button_X))
        {
            // 周囲のネットワークをスキャンします。
            SetNextSubScene(LdnSubScene_AccessPointScanning);
        }
        else if (IsTriggeredAny(padState, Button_Up))
        {
            m_Data.y = (m_Data.y + nn::ldn::NodeCountMax - 1) % nn::ldn::NodeCountMax;
            m_Data.isUpdated = true;
        }
        else if (IsTriggeredAny(padState, Button_Down))
        {
            m_Data.y = (m_Data.y + 1) % nn::ldn::NodeCountMax;
            m_Data.isUpdated = true;
        }
    }

    void LdnUpdater::UpdateAccessPointScanning(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // スキャンを開始します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            auto pScanTask = new LdnScanTask();
            pScanTask->SetChannel(app.config.channel);
            pScanTask->SetScanFilter(CreateScanFilter(app.config));
            m_Runner.AddTask(pScanTask);
            m_Runner.Run();
        }

        // スキャンの完了まで待機します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::Result result = GetResult(m_Runner);
            if (result.IsSuccess())
            {
                auto pScanTask = static_cast<LdnScanTask*>(m_Runner.GetTask(0));
                for (int i = 0; i < pScanTask->GetNetworkCount(); ++i)
                {
                    m_Data.scanResult[i] = pScanTask->GetNetwork(i);
                }
                m_Data.scanResultCount = static_cast<int8_t>(pScanTask->GetNetworkCount());
                SetNextSubScene(LdnSubScene_AccessPointCreated);
            }
            else
            {
                SetNextSubScene(LdnSubScene_Error);
            }
            m_Data.lastError = result;
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateAccessPointRejecting(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // 指定されたステーションの切断を開始します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            m_Runner.AddTask(new LdnRejectTask(m_Data.network.ldn.nodes[m_Data.rejectNodeId]));
            m_Runner.Run();
        }

        // アクセスポイントの終了処理が完了するまで待機します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::Result result = GetResult(m_Runner);
            if (result.IsSuccess() || nn::ldn::ResultNodeNotFound::Includes(result))
            {
                SetNextSubScene(LdnSubScene_AccessPointCreated);
            }
            else
            {
                SetNextSubScene(LdnSubScene_Error);
            }
            m_Data.lastError = result;
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateAccessPointClosing(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // ネットワークを破棄し、アクセスポイントとしての動作を終了します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            m_Runner.AddTask(new LdnCloseAccessPointTask());
            m_Runner.Run();
        }

        // アクセスポイントの終了処理が完了するまで待機します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::Result result = GetResult(m_Runner);
            if (result.IsSuccess())
            {
                SetNextSubScene(LdnSubScene_Initialized);
            }
            else
            {
                SetNextSubScene(LdnSubScene_Error);
            }
            StopCommunication();
            m_Data.lastError = result;
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateFinalizing(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(padState);

        // LDN ライブラリを終了します。
        if (m_Data.subSceneFrame == 0)
        {
            NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);
            m_Runner.AddTask(new LdnFinalizeTask());
            m_Runner.Run();
        }

        // LDN ライブラリの終了処理の完了まで待機します。
        else if (m_Runner.GetState() == TaskRunnerState_Paused)
        {
            nn::os::DestroySystemEvent(&m_StateChangeEvent);
            m_Data.lastError = nn::ResultSuccess();
            SetNextSubScene(LdnSubScene_Initializing);
            Clear(&m_Runner);
        }
    }

    void LdnUpdater::UpdateError(
        const ApplicationStatus& app, const PadState &padState) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_ASSERT_EQUAL(m_Runner.GetTaskCount(), 0);

        // データ通信を終了します。データ通信中でない場合には何もしません。
        if (m_Data.subSceneFrame == 0)
        {
            StopCommunication();
        }

        // 先頭のコントローラからの入力のみを処理します。
        if (IsTriggeredAny(padState, Button_A))
        {
            m_Data.lastError = nn::ResultSuccess();
            SetNextSubScene(LdnSubScene_Finalizing);
        }
        else if (IsTriggeredAny(padState, Button_B))
        {
            m_NextScene = Scene_Exit;
        }
    }

    void LdnUpdater::SetNextScene(Scene scene) NN_NOEXCEPT
    {
        m_NextScene = scene;
    }

    void LdnUpdater::SetNextSubScene(LdnSubScene scene) NN_NOEXCEPT
    {
        m_NextSubScene = scene;
    }

    void LdnUpdater::StartCommunication(const Config& config) NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(m_Socket, nn::socket::InvalidSocket);

        // データの送受信に使用する socket を生成します。
        m_Socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        NN_ABORT_UNLESS_NOT_EQUAL(m_Socket, nn::socket::InvalidSocket);
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(config.port);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;
        NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
            m_Socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)), 0);

        // ブロードキャスト通信を有効化します。
        int isEnabled = 1;
        nn::socket::SetSockOpt(
            m_Socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));

        // データ送信を開始します。
        auto pSendTask = new NetworkSendTask(&m_Data, m_Socket, config.port, config.packetRate);
        m_Sender.AddTask(pSendTask);
        m_Sender.Run();

        // データ受信を開始します。
        auto pReceiveTask = new NetworkReceiveTask(&m_Data, m_Socket, config.port);
        m_Receiver.AddTask(pReceiveTask);
        m_Receiver.Run();
    }

    void LdnUpdater::StopCommunication() NN_NOEXCEPT
    {
        // 通信中で無い場合には何もしません。
        if (m_Socket == nn::socket::InvalidSocket)
        {
            return;
        }

        // データの送信・受信を終了します。
        m_Sender.Cancel();
        m_Receiver.Cancel();

        // データの送受信に使用する socket をクローズします。
        nn::socket::Shutdown(m_Socket, nn::socket::ShutdownMethod::Shut_RdWr);
        nn::socket::Close(m_Socket);
        m_Socket = nn::socket::InvalidSocket;

        // データ送受信の終了まで待機します。
        // socket を shutdown しているので実際には即座に完了することを期待しています。
        while (m_Sender.GetState() == TaskRunnerState_Running)
        {
            m_Sender.GetStateChangeEvent().Wait();
        }
        while (m_Receiver.GetState() == TaskRunnerState_Running)
        {
            m_Receiver.GetStateChangeEvent().Wait();
        }

        // タスクを削除します。
        Clear(&m_Sender);
        Clear(&m_Receiver);
    }

}} // namespace nns::ldn
